<?php

namespace Backend\Core\Engine;

/*
 * This file is part of Fork CMS.
 *
 * For the full copyright and license information, please view the license
 * file that was distributed with this source code.
 */

use Common\Doctrine\Entity\Meta as MetaEntity;
use Common\Uri as CommonUri;
use Backend\Core\Engine\Model as BackendModel;
use Backend\Core\Language\Language as BackendLanguage;

/**
 * This class represents a META-object
 */
class Meta
{
    /**
     * The name of the field we should use to generate default-values
     *
     * @var string
     */
    protected $baseFieldName;

    /**
     * The callback method
     *
     * @var array
     */
    protected $callback = array();

    /**
     * Do we need meta custom
     *
     * @var bool
     */
    protected $custom;

    /**
     * The data, when a existing meta-record is loaded
     *
     * @var array
     */
    protected $data;

    /**
     * The form instance
     *
     * @var Form
     */
    protected $frm;

    /**
     * The id, when an existing meta-record is loaded
     *
     * @var int
     */
    protected $id;

    /**
     * The URL-instance
     *
     * @var Url
     */
    protected $URL;

    /**
     * @param Form $form An instance of Form, the elements will be parsed in here.
     * @param int $metaId The metaID to load.
     * @param string $baseFieldName The field where the URL should be based on.
     * @param bool $custom Add/show custom-meta.
     *
     * @throws Exception
     */
    public function __construct(Form $form, $metaId = null, $baseFieldName = 'title', $custom = false)
    {
        // check if URL is available from the reference
        if (!BackendModel::getContainer()->has('url')) {
            throw new Exception('URL should be available in the reference.');
        }

        // get BackendURL instance
        $this->URL = BackendModel::getContainer()->get('url');

        // should we use meta-custom
        $this->custom = (bool) $custom;

        // set form instance
        $this->frm = $form;

        // set base field name
        $this->baseFieldName = (string) $baseFieldName;

        // metaId was specified, so we should load the item
        if ($metaId !== null) {
            $this->loadMeta($metaId);
        }

        // set default callback
        $this->setURLCallback(
            'Backend\\Modules\\' . $this->URL->getModule() . '\\Engine\\Model',
            'getURL'
        );

        // load the form
        $this->loadForm();
    }

    /**
     * Generate an url, using the predefined callback.
     *
     * @param string $url The base-url to start from.
     *
     * @throws Exception When the function does not exist
     *
     * @return string
     *
     * @deprecated use the generateUrl method on the meta repository
     */
    public function generateURL($url)
    {
        return  Model::get('fork.repository.meta')->generateURL(
            $url,
            $this->callback['class'],
            $this->callback['method'],
            $this->callback['parameters']
        );
    }

    /**
     * Get the current value for the meta-description;
     *
     * @return mixed
     */
    public function getDescription()
    {
        // not set so return null
        if (!isset($this->data['description'])) {
            return;
        }

        // return value
        return $this->data['description'];
    }

    /**
     * Should the description overwrite the default
     *
     * @return null|bool
     */
    public function getDescriptionOverwrite()
    {
        // not set so return null
        if (!isset($this->data['description_overwrite'])) {
            return;
        }

        // return value
        return ($this->data['description_overwrite'] == 'Y');
    }

    /**
     * Get the current value for the metaId;
     *
     * @return null|int
     */
    public function getId()
    {
        // not set so return null
        if (!isset($this->data['id'])) {
            return;
        }

        // return value
        return (int) $this->data['id'];
    }

    /**
     * Get the current value for the meta-keywords;
     *
     * @return mixed
     */
    public function getKeywords()
    {
        // not set so return null
        if (!isset($this->data['keywords'])) {
            return;
        }

        // return value
        return $this->data['keywords'];
    }

    /**
     * Should the keywords overwrite the default
     *
     * @return null|bool
     */
    public function getKeywordsOverwrite()
    {
        // not set so return null
        if (!isset($this->data['keywords_overwrite'])) {
            return;
        }

        // return value
        return ($this->data['keywords_overwrite'] == 'Y');
    }

    /**
     * Get the current value for the page title;
     *
     * @return mixed
     */
    public function getTitle()
    {
        // not set so return null
        if (!isset($this->data['title'])) {
            return;
        }

        // return value
        return $this->data['title'];
    }

    /**
     * Should the title overwrite the default
     *
     * @return null|bool
     */
    public function getTitleOverwrite()
    {
        // not set so return null
        if (!isset($this->data['title_overwrite'])) {
            return;
        }

        // return value
        return ($this->data['title_overwrite'] == 'Y');
    }

    /**
     * Return the current value for an URL
     *
     * @return null|string
     */
    public function getURL()
    {
        // not set so return null
        if (!isset($this->data['url'])) {
            return;
        }

        // return value
        return urldecode($this->data['url']);
    }

    /**
     * Should the URL overwrite the default
     *
     * @return null|bool
     */
    public function getURLOverwrite()
    {
        // not set so return null
        if (!isset($this->data['url_overwrite'])) {
            return;
        }

        // return value
        return ($this->data['url_overwrite'] == 'Y');
    }

    /**
     * Add all element into the form
     */
    protected function loadForm()
    {
        // is the form submitted?
        if ($this->frm->isSubmitted()) {
            /**
             * If the fields are disabled we don't have any values in the post.
             * When an error occurs in the other fields of the form the meta-fields would be cleared
             * therefore we alter the POST so it contains the initial values.
             */
            if (!isset($_POST['page_title'])) {
                $_POST['page_title'] = (isset($this->data['title'])) ? $this->data['title'] : null;
            }
            if (!isset($_POST['meta_description'])) {
                $_POST['meta_description'] = (isset($this->data['description'])) ? $this->data['description'] : null;
            }
            if (!isset($_POST['meta_keywords'])) {
                $_POST['meta_keywords'] = (isset($this->data['keywords'])) ? $this->data['keywords'] : null;
            }
            if (!isset($_POST['url'])) {
                $_POST['url'] = (isset($this->data['url'])) ? $this->data['url'] : null;
            }
            if ($this->custom && !isset($_POST['meta_custom'])) {
                $_POST['meta_custom'] = (isset($this->data['custom'])) ? $this->data['custom'] : null;
            }
            if (!isset($_POST['seo_index'])) {
                $_POST['seo_index'] = (isset($this->data['data']['seo_index'])) ?
                    $this->data['data']['seo_index'] :
                    'none';
            }
            if (!isset($_POST['seo_follow'])) {
                $_POST['seo_follow'] = (isset($this->data['data']['seo_follow'])) ?
                    $this->data['data']['seo_follow'] :
                    'none';
            }
        }

        // add page title elements into the form
        $this->frm->addCheckbox(
            'page_title_overwrite',
            (isset($this->data['title_overwrite']) && $this->data['title_overwrite'] == 'Y')
        );
        $this->frm->addText('page_title', (isset($this->data['title'])) ? $this->data['title'] : null);

        // add meta description elements into the form
        $this->frm->addCheckbox(
            'meta_description_overwrite',
            (isset($this->data['description_overwrite']) && $this->data['description_overwrite'] == 'Y')
        );
        $this->frm->addText(
            'meta_description',
            (isset($this->data['description'])) ? $this->data['description'] : null
        );

        // add meta keywords elements into the form
        $this->frm->addCheckbox(
            'meta_keywords_overwrite',
            (isset($this->data['keywords_overwrite']) && $this->data['keywords_overwrite'] == 'Y')
        );
        $this->frm->addText('meta_keywords', (isset($this->data['keywords'])) ? $this->data['keywords'] : null);

        // add URL elements into the form
        $this->frm->addCheckbox(
            'url_overwrite',
            (isset($this->data['url_overwrite']) && $this->data['url_overwrite'] == 'Y')
        );
        $this->frm->addText('url', (isset($this->data['url'])) ? urldecode($this->data['url']) : null);

        // advanced SEO
        $indexValues = array(
            array('value' => 'none', 'label' => BackendLanguage::getLabel('None')),
            array('value' => 'index', 'label' => 'index'),
            array('value' => 'noindex', 'label' => 'noindex'),
        );
        $this->frm->addRadiobutton(
            'seo_index',
            $indexValues,
            (isset($this->data['data']['seo_index'])) ? $this->data['data']['seo_index'] : 'none'
        );
        $followValues = array(
            array('value' => 'none', 'label' => BackendLanguage::getLabel('None')),
            array('value' => 'follow', 'label' => 'follow'),
            array('value' => 'nofollow', 'label' => 'nofollow'),
        );
        $this->frm->addRadiobutton(
            'seo_follow',
            $followValues,
            (isset($this->data['data']['seo_follow'])) ? $this->data['data']['seo_follow'] : 'none'
        );

        // should we add the meta-custom field
        if ($this->custom) {
            // add meta custom element into the form
            $this->frm->addTextarea('meta_custom', (isset($this->data['custom'])) ? $this->data['custom'] : null);
        }

        $this->frm->addHidden('meta_id', $this->id);
        $this->frm->addHidden('base_field_name', $this->baseFieldName);
        $this->frm->addHidden('custom', $this->custom);
        $this->frm->addHidden('class_name', $this->callback['class']);
        $this->frm->addHidden('method_name', $this->callback['method']);
        $this->frm->addHidden('parameters', \SpoonFilter::htmlspecialchars(serialize($this->callback['parameters'])));
    }

    /**
     * Load a specific meta-record
     *
     * @param int $id The id of the record to load.
     *
     * @throws Exception If no meta-record exists with the provided id
     */
    protected function loadMeta($id)
    {
        $this->id = (int) $id;

        // get item
        $this->data = (array) BackendModel::getContainer()->get('database')->getRecord(
            'SELECT *
             FROM meta AS m
             WHERE m.id = ?',
            array($this->id)
        );

        // validate meta-record
        if (empty($this->data)) {
            throw new Exception('Meta-record doesn\'t exist.');
        }

        // unserialize data
        if (isset($this->data['data'])) {
            $this->data['data'] = @unserialize($this->data['data']);
        }
    }

    /**
     * Saves the meta object
     *
     * @param bool $update Should we update the record or insert a new one.
     *
     * @throws Exception If no meta id was provided.
     *
     * @return int
     *
     * @deprecated just use the entity for doctrine
     */
    public function save($update = false)
    {
        $this->validate();

        $update = (bool) $update;

        //serialize data for save
        if (!empty($this->data['data'])) {
            $this->data['data'] = serialize($this->data['data']);
        }

        // build meta
        $db = BackendModel::getContainer()->get('database');

        if ($this->id !== null && $update === true) {
            $db->update('meta', $this->data, 'id = ?', array($this->id));

            return $this->id;
        } else {
            unset($this->data['id']);

            return (int) $db->insert('meta', $this->data);
        }
    }

    /**
     * Set the callback to calculate an unique URL
     * REMARK: this method has to be public and static
     * REMARK: if you specify arguments they will be appended
     *
     * @param string $className Name of the class to use.
     * @param string $methodName Name of the method to use.
     * @param array $parameters Parameters to parse, they will be passed after ours.
     */
    public function setURLCallback($className, $methodName, $parameters = array())
    {
        $className = (string) $className;
        $methodName = (string) $methodName;
        $parameters = (array) $parameters;

        // store in property
        $this->callback = array('class' => $className, 'method' => $methodName, 'parameters' => $parameters);

        // re-load the form
        $this->loadForm();
    }

    /**
     * Validates the form
     * It checks if there is a value when a checkbox is checked
     */
    public function validate()
    {
        // page title overwrite is checked
        if ($this->frm->getField('page_title_overwrite')->isChecked()) {
            $this->frm->getField('page_title')->isFilled(BackendLanguage::err('FieldIsRequired'));
        }

        // meta description overwrite is checked
        if ($this->frm->getField('meta_description_overwrite')->isChecked()) {
            $this->frm->getField('meta_description')->isFilled(BackendLanguage::err('FieldIsRequired'));
        }

        // meta keywords overwrite is checked
        if ($this->frm->getField('meta_keywords_overwrite')->isChecked()) {
            $this->frm->getField('meta_keywords')->isFilled(BackendLanguage::err('FieldIsRequired'));
        }

        // URL overwrite is checked
        if ($this->frm->getField('url_overwrite')->isChecked()) {
            $this->frm->getField('url')->isFilled(BackendLanguage::err('FieldIsRequired'));
            $url = \SpoonFilter::htmlspecialcharsDecode($this->frm->getField('url')->getValue());
            $generatedUrl = $this->generateURL($url);

            // check if urls are different
            if (CommonUri::getUrl($url) != $generatedUrl) {
                $this->frm->getField('url')->addError(
                    BackendLanguage::err('URLAlreadyExists')
                );
            }
        }

        // if the form was submitted correctly the data array should be populated
        if ($this->frm->isCorrect()) {
            // get meta keywords
            $keywords = $this->frm->getField('meta_keywords_overwrite')->getActualValue(
                $this->frm->getField('meta_keywords')->getValue(),
                $this->frm->getField($this->baseFieldName)->getValue()
            );

            // get meta description
            $description = $this->frm->getField('meta_description_overwrite')->getActualValue(
                $this->frm->getField('meta_description')->getValue(),
                $this->frm->getField($this->baseFieldName)->getValue()
            );

            // get page title
            $title = $this->frm->getField('page_title_overwrite')->getActualValue(
                $this->frm->getField('page_title')->getValue(),
                $this->frm->getField($this->baseFieldName)->getValue()
            );

            // get URL
            $url = $this->frm->getField('url_overwrite')->getActualValue(
                \SpoonFilter::htmlspecialcharsDecode($this->frm->getField('url')->getValue()),
                \SpoonFilter::htmlspecialcharsDecode($this->frm->getField($this->baseFieldName)->getValue())
            );

            // get the real URL
            $url = $this->generateURL($url);

            // get meta custom
            if ($this->custom && $this->frm->getField('meta_custom')->isFilled()) {
                $custom = $this->frm->getField('meta_custom')->getValue();
            } else {
                $custom = null;
            }

            // set data
            $this->data['keywords'] = $keywords;
            $this->data['keywords_overwrite'] = $this->frm->getField('meta_keywords_overwrite')->getActualValue();
            $this->data['description'] = $description;
            $this->data['description_overwrite'] = $this->frm->getField('meta_description_overwrite')->getActualValue();
            $this->data['title'] = $title;
            $this->data['title_overwrite'] = $this->frm->getField('page_title_overwrite')->getActualValue();
            $this->data['url'] = $url;
            $this->data['url_overwrite'] = $this->frm->getField('url_overwrite')->getActualValue();
            $this->data['custom'] = $custom;
            if ($this->frm->getField('seo_index')->getValue() == 'none') {
                unset($this->data['data']['seo_index']);
            } else {
                $this->data['data']['seo_index'] = $this->frm->getField('seo_index')->getValue();
            }
            if ($this->frm->getField('seo_follow')->getValue() == 'none') {
                unset($this->data['data']['seo_follow']);
            } else {
                $this->data['data']['seo_follow'] = $this->frm->getField('seo_follow')->getValue();
            }
        }
    }

    /**
     * @return array
     */
    public function getData()
    {
        return $this->data;
    }

    /**
     * @return MetaEntity
     */
    public function getMetaEntity()
    {
        $this->validate();

        return MetaEntity::fromBackendMeta($this);
    }
}
