<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Requests;
use PayPal;

class Checkout extends Controller
{

	private $_apiContext;

	public function __construct() {

		$this->_apiContext = PayPal::ApiContext(env('PAYPAL_CLIENT_ID'), env('PAYPAL_SECRET'));

        $this->_apiContext->setConfig(array(
            'mode' => 'sandbox',
            'service.EndPoint' => 'https://api.sandbox.paypal.com',
            'http.ConnectionTimeOut' => 30,
            'log.LogEnabled' => true,
            'log.FileName' => storage_path('logs/paypal.log'),
            'log.LogLevel' => 'FINE'
        ));
	}

    // stripe credit card payment
    public function credit_card() {

    	// check amount
    	$amount = \Cart::total();

		if( $amount < 1 )
			throw new \Exception("Error. Total amount is not valid.");
    	
    	return view('checkout.credit-card');

    }

    // process credit card payment
    public function credit_card_processing( Request $r ) {

    	$this->validate( $r, [ 'stripeToken' => 'required', 
    	                		'customer' => 'required', 
    	                		'email' => 'required|email' ]  );

    	// set stripe secret
		\Stripe\Stripe::setApiKey(env('STRIPE_SECRET_KEY'));

		// Get the credit card details submitted by the form
		$token = $r->stripeToken;

		// Create the charge on Stripe's servers - this will charge the user's card
		try {

			// setup amount for stripe (in cents)
			$amount = \Cart::total();

			// check amount
			if( $amount < 1 )
				throw new \Exception("Error. Total amount can't be less than 1.00", 1);
				

			$amount *= 100;
			$amount = (int) $amount;

			// charge the money
		  	$charge = \Stripe\Charge::create(array(
		    	"amount" => $amount, // amount in cents, again
		    	"currency" => "usd",
		    	"source" => $token,
		    	"description" => "Web Domain(s)"
		    ));


		  	// save this order in database
		  	$order = new \App\Orders;
		  	$order->customer = $r->customer;
		  	$order->email = $r->email;
		  	$order->total = \Cart::total();
		  	$order->order_contents = \Cart::content();
		  	$order->payment_type = 'Stripe';
		  	$order->order_status = 'Paid';
		  	$order->order_date = date("Y-m-d H:i:s");
		  	$order->save();

		  	// update domains statuses to "SOLD"
		  	foreach( \Cart::content() as $domains ) 
		  	{
		  		$d = \App\Domain::find($domains->id);
		  		$d->domain_status = 'SOLD';
		  		$d->save();
		  	}


		  	// mail the user
		  	\Mail::send('emails.user-order-confirmation', ['order' => $order], function ($m) use ($order) {
		        $m->from(env('SENDING_EMAIL'), \App\Options::get_option('site_title'));

		        $m->to($order->email, $order->customer)->subject('Your Order Confirmation!');
		    });

		    // mail the admin
		  	\Mail::send('emails.admin-order-confirmation', ['order' => $order], function ($m) use ($order) {
		        $m->from(env('SENDING_EMAIL'), \App\Options::get_option('site_title'));
		        $m->replyTo($order->email, $order->customer);
		        $m->to(\App\Options::get_option('admin_email'), 'Admin')->subject('New Order Confirmation!');
		    });


		    // remove items from cart
		    \Cart::destroy();

		  	// redirect with success message ( checkout/success )
		  	return redirect('checkout/success');

		} catch(\Exception $e) {
		  	return redirect('checkout/credit-card')
		  				->withErrors([ $e->getMessage() ])
		  				->withInput();
		}
    }


    // success route
    public function success() {
    	return view('checkout.success');
    }

    // paypal route
    public function paypal() {

    	try {
	    	// setup payer object
	    	$payer = PayPal::Payer();
		    $payer->setPaymentMethod('paypal');

		    // set amount details
		    /*
		    $amountDetails = PayPal::Details();
		    $amountDetails->setSubtotal(\Cart::total());
		    $amountDetails->setTax(0);
		    $amountDetails->setShipping(0);*/

		    // setup amount
		    $amount = \PayPal::Amount();
		    $amount->setCurrency('USD');
		    $amount->setTotal(\Cart::total());
		   	//$amount->setDetails($amountDetails);

		    // setup item list
		    $itemsArray = array();

		    foreach( \Cart::content() as $domain ) {

		    	$item = PayPal::Item();

			    $item->setName('Domain ' . $domain->name )
	  				->setCurrency('USD')
	    		  	->setQuantity(1)
	    		  	->setPrice( $domain->price );

	    		 $itemsArray[] = $item;

			}

			// add item list
			$itemList = PayPal::ItemList();
			$itemList->setItems( $itemsArray );

		    // setup transaction
		    $transaction = PayPal::Transaction();
		    $transaction->setAmount( $amount );
		    $transaction->setItemList( $itemList );
		    $transaction->setDescription('Domains');

		    // setup url's
		    $redirectUrls = PayPal:: RedirectUrls();
		    $redirectUrls->setReturnUrl(action('Checkout@paypal_complete'));
		    $redirectUrls->setCancelUrl(action('Ajax@cart_contents'));

		    // setup payment object
		    $payment = PayPal::Payment();
		    $payment->setIntent('sale');
		    $payment->setPayer($payer);
		    $payment->setRedirectUrls($redirectUrls);
		    $payment->setTransactions(array($transaction));

		    // get redirect url
		    $response = $payment->create($this->_apiContext);
		    $redirectUrl = $response->links[1]->href;

		    return redirect( $redirectUrl );

		} catch( \Exception $e ) {

			dd($e);

		}

    }

    public function paypal_complete(Request $request) {

    	$this->validate( $request, [ 'paymentId' => 'required', 
    	                			'token' => 'required', 
    	                			'PayerID' => 'required' ] );

    	// setup variables
	    $id = $request->get('paymentId');
	    $token = $request->get('token');
	    $payer_id = $request->get('PayerID');

	    try{
		    // check payment by ID
		    $payment = PayPal::getById($id, $this->_apiContext);
		    $paymentExecution = PayPal::PaymentExecution();

		    // execute payment
		    $paymentExecution->setPayerId($payer_id);
		    $executePayment = $payment->execute($paymentExecution, $this->_apiContext);

		    // save this order in database
		  	$order = new \App\Orders;
		  	$order->customer = $payment->payer->payer_info->first_name . ' ' . $payment->payer->payer_info->last_name;
		  	$order->email = $payment->payer->payer_info->email;
		  	$order->total = \Cart::total();
		  	$order->order_contents = \Cart::content();
		  	$order->payment_type = 'Paypal';
		  	$order->order_status = 'Paid';
		  	$order->order_date = date("Y-m-d H:i:s");
		  	$order->save();

		  	// update domains statuses to "SOLD"
		  	foreach( \Cart::content() as $domains ) 
		  	{
		  		$d = \App\Domain::find($domains->id);
		  		$d->domain_status = 'SOLD';
		  		$d->save();
		  	}


		  	// mail the user
		  	\Mail::send('emails.user-order-confirmation', ['order' => $order], function ($m) use ($order) {
		        $m->from(env('SENDING_EMAIL'), \App\Options::get_option('site_title'));
		        
		        $m->to($order->email, $order->customer)->subject('Your Order Confirmation!');
		    });

		    // mail the admin
		  	\Mail::send('emails.admin-order-confirmation', ['order' => $order], function ($m) use ($order) {
		        $m->from(env('SENDING_EMAIL'), \App\Options::get_option('site_title'));
				$m->replyTo($order->email, $order->customer);
		        $m->to(\App\Options::get_option('admin_email'), 'Admin')->subject('New Order Confirmation!');
		    });


		    // remove items from cart
		    \Cart::destroy();

		  	// redirect with success message ( checkout/success )
		  	return redirect('checkout/success');

		} catch ( \Exception $e ) {
			return redirect('checkout')
		  				->withErrors([ $e->getMessage() ]);

		}
    	
    }

}
