<?php

namespace App\Controller\Member;

use App\Controller\Member\AppMemberController;
use Cake\Event\Event;
use Cake\Routing\Router;
use Cake\Network\Exception\NotFoundException;

class CampaignsController extends AppMemberController
{

    public function initialize()
    {
        parent::initialize();
    }

    public function beforeFilter(Event $event)
    {
        parent::beforeFilter($event);
        if (in_array($this->request->action, ['ipn'])) {
            $this->eventManager()->off($this->Csrf);
            $this->eventManager()->off($this->Security);
        }
        $this->Auth->allow(['ipn']);
    }

    public function isAuthorized($user = null)
    {
        // The owner of an article can edit and delete it
        if (in_array($this->request->action, ['pay', 'pause', 'resume'])) {
            $id = (int) $this->request->params['pass'][0];
            return $this->Campaigns->isOwnedBy($id, $user['id']);
        }

        return parent::isAuthorized($user);
    }
    
    protected function checkEnableAdvertising()
    {
        if( get_option('enable_advertising', 'yes') == 'no' ) {
            $this->Flash->error(__('Creating campaigns is currently disabled.'));
            return $this->redirect( ['controller' => 'Users', 'action' => 'dashboard' ] );
        }
    }

    public function view($id = null)
    {
        $this->checkEnableAdvertising();
        
        if (!$id) {
            throw new NotFoundException(__('Invalid campaign'));
        }

        $campaign = $this->Campaigns->findById($id)
            ->contain(['CampaignItems'])
            ->where(['user_id' => $this->Auth->user('id')])
            ->first();
        if (!$campaign) {
            throw new NotFoundException(__('Campaign Not Found'));
        }

        $this->set('campaign', $campaign);
    }

    public function index()
    {
        $this->checkEnableAdvertising();
        
        $conditions = [];
        
        $filter_fields = ['id', 'status', 'ad_type', 'name', 'other_fields'];
        
        //Transform POST into GET
        if ($this->request->is(['post', 'put']) && isset($this->request->data['Filter'])) {
            
            $filter_url = [];
            
            $filter_url['controller'] = $this->request->params['controller'];
            
            $filter_url['action'] = $this->request->params['action'];
            
            // We need to overwrite the page every time we change the parameters
            $filter_url['page'] = 1;

            // for each filter we will add a GET parameter for the generated url
            foreach ($this->request->data['Filter'] as $name => $value) {
                if (in_array($name, $filter_fields) && $value) {
                    // You might want to sanitize the $value here
                    // or even do a urlencode to be sure
                    $filter_url[$name] = urlencode($value);
                }
            }
            // now that we have generated an url with GET parameters,
            // we'll redirect to that page
            return $this->redirect($filter_url);
        } else {
            // Inspect all the named parameters to apply the filters
            foreach ($this->request->query as $param_name => $value) {
                if (in_array($param_name, $filter_fields)) {

                    if (in_array($param_name, ['name'])) {
                        $conditions[] = [
                            ['Campaigns.' . $param_name . ' LIKE' => '%' . $value . '%']
                        ];
                    } elseif (in_array($param_name, ['other_fields'])) {
                        $conditions['OR'] = [
                            ['Campaigns.website_title LIKE' => '%' . $value . '%'],
                            ['Campaigns.website_url LIKE' => '%' . $value . '%'],
                            ['Campaigns.banner_name LIKE' => '%' . $value . '%'],
                            ['Campaigns.banner_size LIKE' => '%' . $value . '%']
                        ];
                    } elseif (in_array($param_name, ['id', 'status', 'ad_type']) ) {
                        if( $param_name == 'status' && !in_array($value, ['Active', 'Paused', 'Canceled', 'Finished', 'Under Review', 'Pending Payment', 'Invalid Payment']) ) {
                            continue;
                        }
                        if( $param_name == 'ad_type' && !in_array($value, [1, 2]) ) {
                            continue;
                        }
                        $conditions['Campaigns.' . $param_name] = $value;
                    }
                    $this->request->data['Filter'][$param_name] = $value;
                }
            }
        }
        
        $query = $this->Campaigns->find()
            ->contain(['CampaignItems'])
            ->where(['user_id' => $this->Auth->user('id')])
            ->where($conditions);
        $campaigns = $this->paginate($query);

        $this->set('campaigns', $campaigns);
    }

    public function createInterstitial()
    {
        $this->checkEnableAdvertising();
        
        if( get_option('enable_interstitial', 'yes') == 'no' ) {
            $this->Flash->error(__('Creating interstitial campaigns is currently disabled.'));
            return $this->redirect( ['controller' => 'Users', 'action' => 'dashboard' ] );
        }
        
        $campaign = $this->Campaigns->newEntity(null, ['associated' => ['CampaignItems']]);
        $this->set('campaign', $campaign);
        
        if ($this->request->is('post')) {
            $campaign->user_id = $this->Auth->user('id');
            $campaign->ad_type = 1;
            $campaign->status = 'Pending Payment';

            $this->request->data['price'] = 0;

            foreach ($this->request->data['campaign_items'] as $key => $value) {
                if (empty($value['purchase'])) {
                    unset($this->request->data['campaign_items'][$key]);
                    continue;
                }
                $this->request->data['price'] += $value['purchase'] * $value['advertiser_price'];
            }
            
            if(count($this->request->data['campaign_items']) == 0){
                return $this->Flash->error(__('You must purchase at least from one country.'));
            }
            
            $campaign = $this->Campaigns->patchEntity($campaign, $this->request->data);

            if ($this->Campaigns->save($campaign)) {
                $this->Flash->success(__('Your campaign has been added. After payiny, we will review it and it will appear on the website.'));
                return $this->redirect(['action' => 'pay', $campaign->id]);
            } else {
                //debug($campaign->errors());
                $this->Flash->error(__('Unable to create your campaign.'));
            }
        }
        $this->set('campaign', $campaign);
    }
    
    public function createBanner()
    {
        $this->checkEnableAdvertising();
        
        if( get_option('enable_banner', 'yes') == 'no' ) {
            $this->Flash->error(__('Creating banner campaigns is currently disabled.'));
            return $this->redirect( ['controller' => 'Users', 'action' => 'dashboard' ] );
        }
        
        $campaign = $this->Campaigns->newEntity(null, ['associated' => ['CampaignItems']]);
        $this->set('campaign', $campaign);
        
        if ($this->request->is('post')) {
            $campaign->user_id = $this->Auth->user('id');
            $campaign->ad_type = 2;
            $campaign->status = 'Pending Payment';

            $this->request->data['price'] = 0;

            foreach ($this->request->data['campaign_items'] as $key => $value) {
                if (empty($value['purchase'])) {
                    unset($this->request->data['campaign_items'][$key]);
                    continue;
                }
                $this->request->data['price'] += $value['purchase'] * $value['advertiser_price'];
            }
            
            if(count($this->request->data['campaign_items']) == 0){
                return $this->Flash->error(__('You must purchase at least from one country.'));
            }

            $campaign = $this->Campaigns->patchEntity($campaign, $this->request->data);

            if ($this->Campaigns->save($campaign)) {
                $this->Flash->success(__('Your campaign has been added. After payiny, we will review it and it will appear on the website.'));
                return $this->redirect(['action' => 'pay', $campaign->id]);
            } else {
                $this->Flash->error(__('Unable to create your campaign.'));
            }
        }
        $this->set('campaign', $campaign);
    }

    public function pay($id)
    {
        $this->checkEnableAdvertising();
        
        $campaign = $this->Campaigns->findById($id)
            ->where([
                'user_id' => $this->Auth->user('id'),
                'status' => 'Pending Payment'
            ])
            ->first();

        if (!$campaign) {
            $this->Flash->success(__('Campaign not found'));
            return $this->redirect(['action' => 'view', $id]);
        }
        
        $this->set('campaign', $campaign);
    }
    
    public function checkout()
    {
        $this->autoRender = false;

        $this->response->type('json');
        
        if (!$this->request->is('ajax')) {
            $content = [
                'status' => 'error',
                'message' => __('Bad Request.'),
                'form' => ''
            ];
            $this->response->body(json_encode($content));
            return $this->response;
        }
        
        $campaign = $this->Campaigns->findById($this->request->data['id'])
            ->where(['user_id' => $this->Auth->user('id')])
            ->first();
        
        if (!$campaign) {
            $content = [
                'status' => 'error',
                'message' => __('Bad Request.'),
                'form' => ''
            ];
            $this->response->body(json_encode($content));
            return $this->response;
        }
        
        if ('paypal' == $this->request->data['payment_method']) {
            
            $return_url = Router::url(['controller' => 'Campaigns', 'action' => 'view', $campaign->id], true);

            $paymentData = [
                'business' => get_option('paypal_email'),
                'cmd' => '_xclick',
                'currency_code' => get_option('currency_code'),
                'amount' => $campaign->price,
                'item_name' => 'Advertising Campaign',
                'item_number' => '#'.$campaign->id,
                'page_style' => 'paypal',
                'return' => $return_url,
                'rm' => '0',
                'cancel_return' => $return_url,
                'custom' => $campaign->id,
                'no_shipping' => 1,
                'lc' => 'US'
            ];
            
            $url = 'https://www.sandbox.paypal.com/cgi-bin/webscr';

            if (get_option('paypal_sandbox', 'no') == 'no') {
                $url = 'https://www.paypal.com/cgi-bin/webscr';
            }

            $form = $this->redirect_post($url, $paymentData);
            
            $campaign->payment_method = 'paypal';
            $this->Campaigns->save($campaign);
            
            $content = [
                'status' => 'success',
                'message' => '',
                'form' => $form
            ];
            $this->response->body(json_encode($content));
            return $this->response;
        }

        if ('payza' == $this->request->data['payment_method']) {
            
            $return_url = Router::url(['controller' => 'Campaigns', 'action' => 'view', $campaign->id], true);
            $alert_url = Router::url(['controller' => 'Campaigns', 'action' => 'ipn'], true);

            $paymentData = [
                'ap_merchant' => get_option('payza_email'),
                'apc_1' => $campaign->id,
                'ap_purchasetype' => 'service',
                'ap_amount' => $campaign->price,
                'ap_quantity' => 1,
                'ap_itemname' => 'Advertising Campaign',
                'ap_itemcode' => '#'.$campaign->id,
                'ap_currency' => get_option('currency_code'),
                'ap_returnurl' => $return_url,
                'ap_cancelurl' => $return_url,
                'ap_alerturl' => $alert_url,
                'ap_ipnversion' => 2,
            ];
            
            $url = 'https://secure.payza.com/checkout';
            
            $form = $this->redirect_post($url, $paymentData);
            
            $campaign->payment_method = 'payza';
            $this->Campaigns->save($campaign);
            
            $content = [
                'status' => 'success',
                'message' => '',
                'form' => $form
            ];
            $this->response->body(json_encode($content));
            return $this->response;
            
        }

    }
    
    public function ipn()
    {
        $this->autoRender = false;
        
        if (!empty($this->request->data)) {
            
            // PayPal IPN
            if( isset($this->request->data['txn_id']) ) {
                $this->ipn_paypal($this->request->data);
            }
            
            // Payza IPN
            if( isset($this->request->data['ap_merchant']) ) {
                $this->ipn_payza($this->request->data);
            }
            
        }

    }
    
    protected function ipn_payza($data)
    {
        $token = [
            'token' => urlencode($data['token'])
        ];
        
        // https://dev.payza.com/resources/references/ipn-variables

        $url = 'https://secure.payza.com/ipn2.ashx';

        $res = curlRequest($url, 'POST', $token );

        if(strlen($res) > 0) {
        
            $campaign_id = (int) $data['apc_1'];
            $campaign = $this->Campaigns->get($campaign_id);

            if (urldecode($res) != "INVALID TOKEN") {
                switch ($data['ap_transactionstate']) {
                    case 'Refunded':
                        $campaign->status = 'Refunded';
                        break;
                    case 'Completed':
                        $campaign->status = 'Under Review';
                        break;
                }

                $this->Campaigns->save($campaign);
                $message = 'VERIFIED';
            } else {
                $campaign->status = 'Invalid Payment';
                $this->Campaigns->save($campaign);
                $message = 'INVALID';
            }

        }
    }
    
    protected function ipn_paypal($data)
    {
        $data['cmd'] = '_notify-validate';

        // https://developer.paypal.com/docs/classic/ipn/integration-guide/IPNTesting/?mark=IPN%20troubleshoot#invalid

        $paypalURL = 'https://www.sandbox.paypal.com/cgi-bin/webscr?';

        if (get_option('paypal_sandbox', 'no') == 'no') {
            $paypalURL = 'https://www.paypal.com/cgi-bin/webscr?';
        }

        $res = curlRequest($paypalURL, 'POST', $data );

        $campaign_id = (int) $data['custom'];
        $campaign = $this->Campaigns->get($campaign_id);

        if (strcmp($res, "VERIFIED") == 0) {
            switch ($data['payment_status']) {
                case 'Refunded':
                    $campaign->status = 'Refunded';
                    break;
                case 'Completed':
                    $campaign->status = 'Under Review';
                    break;
            }

            $this->Campaigns->save($campaign);
            $message = 'VERIFIED';
        } elseif (strcmp($res, "INVALID") == 0) {
            $campaign->status = 'Invalid Payment';
            $this->Campaigns->save($campaign);
            $message = 'INVALID';
        }
    }
    
    protected function redirect_post($url, array $data)
    {
        ob_start();
        ?>
        <form id="checkout-redirect-form" method="post" action="<?= $url; ?>">
            <?php
            if ( !is_null($data) ) {
                foreach ($data as $k => $v) {
                    echo '<input type="hidden" name="' . $k . '" value="' . $v . '"> ';
                }
            }
            ?>
        </form>
        <?php
        $form = ob_get_contents();
        ob_end_clean();
        
        return $form;
    }

    public function pause($id)
    {
        $this->checkEnableAdvertising();
        
        $this->request->allowMethod(['post', 'put']);

        $campaign = $this->Campaigns->findById($id)
            ->where(['user_id' => $this->Auth->user('id')])
            ->where(['status' => 'Active'])
            ->first();

        if (!$campaign) {
            $this->Flash->success(__('Campaign not found'));
            return $this->redirect(['action' => 'index']);
        }

        $campaign->status = 'Paused';
        $this->Campaigns->save($campaign);

        return $this->redirect(['action' => 'index']);
    }

    public function resume($id)
    {
        $this->checkEnableAdvertising();
        
        $this->request->allowMethod(['post', 'put']);

        $campaign = $this->Campaigns->findById($id)
            ->where(['user_id' => $this->Auth->user('id')])
            ->where(['status' => 'Paused'])
            ->first();

        if (!$campaign) {
            $this->Flash->success(__('Campaign not found'));
            return $this->redirect(['action' => 'index']);
        }

        $campaign->status = 'Active';
        $this->Campaigns->save($campaign);

        return $this->redirect(['action' => 'index']);
    }
}
