<?php

namespace App\Model\Table;

use Cake\ORM\Table;
use Cake\Validation\Validator;
use Cake\Auth\DefaultPasswordHasher;

class UsersTable extends Table
{

    public function initialize(array $config)
    {
        $this->hasMany('Campaigns');
        $this->hasMany('Links');
        $this->hasMany('Statistics');
        $this->hasMany('Withdraws');
        $this->addBehavior('Timestamp');
    }

    public function validationDefault(Validator $validator)
    {
        return $validator
                ->notEmpty('username', 'A username is required')
                ->add('username', [
                    'alphaNumeric' => [
                        'rule' => ['alphaNumeric'],
                        'message' => __('alphaNumeric Only')
                    ],
                    'minLength' => [
                        'rule' => ['minLength', 5],
                        'message' => __('Minimum Length 5')
                    ],
                    'maxLength' => [
                        'rule' => ['maxLength', 15],
                        'message' => __('Maximum Length 15')
                    ]
                ])
                ->add('username', [
                    'unique' => [
                        'rule' => 'validateUnique',
                        'provider' => 'table',
                        'message' => __('Username already exists')
                    ]
                ])
                ->add('username_compare', [
                    'compare' => [
                        'rule' => ['compareWith', 'username'],
                        'message' => __('Not the same')
                    ]
                ])
                ->notEmpty('password', 'A password is required')
                ->add('password', [
                    'minLength' => [
                        'rule' => ['minLength', 5],
                        'message' => __('Minimum Length 5')
                    ],
                    'maxLength' => [
                        'rule' => ['maxLength', 25],
                        'message' => __('Maximum Length 25')
                    ]
                ])
                ->add('password_compare', [
                    'compare' => [
                        'rule' => ['compareWith', 'password'],
                        'message' => __('Not the same')
                    ]
                ])
                ->notEmpty('email', 'An email is required')
                ->add('email', 'validFormat', [
                    'rule' => 'email',
                    'message' => __('E-mail must be valid')
                ])
                ->add('email', [
                    'unique' => [
                        'rule' => 'validateUnique',
                        'provider' => 'table',
                        'message' => __('E-mail must be unique')
                    ]
                ])
                ->add('email_compare', [
                    'compare' => [
                        'rule' => ['compareWith', 'email'],
                        'message' => __('Not the same')
                    ]
                ])
                ->notEmpty('first_name', __('This value should not be blank.'))
                ->notEmpty('last_name', __('This value should not be blank.'))
                ->notEmpty('address1', __('This value should not be blank.'))
                ->notEmpty('city', __('This value should not be blank.'))
                ->notEmpty('state', __('This value should not be blank.'))
                ->notEmpty('zip', __('This value should not be blank.'))
                ->notEmpty('country', __('This value should not be blank.'))
                ->notEmpty('phone_number', __('This value should not be blank.'))
                ->notEmpty('withdrawal_method', __('This value should not be blank.'))
                ->notEmpty('withdrawal_email', __('This value should not be blank.'))
                ->add('withdrawal_email', 'validFormat', [
                    'rule' => 'email',
                    'message' => __('E-mail must be valid')
                ]);
    }

    public function validationChangeEmail(Validator $validator)
    {
        //$validator = $this->validateDefault($validator);
        return $validator
                ->notEmpty('temp_email', 'An email is required')
                ->add('temp_email', 'validFormat', [
                    'rule' => 'email',
                    'message' => __('E-mail must be valid')
                ])
                ->add('temp_email', 'custom', [
                    'rule' => function ($value, $context) {
                        $count = $this->find('all')
                            ->where(['email' => $value])
                            ->count();
                        if ($count > 0) {
                            return false;
                        } else {
                            return true;
                        }
                    },
                        'message' => __('E-mail must be unique')
                    ])
                    ->add('confirm_email', [
                        'compare' => [
                            'rule' => ['compareWith', 'temp_email'],
                            'message' => __('Not the same')
                        ]
                    ]);
    }

    public function validationChangePassword(Validator $validator)
    {
        //$validator = $this->validateDefault($validator);
        return $validator
            ->notEmpty('current_password', 'Please enter current password.')
            ->add('current_password', 'custom', [
                'rule' => function ($value, $context) {
                    $user = $this->findById($context['data']['id'])->first();
                    return (new DefaultPasswordHasher)->check($value, $user->password);
                },
                'message' => __('Please enter current password.')
            ])
            ->notEmpty('password', 'A password is required')
            ->add('password', [
                'minLength' => [
                    'rule' => ['minLength', 5],
                    'message' => __('Minimum Length 5')
                ],
                'maxLength' => [
                    'rule' => ['maxLength', 25],
                    'message' => __('Maximum Length 25')
                ]
            ])
            ->add('confirm_password', [
                'compare' => [
                    'rule' => ['compareWith', 'password'],
                    'message' => __('Not the same')
                ]
            ]);
    }

    public function validationForgotPassword(Validator $validator)
    {
        //$validator = $this->validateDefault($validator);
        return $validator
            ->notEmpty('email', 'An email is required')
            ->add('email', 'validFormat', [
                'rule' => 'email',
                'message' => __('E-mail must be valid')
            ])
            ->notEmpty('password', 'A password is required')
            ->add('password', [
                'minLength' => [
                    'rule' => ['minLength', 5],
                    'message' => __('Minimum Length 5')
                ],
                'maxLength' => [
                    'rule' => ['maxLength', 25],
                    'message' => __('Maximum Length 25')
                ]
            ])
            ->add('confirm_password', [
                'compare' => [
                    'rule' => ['compareWith', 'password'],
                    'message' => __('Not the same')
                ]
            ]);
    }
}
