<?php

/**
 *
 * PHP Pro Bid $Id$ OjGvVJ7T/TwhDrLXyFcYi231mC0/haSd/RZywZe+rsw=
 *
 * @link        http://www.phpprobid.com
 * @copyright   Copyright (c) 2014 Only For Testing BBC No Leech & CodeCube SRL
 * @license     http://www.phpprobid.com/license Commercial License
 *
 * @version     7.2
 */
/**
 * thumbnail view helper class
 *
 * this view helper will generate thumbnails for images, and if the data input is of a different type, then it will
 * add placeholders
 */

namespace Ppb\View\Helper;

use Cube\View\Helper\AbstractHelper,
    Cube\Controller\Front,
    Ppb\Service;

class Thumbnail extends AbstractHelper
{
    const DEFAULT_EXTENSION = 'img';

    /**
     * the path for the "no image" pic
     */
    const IMAGE_PLACEHOLDER = 'image.png';

    /**
     * video file placeholder
     */
    const VIDEO_PLACEHOLDER = 'video.png';

    /**
     * digital download file placeholder
     */
    const DOWNLOAD_PLACEHOLDER = 'download.png';

    /**
     * the path for the "broken image" pic
     */
    const BROKEN_IMAGE = 'broken.gif';


    /**
     * minimum width in pixels of a thumbnail image
     */
    const MIN_WIDTH = 30;

    /**
     * maximum width in pixels of a thumbnail images
     */
    const MAX_WIDTH = 1200;

    /**
     *
     * this variable will hold the name of the file which will be displayed
     *
     * @var string
     */
    protected $_name;

    /**
     *
     * the extension of the file
     *
     * @var string
     */
    protected $_extension;

    /**
     *
     * we set the value of the thumbnail to be output
     *
     * @var integer
     */
    protected $_width = 80;

    /**
     *
     * square thumbnail
     *
     * @var bool
     */
    protected $_square = true;

    /**
     *
     * application base url
     *
     * @var string
     */
    protected $_baseUrl = null;

    /**
     *
     * allowed extensions
     *
     * @var array
     */
    protected $_extensions = array(
        'gif', 'jpg', 'jpeg', 'png', 'img',
        'avi', 'mpg', 'mpeg', 'mov', 'mp4', 'flv',
    );

    /**
     *
     * get file name
     *
     * @return string
     */
    public function getName()
    {
        return $this->_name;
    }

    /**
     *
     * set file name
     *
     * @param string $name
     *
     * @return $this
     */
    public function setName($name)
    {
        $name = preg_replace("/[^a-zA-Z0-9_-]/", '', $name);
        $this->_name = (string)$name;

        return $this;
    }

    /**
     *
     * get file extension
     *
     * @return string
     */
    public function getExtension()
    {
        return $this->_extension;
    }

    /**
     *
     * set file extension
     *
     * @param string $extension
     *
     * @return $this
     */
    public function setExtension($extension)
    {
        if (!in_array($extension, $this->_extensions)) {
            $extension = self::DEFAULT_EXTENSION;
        }

        $this->_extension = (string)$extension;

        return $this;
    }

    /**
     *
     * get thumbnail width
     *
     * @return integer
     */
    public function getWidth()
    {
        return $this->_width;
    }

    /**
     *
     * set thumbnail width
     *
     * @param int $width
     *
     * @return $this
     */
    public function setWidth($width)
    {
        $width = intval($width);

        if ($width < self::MIN_WIDTH) {
            $width = self::MIN_WIDTH;
        }

        if ($width > self::MAX_WIDTH && self::MAX_WIDTH > 0) {
            $width = self::MAX_WIDTH;
        }

        $this->_width = $width;

        return $this;
    }

    /**
     *
     * get square option
     *
     * @return bool
     */
    public function getSquare()
    {
        return $this->_square;
    }

    /**
     *
     * set square option
     *
     * @param bool $square
     *
     * @return $this
     */
    public function setSquare($square = true)
    {
        $this->_square = (bool)$square;

        return $this;
    }

    /**
     *
     * get base url
     *
     * @return string
     */
    public function getBaseUrl()
    {
        if ($this->_baseUrl === null) {
            $this->setBaseUrl(
                Front::getInstance()->getRequest()->getBaseUrl());
        }

        return $this->_baseUrl;
    }

    /**
     *
     * set base url
     *
     * @param string $baseUrl
     *
     * @return $this
     */
    public function setBaseUrl($baseUrl)
    {
        $this->_baseUrl = $baseUrl;

        return $this;
    }

    /**
     *
     * generate cached thumbnail and return image link
     * (has backwards compatibility for v6's uplimg folder)
     *
     * @param string    $image
     * @param int|null  $width
     * @param bool|null $square
     * @param array     $params
     *
     * @return string
     */
    public function generateLink($image, $width = null, $square = null, $params = array())
    {
        $this->setWidth($width)
            ->setSquare($square);

        $type = (isset($params['type'])) ? $params['type'] : Service\ListingsMedia::TYPE_IMAGE;

        switch ($type) {
            case Service\ListingsMedia::TYPE_VIDEO:
                $image = \Ppb\Utility::getPath('img') . \Ppb\Utility::URI_DELIMITER . self::VIDEO_PLACEHOLDER;
                break;
            case Service\ListingsMedia::TYPE_DOWNLOAD:
                $image = \Ppb\Utility::getPath('img') . \Ppb\Utility::URI_DELIMITER . self::DOWNLOAD_PLACEHOLDER;
                break;

            default:
                if ($image == null) {
                    $image = \Ppb\Utility::getPath('img') . \Ppb\Utility::URI_DELIMITER . self::IMAGE_PLACEHOLDER;
                }
                else {
                    if (!preg_match('#^http(s)?://(.*)+$#i', $image) && !preg_match('#^uplimg/(.*)+$#i', $image)) {
                        $image = \Ppb\Utility::getFolder('uploads') . \Ppb\Utility::URI_DELIMITER . $image;
                    }
                }
                break;
        }

        return $this->getBaseUrl() . \Ppb\Utility::URI_DELIMITER
        . \Ppb\Utility::getFolder('cache') . \Ppb\Utility::URI_DELIMITER
        . $this->_generateThumb($image);
    }

    /**
     *
     * generates a path to a media file
     * if we have a remote image or a v6 image, return the path unmodified
     *
     * @param string $image
     *
     * @return string
     */
    public function generateImagePath($image)
    {
        if (preg_match('#^uplimg/(.*)+$#i', $image)) {
            // we have a v6 image - add base url
            return $this->getBaseUrl() . '/' . $image;
        }
        else if (!preg_match('#^http(s)?://(.*)+$#i', $image)) {
            // we have a v7 image - add base url and uploads folder
            return $this->getBaseUrl() . \Ppb\Utility::URI_DELIMITER
            . \Ppb\Utility::getFolder('uploads') . \Ppb\Utility::URI_DELIMITER
            . $image;
        }

        return $image;
    }

    /**
     *
     * get cache name
     *
     * @return string
     */
    protected function _getCacheName()
    {
        return $this->getName() . '-'
        . $this->getWidth() . '-'
        . (($this->getSquare()) ? 'sq' : 're') . '.png';
    }

    /**
     *
     * generates a thumbnail of a given image, using the width and square parameters
     *
     * @param string $image
     *
     * @return string
     */
    protected function _generateThumb($image)
    {
        $fileName = null;
        $thumbImgWidth = null;
        $thumbImgHeight = null;

        $pathInfo = pathinfo($image);

        $baseName = (isset($pathInfo['filename'])) ? $pathInfo['filename'] : null;
        $extension = (isset($pathInfo['extension'])) ? $pathInfo['extension'] : null;

        $this->setName($baseName)
            ->setExtension($extension);

        $cacheName = $this->_getCacheName();

        $cacheFilePath = \Ppb\Utility::getPath('cache') . DIRECTORY_SEPARATOR . $cacheName;

        if (file_exists($cacheFilePath)) {
            return $cacheName;
        }
        else {
            $output = $this->createResizedImage($image);

            touch($cacheFilePath);
            imagejpeg($output, $cacheFilePath, 90);
            imagedestroy($output);

            $fileName = $cacheName;

            return $fileName;
        }
    }

    /**
     *
     * create resized image
     *
     * @param string $image
     *
     * @return string
     */
    public function createResizedImage($image)
    {
        $fileName = null;
        $thumbImgWidth = null;
        $thumbImgHeight = null;

        $imgInfo = @getimagesize($image);
        list($imgWidth, $imgHeight, $imgType) = $imgInfo;

        if ($imgInfo === false || $imgType > 3) {
            $image = \Ppb\Utility::getPath('img') . \Ppb\Utility::URI_DELIMITER . self::BROKEN_IMAGE;
            $imgInfo = @getimagesize($image);
            list($imgWidth, $imgHeight, $imgType) = $imgInfo;
        }

        $width = $this->getWidth();
        $square = $this->getSquare();

        // now generate the thumbnail and save it as a cache image as well
        $resizeWidth = ($imgWidth > $imgHeight) ? $imgWidth : $imgHeight;

        if ($resizeWidth < $width) {
            $width = $resizeWidth;
        }

        switch ($imgInfo['mime']) {
            case 'image/gif':
                $imageCreateFunction = 'ImageCreateFromGIF';
                break;
            case 'image/png':
                $imageCreateFunction = 'ImageCreateFromPNG';
                break;
            default:
                $imageCreateFunction = 'ImageCreateFromJPEG';
                break;
        }

        if ($square === true) {
            $thumbImgWidth = $width;
            $thumbImgHeight = $width;

            if ($imgWidth > $imgHeight) {
                $shrinkRatio = $imgWidth / $width;
                $height = $imgHeight / $shrinkRatio;
                $startX = 0;
                $startY = (abs($thumbImgHeight - $height)) / 2;
            }
            else {
                $shrinkRatio = $imgHeight / $width;
                $height = $width;
                $width = $imgWidth / $shrinkRatio;
                $startY = 0;
                $startX = (abs($thumbImgWidth - $width)) / 2;
            }
        }
        else {
            $shrinkRatio = $imgWidth / $width;
            $height = $imgHeight / $shrinkRatio;
            $startX = 0;
            $startY = 0;
        }

        $thumbnail = $imageCreateFunction($image);

        if (!$thumbnail) {
            $output = imagecreate($width, $height); /* Create a blank image */

            $white = imagecolorallocate($output, 255, 255, 255);
            $black = imagecolorallocate($output, 0, 0, 0);

            imagefilledrectangle($output, 0, 0, 150, 30, $white);

            imagestring($output, 1, 5, 5, 'Error loading ' . $image, $black); /* Output an error message */
        }
        else {
            if ($square) {
                $output = imagecreatetruecolor($thumbImgWidth, $thumbImgHeight);
            }
            else {
                $output = imagecreatetruecolor($width, $height);
            }

            $backgroundColor = imagecolorallocate($output, 255, 255, 255);
            imagefill($output, 0, 0, $backgroundColor);

            imagecopyresampled($output, $thumbnail, $startX, $startY, 0, 0, $width, $height, $imgWidth,
                $imgHeight);
        }

        return $output;
    }

    /**
     *
     * the helper will display a thumbnail image,
     * or in case media is uploaded, it will add a default thumbnail
     *
     * @param string $image
     * @param int    $width
     * @param string $square (Y|N|null)
     * @param array  $params (alt - string; zoom - bool; type - string)
     *
     * @return string|$this
     */
    public function thumbnail($image = null, $width = null, $square = null, $params = array())
    {
        $dataSrc = null;

        if ($image === null && $width === null && $square === null) {
            return $this;
        }

        $zoom = (isset($params['zoom'])) ? $params['zoom'] : false;

        if ($zoom === true) {
            $dataSrc = 'data-src="' . $this->generateImagePath($image) . '"';

            // add js
            /** @var \Cube\View\Helper\Script $helper */
            $helper = $this->getView()->getHelper('script');
            $helper->addBodyCode('<script type="text/javascript" src="' . $this->getBaseUrl() . '/js/jquery.zoom.min.js"></script>')
                ->addBodyCode("<script type=\"text/javascript\">
                    $(document).ready(function() {
                        $('a.zoom').zoom();
                    });
                </script>");
        }

        $class = (isset($params['class'])) ? $params['class'] : 'img-thumbnail img-responsive';

        return '<img class="' . $class . '" src="'
        . $this->generateLink($image, $width, $square, $params) . '" '
        . $dataSrc . ' '
        . ((!empty($params['alt'])) ? 'alt="' . str_ireplace('"', '', $params['alt']) . '"' : '')
        . $this->_endTag;

    }

}

