<?php

/**
 *
 * PHP Pro Bid $Id$ NTy2mpVmrlG3cUmO3bzWz8vBnIAT8ztL2ZDN030WzGA=
 *
 * @link        http://www.phpprobid.com
 * @copyright   Copyright (c) 2014 Only For Testing BBC No Leech & CodeCube SRL
 * @license     http://www.phpprobid.com/license Commercial License
 *
 * @version     7.0
 */
/**
 * locations table service class
 * TODO: add children like on the categories service
 */

namespace Ppb\Service\Table;

use Cube\Db\Expr,
        Ppb\Db\Table\Locations as LocationsTable,
        Cube\Navigation,
        Cube\Navigation\Page\AbstractPage;

class Locations extends AbstractServiceTable
{

    /**
     *
     * locations table navigation object
     *
     * @var \Cube\Navigation
     */
    protected $_data;

    /**
     *
     * class constructor
     */
    public function __construct()
    {
        parent::__construct();

        $this->setInsertRows(3);
        $this->setTable(
            new LocationsTable());
    }


    /**
     *
     * get locations table data
     *
     * @return \Cube\Navigation
     */
    public function getData()
    {
        if (empty($this->_data)) {
            $this->setData();
        }

        return $this->_data;
    }

    /**
     *
     * set locations table data.
     * This data will be used for traversing the categories tree
     *
     * @param string|\Cube\Db\Select $where SQL where clause, or a select object
     * @param array|\Traversable     $data  Optional. Custom categories data
     * @return \Ppb\Service\Table\Categories
     */
    public function setData($where = null, array $data = null)
    {
        if ($data === null) {
            $locations = $this->_table->fetchAll($where);

            $data = array();
            foreach ($locations as $row) {
                $data[$row['parent_id']][] = array(
                    'id'      => (int)$row['id'],
                    'label'   => $row['name'],
                    'order'   => (int)$row['order_id'],
                    'isoCode' => $row['iso_code'],
                    'type'    => '\\Ppb\\Navigation\\Page\\Location',
                );
            }

            reset($data);

            $tree = $this->_createTree($data, current($data));

            $this->_data = new Navigation($tree);
        }
        else {
            $this->_data = $data;
        }

        return $this;
    }

    protected function _createTree(&$list, $parent)
    {
        $tree = array();

        foreach ($parent as $row) {
            if (isset($list[$row['id']])) {
                $row['pages'] = $this->_createTree($list, $list[$row['id']]);
            }

            $tree[] = $row;
        }

        return $tree;
    }

    /**
     *
     * get all table columns needed to generate the
     * locations management table in the admin area
     *
     * @return array
     */
    public function getColumns()
    {
        return array(
            array(
                'label'      => '',
                'class'      => 'size-tiny',
                'element_id' => null,
                'children'   => array(
                    'key'   => 'parent_id',
                    'value' => 'id',
                ),
            ),
            array(
                'label'      => 'Name',
                'element_id' => 'name',
            ),
            array(
                'label'      => 'ISO Code',
                'class'      => 'size-mini',
                'element_id' => 'iso_code',
            ),
            array(
                'label'      => 'Order ID',
                'class'      => 'size-mini',
                'element_id' => 'order_id',
            ),
            array(
                'label'      => 'Delete',
                'class'      => 'size-mini',
                'element_id' => array(
                    'id', 'delete'
                ),
            ),
        );
    }

    /**
     *
     * get all form elements that are needed to generate the
     * locations management table in the admin area
     *
     * @return array
     */
    public function getElements()
    {
        return array(
            array(
                'id'      => 'id',
                'element' => 'hidden',
            ),
            array(
                'id'         => 'name',
                'element'    => 'text',
                'attributes' => array(
                    'class' => 'form-control input-large',
                ),
            ),
            array(
                'id'         => 'iso_code',
                'element'    => 'text',
                'attributes' => array(
                    'class' => 'form-control input-mini',
                ),
            ),
            array(
                'id'         => 'order_id',
                'element'    => 'text',
                'attributes' => array(
                    'class' => 'form-control input-mini',
                ),
            ),
            array(
                'id'      => 'delete',
                'element' => 'checkbox',
            ),
        );
    }

    /**
     *
     * get locations
     * to be used for the locations selector
     *
     * @param int     $id
     * @param string  $column
     * @param boolean $null whether to show a default option
     * @return array
     */
    public function getMultiOptions($id = null, $column = null, $null = false)
    {
        $data = array();

        if ($null !== false) {
            $data[0] = $null;
        }

        $select = $this->_table->select()
                ->order(array('order_id ASC', 'name ASC'));
        if ($id !== null) {
            if (is_array($id)) {
                $select->where("id IN (?)", new Expr(implode(', ', $id)));
            }
            else {
                $select->where("parent_id = ?", $id);
            }
        }
        else {
            $select->where("parent_id IS NULL");
        }

        $rows = $this->_table->fetchAll($select);

        foreach ($rows as $row) {
            if ($column === null) {
                $column = 'name';
            }
            $data[(string)$row['id']] = $row[$column];
        }

        return $data;
    }

    /**
     *
     * fetches all matched rows
     *
     * @param string|\Cube\Db\Select $where SQL where clause, or a select object
     * @param string|array           $order
     * @param int                    $count
     * @param int                    $offset
     * @return array
     */
    public function fetchAll($where = null, $order = null, $count = null, $offset = null)
    {
        if ($where === null) {
            $where = 'parent_id IS NULL';
        }
        if ($order === null) {
            $order = 'order_id ASC, name ASC';
        }

        return parent::fetchAll($where, $order, $count, $offset);
    }

    /**
     *
     * save data in the table (update if an id exists or insert otherwise)
     *
     *
     * @param array $data
     * @return \Ppb\Service\Table\AbstractServiceTable
     * @throws \InvalidArgumentException
     */
    public function save(array $data)
    {

        if (isset($data['parent_id'])) {
            $parentId = (string)$data['parent_id'];
            unset($data['parent_id']);

            foreach ($data['id'] as $key => $value) {
                $data['parent_id'][$key] = $parentId;
            }
        }

        return parent::save($data);
    }


    /**
     *
     * get location breadcrumbs array
     *
     * @param int $id
     * @return array
     */
    public function getBreadcrumbs($id)
    {
        $breadcrumbs = array();
        $page = $this->getData()->findOneBy('id', $id);

        if ($page instanceof AbstractPage) {

            $breadcrumbs[(int)$page->id] = $page->label;

            while (($parent = $page->getParent()) instanceof AbstractPage) {
                $breadcrumbs[(int)$parent->id] = $parent->label;
                $page = $parent;
            }
        }

        return array_reverse($breadcrumbs, true);
    }

}

