<?php

/**
 * 
 * PHP Pro Bid $Id$ MmO0NMhWYwhCWcxK0vaXQglE15jXqK2q02rmIgNhJfo=
 * 
 * @link        http://www.phpprobid.com
 * @copyright   Copyright (c) 2013 Only For Testing BBC No Leech & CodeCube SRL
 * @license     http://www.phpprobid.com/license Commercial License
 * 
 * @version     7.0
 */
/**
 * sales listings table service class
 * creates/edits/removes sales listings
 * when editing or removing a sale listing, check if the corresponding row in the 
 * sales table is empty, and if it is, remove that as well
 */

namespace Ppb\Service;

use Ppb\Db\Table,
    Cube\Db\Expr;

class SalesListings extends AbstractService
{

    /**
     *
     * sales table service class
     * 
     * @var \Ppb\Service\Sales
     */
    protected $_sales = null;

    /**
     *
     * listings table service class
     * 
     * @var \Ppb\Service\Listings
     */
    protected $_listings = null;

    /**
     *
     * reputation table service class
     * 
     * @var \Ppb\Service\Reputation
     */
    protected $_reputation = null;

    /**
     * 
     * class constructor
     */
    public function __construct()
    {
        parent::__construct();

        $this->setTable(
                new Table\SalesListings());
    }

    /**
     * 
     * get sales table service class
     * 
     * @return \Ppb\Service\Sales
     */
    public function getSales()
    {
        if (!$this->_sales instanceof Sales) {
            $this->setSales();
        }

        return $this->_sales;
    }

    /**
     * 
     * set sales table service class
     * 
     * @param \Ppb\Service\Sales $sales
     * @return \Ppb\Service\SalesListings
     */
    public function setSales(Sales $sales = null)
    {
        if (!$sales instanceof Sales) {
            $sales = new Sales();
        }

        $this->_sales = $sales;

        return $this;
    }

    /**
     * 
     * get listings table service class
     * 
     * @return \Ppb\Service\Listings
     */
    public function getListings()
    {
        if (!$this->_listings instanceof Listings) {
            $this->setListings();
        }

        return $this->_listings;
    }

    /**
     * 
     * set listings table service class
     * 
     * @param \Ppb\Service\Listings $listings
     * @return \Ppb\Service\SalesListings
     */
    public function setListings(Listings $listings = null)
    {
        if (!$listings instanceof Listings) {
            $listings = new Listings();
        }

        $this->_listings = $listings;

        return $this;
    }

    /**
     * 
     * get reputation table service class
     * 
     * @return \Ppb\Service\Reputation
     */
    public function getReputation()
    {
        if (!$this->_reputation instanceof Reputation) {
            $this->setReputation();
        }

        return $this->_reputation;
    }

    /**
     * 
     * set reputation table service class
     * 
     * @param \Ppb\Service\Reputation $reputation
     * @return \Ppb\Service\SalesListings
     */
    public function setReputation(Reputation $reputation = null)
    {
        if (!$reputation instanceof Reputation) {
            $reputation = new Reputation();
        }

        $this->_reputation = $reputation;

        return $this;
    }

    /**
     * 
     * create or edit a sale listing.
     * - when editing, if changing "sale_id", check if there are other listings 
     * correspoding to that "sale_id", and if not, remove the row in the "sales" table 
     * - when editing a sale listing, the quantity cannot be altered
     * - when creating a sale listing, update the quantity of the listing, and if it is 0, 
     * then close the listing
     * - when creating a sale listing, also create two rows in the reputation table, 
     * for the reputation that is to be left
     * 
     * TODO: review the empty sale deletion process
     * TODO: sale_id and listing_id might not be needed when editing a sale listing 
     * 
     * @param array $data
     * @return \Ppb\Service\SalesListings
     * @throws \InvalidArgumentException
     */
    public function save($data)
    {
        $row = null;
        $listings = array();

        $data = $this->_prepareSaveData($data);

        if (empty($data['sale_id']) || empty($data['listing_id'])) {
            throw new \InvalidArgumentException("A sale id and a listing id are required 
                when adding/editing a row from the sales listings table.");
        }
        if (array_key_exists('id', $data)) {
            $select = $this->_table->select()
                    ->where("id = ?", $data['id']);

            unset($data['id']);

            $row = $this->_table->fetchRow($select);
        }

        if (count($row) > 0) {
            unset($data['quantity']);
            $this->_table->update($data, "id='{$row['id']}'");

            if ($row['sale_id'] != $data['sale_id']) {

                $listings = $this->_table->fetchAll(
                        $this->_table->select('id')
                                ->where('sale_id = ?', $row['sale_id']));

                if (!count($listings)) {
                    $this->getSales()->delete($row['sale_id']);
                }
            }
        }
        else {
            $data['created_at'] = new Expr('now()');
            $this->_table->insert($data);
            $saleListingId = $this->_table->lastInsertId();

            $listings = $this->getListings();

            $listing = $listings->findBy('id', $data['listing_id']);

            $quantity = $listing['quantity'] - $data['quantity'];

            $params = array(
                'id' => $listing['id'],
                'quantity' => $quantity,
                'partial' => true,
            );

            $listings->setUser(null)
                    ->save($params);

            if ($quantity == 0) {
                $listings->close($listing);
            }

            $reputation = $this->getReputation();
            $sale = $this->getSales()->findBy('id', $data['sale_id']);

            // prepare reputation row: seller => buyer
            $reputation->save(array(
                'user_id' => $sale['buyer_id'],
                'poster_id' => $sale['seller_id'],
                'sale_listing_id' => $saleListingId,
                'reputation_type' => Reputation::PURCHASE,
            ));

            // prepare reputation row: buyer => seller
            $reputation->save(array(
                'user_id' => $sale['seller_id'],
                'poster_id' => $sale['buyer_id'],
                'sale_listing_id' => $saleListingId,
                'reputation_type' => Reputation::SALE,
            ));
        }

        return $this;
    }

}

