<?php

/**
 *
 * PHP Pro Bid $Id$ nZQjjp+X92JH2PcKl5SX3QN7SQ3ANqULW3QtGgo8080=
 *
 * @link        http://www.phpprobid.com
 * @copyright   Copyright (c) 2014 Only For Testing BBC No Leech & CodeCube SRL
 * @license     http://www.phpprobid.com/license Commercial License
 *
 * @version     7.3
 */
/**
 * sale transaction fee class
 */

namespace Ppb\Service\Fees;

use Ppb\Service,
    Cube\Controller\Front,
    Ppb\Db\Table\Row\Sale as SaleModel;

class SaleTransaction extends Service\Fees
{

    /**
     *
     * fees to be included
     *
     * @var array
     */
    protected $_fees = array(
        self::SALE => 'Sale Transaction Fee',
    );

    /**
     *
     * sale object
     *
     * @var \Ppb\Db\Table\Row\Sale
     */
    protected $_sale;

    /**
     *
     * total amount to be paid after the calculate method is called
     *
     * @var float
     */
    protected $_totalAmount;

    /**
     *
     * class constructor
     *
     * @param \Ppb\Db\Table\Row\Sale                $sale
     * @param integer|string|\Ppb\Db\Table\Row\User $user the user that will be paying
     */
    public function __construct(SaleModel $sale = null, $user = null)
    {
        parent::__construct();

        if ($sale !== null) {
            $this->setSale($sale);
        }

        if ($user !== null) {
            $this->setUser($user);
        }
    }

    /**
     *
     * set sale model
     * also, based on the sale model, set the total amount that will be used to calculate the fees against
     * the amount will be the total amount without tax, postage and insurance
     *
     * @param \Ppb\Db\Table\Row\Sale $sale
     *
     * @return $this
     */
    public function setSale(SaleModel $sale)
    {
        $this->_sale = $sale;

        $this->setAmount(
            $sale->calculateTotal(true));

        return $this;
    }

    /**
     *
     * get sale model
     *
     * @return \Ppb\Db\Table\Row\Sale
     */
    public function getSale()
    {
        return $this->_sale;
    }

    /**
     *
     * calculate and return an array containing all fees to be applied for the sale in question
     *
     * @return array
     */
    public function calculate()
    {
        $data = array();
        $this->_totalAmount = 0;

        $settings = $this->getSettings();

        foreach ($this->_fees as $key => $value) {
            $feeAmount = $this->getFeeAmount($key);

            if ($feeAmount > 0 || $settings['display_free_fees']) {
                $data[] = array(
                    'key'      => $key,
                    'name'     => array(
                        'string' => 'Sale Transaction Fee - Sale ID: #%s',
                        'args'   => array($this->_sale['id']),
                    ),
                    'amount'   => $feeAmount,
                    'tax_rate' => $this->getTaxType()->getData('amount'),
                    'currency' => $settings['currency'],
                );
            }

            $this->_totalAmount += $feeAmount;
        }

        return $data;
    }

    /**
     *
     * activate the affected sale
     * the callback will also process the listing in case a payment has been reversed etc
     *
     * @param bool  $ipn  true if payment is completed, false otherwise
     * @param array $post array keys: {sale_id}
     *
     * @return \Ppb\Service\Fees\SaleTransaction
     */
    public function callback($ipn, array $post)
    {
        $salesService = new Service\Sales();
        $sale = $salesService->findBy('id', $post['sale_id']);

        $flag = ($ipn) ? 1 : 0;
        $sale->save(array(
            'active' => $flag,
        ));

        if ($ipn) {
            $user = Front::getInstance()->getBootstrap()->getResource('user');

            $type = null;
            if (!empty($user['id'])) {
                $type = ($sale['buyer_id'] == $user['id']) ? 'bought' : 'sold';
            }

            $this->setRedirect(array(
                'module'     => 'members',
                'controller' => 'invoices',
                'action'     => 'browse',
                'params'     => array(
                    'type'    => $type,
                    'sale_id' => $post['sale_id'],
                ),
            ));
        }

        return $this;
    }

    /**
     *
     * only apply preferred sellers reduction if it should apply to sale fees
     *
     * @param float $amount
     *
     * @return float
     */
    protected function _applyPreferredSellersReduction($amount)
    {
        $settings = $this->getSettings();

        if ($settings['preferred_sellers_apply_sale']) {
            $amount = parent::_applyPreferredSellersReduction($amount);
        }

        return $amount;
    }

    /**
     *
     * get total amount to be paid resulted from the calculate() method
     *
     * @return float
     */
    public function getTotalAmount()
    {
        return $this->_totalAmount;
    }

}

