<?php

/**
 *
 * PHP Pro Bid $Id$ hybLqHSgDLi0Z9pYc0E4PuOK4rqQSCemO0dC0O2185c=
 *
 * @link        http://www.phpprobid.com
 * @copyright   Copyright (c) 2014 Only For Testing BBC No Leech & CodeCube SRL
 * @license     http://www.phpprobid.com/license Commercial License
 *
 * @version     7.0
 */
/**
 * offers table row object model
 */

namespace Ppb\Db\Table\Row;

use Ppb\Service;

class Offer extends AbstractRow
{
    /**
     * offer statuses
     */

    const STATUS_PENDING = 'pending';
    const STATUS_ACCEPTED = 'accepted';
    const STATUS_DECLINED = 'declined';
    const STATUS_WITHDRAWN = 'withdrawn';

    /**
     *
     * allowed offer statuses
     *
     * @var array
     */
    public static $statuses = array(
        self::STATUS_PENDING   => 'Pending',
        self::STATUS_ACCEPTED  => 'Accepted',
        self::STATUS_DECLINED  => 'Declined',
        self::STATUS_WITHDRAWN => 'Withdrawn'
    );

    /**
     *
     * accept the offer and create a new sale
     * the listing is selected with the for update clause so that
     * no other transactions can update the listing while
     * this action is in progress
     *
     * @return bool
     */
    public function accept()
    {
        /** @var \Ppb\Db\Table\Row\Listing $listing */
        $listing = $this->findParentRow('\Ppb\Db\Table\Listings', null, $this->getTable()->select()->forUpdate());

        $quantity = $this->getData('quantity');
        if ($listing->isOwner() && $listing->canAcceptOffer($quantity)) {
            $this->save(array(
                'status' => self::STATUS_ACCEPTED,
            ));

            //MAIL OFFER ACCEPTED BUYER NOTIFICATION
            $mail = new \Listings\Model\Mail\BuyerNotification();
            $mail->offerAccepted($listing, $this)->send();

            $service = new Service\Sales();
            $data = array(
                'buyer_id'  => $this->getData('user_id'),
                'seller_id' => $listing['user_id'],
                'listings'  => array(
                    array(
                        'listing_id' => $listing['id'],
                        'price'      => $this->getData('amount'),
                        'quantity'   => (int)$quantity,
                    ),
                ),
            );
            $service->save($data);

            return true;
        }

        return false;
    }

    /**
     *
     * decline an offer (only the listing owner can do this)
     *
     * @return bool
     */
    public function decline()
    {
        /** @var \Ppb\Db\Table\Row\Listing $listing */
        $listing = $this->findParentRow('\Ppb\Db\Table\Listings');

        if ($listing->isOwner() && $this->getData('status') == self::STATUS_PENDING) {
            $this->save(array(
                'status' => self::STATUS_DECLINED,
            ));

            //MAIL OFFER DECLINED BUYER NOTIFICATION
            $mail = new \Listings\Model\Mail\BuyerNotification();
            $mail->offerDeclined($listing, $this)->send();

            return true;
        }

        return false;
    }

    /**
     *
     * check if the logged in user can withdraw the offer
     * (only the poster can withdraw it)
     *
     * @return bool
     */
    public function canWithdraw()
    {
        $user = $this->getUser();

        return ($this->getData('user_id') == $user['id'] &&
            $this->getData('status') == self::STATUS_PENDING) ? true : false;
    }

    /**
     *
     * withdraw an offer
     * (only the poster can do this)
     *
     * @return bool
     */
    public function withdraw()
    {
        if ($this->canWithdraw()) {
            /** @var \Ppb\Db\Table\Row\Listing $listing */
            $listing = $this->findParentRow('\Ppb\Db\Table\Listings');

            $this->save(array(
                'status' => self::STATUS_WITHDRAWN,
            ));

            //MAIL OFFER WITHDRAWN SELLER NOTIFICATION
            $mail = new \Listings\Model\Mail\OwnerNotification();
            $mail->offerWithdrawn($listing, $this)->send();

            return true;
        }

        return false;
    }

}