/* globals wpforms_offline_forms,jQuery */

;(function($) {

	/**
	 * Main logic holder for Offline Forms addon.
	 *
	 * @since 1.0.0
	 */
	var WPFormsOfflineForms = {

		isOffline: false,
		forms: [],
		formsData: {},

		/**
		 * Manage data in LocalStorage. With namespaces, expiration and proper nesting.
		 *
		 * @since 1.0.0
		 */
		storage: {

			local: store.namespace( 'wpforms' ).namespace( 'offline' ),

			// Number of seconds each storage record will be valid for.
			expire: 60 * 60 * 24,

			/**
			 * Save the values into the storage under form namespace.
			 *
			 * @param {Number} form_id
			 * @param {Array} form_data
			 */
			set: function ( form_id, form_data ) {

				var form = this.local.namespace( parseInt( form_id, 10 ) );

				// Doing +1 to always have a unique key for entry and
				// to NOT iterate/delete old/save new data, as this is an expensive & DOM-blocking operation.
				form.set( form.size() + 1, form_data, this.expire );
			},

			/**
			 * Get and remove from storage the single the oldest offline form entry.
			 *
			 * @param {Number} form_id
			 *
			 * @returns {Object} Single entry having {entry_id:1,entry:"{}"} structure.
			 */
			get: function ( form_id ) {

				form_id = parseInt( form_id, 10 );
				var entries = this.getAll( form_id ),
					first = {};

				// Now we need to take only the first entry and remove it from the saved.
				if ( entries.length > 0 ) {
					first = entries.shift();
				}

				// Remove the requested value from storage.
				this.local.namespace( form_id )
					.remove( first.entry_id );

				return first;
			},

			/**
			 * Get all the offline form entries.
			 *
			 * @param {Number} form_id
			 *
			 * @returns {Array} All the entries, each entry having {entry_id:1,entry:"{}"} structure.
			 */
			getAll: function ( form_id ) {

				var data = this.local.namespace( parseInt( form_id, 10 ) ).getAll(),
					entries = [],
					entry_id;

				// Remap data object to an array to have handy Array.prototype.* functions available.
				for ( entry_id in data ) {
					if ( ! data.hasOwnProperty( entry_id ) || $.isEmptyObject( data[ entry_id ] ) ) {
						continue;
					}
					entries.push( {
						entry_id: entry_id,
						entry: data[ entry_id ]
					} );
				}

				return entries;
			},

			/**
			 * Remove all offline entries from a storage for a given form.
			 *
			 * @param {Number} form_id
			 */
			clearAll: function ( form_id ) {

				this.local.namespace( parseInt( form_id, 10 ) ).clearAll();
			}
		},

		/**
		 * Manage offline forms notifications.
		 *
		 * @since 1.0.0
		 */
		notifications: {

			$_form: false,
			_actions: [],

			/**
			 * Set a form that notifications will be applied to.
			 *
			 * @param $form jQuery object of a form on a page with offline functionality.
			 *
			 * @returns {WPFormsOfflineForms.notifications}
			 */
			setForm: function ( $form ) {

				this.$_form = $form;

				// Always clear actions when form is defined or changed.
				this._clearActions();

				return this;
			},

			/**
			 * Set actions that might be applied to a notification.
			 *
			 * @param {Object} actions Object of actions, key:value storage. If false - clears all actions.
			 *
			 * @returns {WPFormsOfflineForms.notifications}
			 */
			setActions: function ( actions ) {
				if ( ! actions ) {
					this._clearActions();

					return this;
				}

				var action;

				for ( action in actions ) {
					if ( ! actions.hasOwnProperty( action ) ) {
						continue;
					}
					this._actions.push( {
						key: action,
						label: actions[ action ]
					} );
				}

				return this;
			},

			/**
			 * Set the actions to a default empty state.
			 *
			 * @private
			 */
			_clearActions: function () {
				this._actions = [];
			},

			/**
			 * Get a template for the whole notification.
			 *
			 * @returns {*|HTMLElement}
			 *
			 * @private
			 */
			_getNotificationTemplate: function () {
				return $( '<div class="wpforms-notice wpforms-info"></div>' );
			},

			/**
			 * Get a template for the actions block in a notification.
			 *
			 * @returns {*|HTMLElement}
			 *
			 * @private
			 */
			_getActionsTemplate: function () {
				return $( '<div class="wpforms-notice-actions"></div>' );
			},

			/**
			 * Get a template for the single action in actions block.
			 *
			 * @returns {*|HTMLElement}
			 *
			 * @private
			 */
			_getActionTemplate: function () {
				return $( '<a class="wpforms-notice-action"></a>' );
			},

			/**
			 * Display various notifications.
			 */
			display: function ( message, isDismissible ) {

				var class_dismiss = 'non-dismissible';
				if ( isDismissible ) {
					class_dismiss = '';
				}

				// Populate notifications holder with text.
				var $notification = this._getNotificationTemplate()
										.addClass( class_dismiss )
										.html( message );

				// Populate actions holder with links to actions.
				if ( this._actions.length ) {
					var $actions = this._getActionsTemplate(),
						action;

					for ( action in this._actions ) {
						// Add a link to the end of actions holder.
						$actions.append(
							this._getActionTemplate()
								.attr( 'href', '#' + encodeURIComponent( this._actions[ action ].key ) )
								.text( this._actions[ action ].label )
						);
					}

					// Add actions holder to the end of notification holder.
					$notification.append( $actions );
				}

				// Finally, display a notification.
				$( '.wpforms-field-container', this.$_form ).before( $notification );

				var $form = this.$_form;

				// Scroll to form top, so that user will be able to see the message.
				$( 'html, body' ).animate( {
					scrollTop: $form.offset().top - 100
				}, 750 );
			},

			/**
			 * Hide all non-dismissible notifications.
			 */
			hide: function () {
				$( '.wpforms-notice', this.$_form ).not( '.non-dismissible' ).remove();
			},

			/**
			 * Hide all notifications for a form.
			 */
			hideAll: function () {
				$( '.wpforms-notice', this.$_form ).remove();
			}
		},

		/**
		 * Start the engine.
		 *
		 * @since 1.0.0
		 */
		init: function () {

			WPFormsOfflineForms.populateForms();

			// Listen for future changes in connection status.
			window.addEventListener( 'online', WPFormsOfflineForms.connectionEvent );
			window.addEventListener( 'offline', WPFormsOfflineForms.connectionEvent );

			$( document ).on( 'wpformsReady', WPFormsOfflineForms.checkConnectivityLoop );
			$( document ).on( 'wpformsReady', WPFormsOfflineForms.checkOfflineRecords );
			$( document ).on( 'wpformsReady', WPFormsOfflineForms.saveFormDefaults );

			WPFormsOfflineForms.bindUIActions();
		},

		/**
		 * Get the array of forms jQuery objects that have offline functionality.
		 *
		 * @since 1.0.0
		 */
		populateForms: function () {

			$.each( wpforms_offline_forms.offline_form_ids, function ( index, form_id ) {
				WPFormsOfflineForms.forms.push( $( '#wpforms-' + form_id ) );
			} );
		},

		/**
		 * Check Internet connection in a loop by accessing a static file.
		 *
		 * @since 1.0.0
		 */
		checkConnectivityLoop: function () {

			var interval = parseInt( wpforms_offline_forms.check_connection_interval, 10 );

			if ( ! WPFormsOfflineForms._isNaturalNumber( interval ) ) {
				return;
			}

			setInterval( function () {
				WPFormsOfflineForms.processConnectivityAjaxCheck( null );
			}, interval * 1000 );
		},

		/**
		 * Send an ajax request to a static file to check the Internet connection.
		 *
		 * @since 1.0.0
		 *
		 * @param {function} callback To be called after all the checks.
		 */
		processConnectivityAjaxCheck: function ( callback ) {

			var timeout = parseInt( wpforms_offline_forms.check_connection_timeout, 10 );

			if ( ! WPFormsOfflineForms._isNaturalNumber( timeout ) ) {
				return;
			}

			$.ajax( {
				method: 'HEAD',
				url: wpforms_offline_forms.check_connection_url,
				cache: false,
				timeout: timeout * 1000,
				complete: function ( jqXHR, textStatus ) {
					if ( jqXHR.status !== 0 && textStatus !== 'timeout' ) {
						WPFormsOfflineForms.isOffline = false;
						navigator.onLine = true;

						WPFormsOfflineForms.notificationOfflineHide();
					}
					else {
						WPFormsOfflineForms.isOffline = true;
						navigator.onLine = false;

						WPFormsOfflineForms.notificationOfflineDisplay();
					}

					if ( typeof callback === 'function' ) {
						callback();
					}
				}
			} );
		},

		/**
		 * On form load check whether we have entries for it in our storage,
		 * that are not yet submitted. Display to a user a notification with choices.
		 *
		 * @since 1.0.0
		 */
		checkOfflineRecords: function () {

			$.each( WPFormsOfflineForms.forms, function ( index, form ) {
				var entries = WPFormsOfflineForms.storage.getAll( form.find( 'form' ).data( 'formid' ) ),
					notifications = WPFormsOfflineForms.notifications.setForm( form );

				notifications.hideAll();

				if ( entries.length === 0 ) {
					return;
				}

				if ( entries.length === 1 ) {
					if ( ! WPFormsOfflineForms.isOffline ) {
						notifications.setActions( {
							offline_restore: wpforms_offline_forms.text_restore_btn,
							offline_clear: wpforms_offline_forms.text_clear_btn
						} );
					}

					notifications.display( wpforms_offline_forms.text_restore_single );
				}
				else {
					if ( ! WPFormsOfflineForms.isOffline ) {
						notifications.setActions( {
							offline_restore: wpforms_offline_forms.text_restore_btn,
							offline_clear: wpforms_offline_forms.text_clear_all_btn
						} );
					}

					notifications.display( wpforms_offline_forms.text_restore_plural );
				}
			} );
		},

		/**
		 * When form has finished loading,
		 * retrieve and save in memory all form default values.
		 * We will reuse this data to
		 *
		 * @since 1.0.0
		 */
		saveFormDefaults: function () {

			$.each(wpforms_offline_forms.offline_form_ids, function(index, form_id){
				WPFormsOfflineForms.formsData[form_id] = $( '#wpforms-' + form_id).find( 'form' ).serializeArray();
			});
		},

		/**
		 * When form is submitted we need to clear it.
		 * We are actually restoring to its predefined state.
		 *
		 * @since 1.0.0
		 *
		 * @param {Number} form_id
		 */
		processRestoreFormDefault: function ( form_id ) {

			var $form_holder = $( '#wpforms-' + form_id );

			if (
				typeof WPFormsOfflineForms.formsData[ form_id ] === 'undefined' ||
				! $form_holder.length
			) {
				return;
			}

			$form_holder.find( 'form' ).deserialize( WPFormsOfflineForms.formsData[ form_id ], {
				change: function () {
					$( this ).trigger( 'change' );
				}
			} );

			/*
			 * In case we have a multi-page form we need to post-process it.
			 */
			if ( $form_holder.find( '.wpforms-page-indicator' ).length ) {
				// Find a prev button to trigger navigation.
				if ( $form_holder.find( '.wpforms-page-prev[data-page="2"]' ).length ) {
					$form_holder.find( '.wpforms-page' ).hide();
					$form_holder.find( '.wpforms-page-prev[data-page="2"]' ).trigger( 'click' );
				}
				else {
					// Create a button that will be used to fake clicks.
					var $button = $( '<button class="wpforms-page-button wpforms-page-prev"></button>' );
					$button
						.css( 'visibility', 'hidden' )
						.data( 'action', 'prev' )
						.data( 'page', '2' )
						.data( 'formid', form_id );

					$form_holder.find( '.wpforms-page' ).hide();
					$form_holder.find( '.wpforms-field-pagebreak' ).first().append( $button );
					$button.trigger( 'click' );
					$button.remove();
				}
			}

			this.checkOfflineRecords();
		},

		/**
		 * Bind actions for each individual relevant forms.
		 *
		 * @since 1.0.0
		 */
		bindUIActions: function () {

			var _ = this;

			/*
			 * Process the form submission.
			 */
			$.each( WPFormsOfflineForms.forms, function ( index, form ) {
				_.processFormSave( form.find( 'form' ) );
			} );

			/*
			 * Process the form Submit button click.
			 * Listen event for offline-capable forms only.
			 */
			$.each( WPFormsOfflineForms.forms, function ( index, form ) {

				// Always hard-check internet status. Submit afterwards to have status already set up.
				$( document ).on( 'click', '#' + form.attr( 'id' ) + ' .wpforms-submit-container button[type="submit"]', function ( e ) {
					e.preventDefault();
					e.stopPropagation();
					e.stopImmediatePropagation();

					var $submit = $( this ),
						altText = $submit.data( 'alt-text' ),
						$form = $( this ).closest( 'form' ),
						button_text = $submit.text();

					$submit.prop( 'disabled', true );

					if ( altText ) {
						$submit.text( altText );
					}

					_.processConnectivityAjaxCheck( function () {
						$form.submit();
						$submit.prop( 'disabled', false );
						$submit.text( button_text );
					} );
				} );
			} );

			/*
			 * Process the restore functionality for a form.
			 */
			$( document ).on( 'click', '.wpforms-notice-action[href="#offline_restore"]', function ( e ) {
				e.preventDefault();

				_.processRestore( $( this ).closest( 'form' ) );
			} );

			/*
			 * Process the clear functionality for a form.
			 */
			$( document ).on( 'click', '.wpforms-notice-action[href="#offline_clear"]', function ( e ) {
				e.preventDefault();

				_.processClearAll( $( this ).closest( 'form' ) );
			} );
		},

		/**
		 * Applicable only if we are offline.
		 * Get all the fields values of the form, except ignored for security.
		 * Save data in the storage.
		 *
		 * @since 1.0.0
		 *
		 * @param $form jQuery object of the form.
		 */
		processFormSave: function ( $form ) {

			$form.submit( function ( e ) {
				var form_id = parseInt( $form.data( 'formid' ), 10 );

				// If online - proceed as usual.
				if ( ! WPFormsOfflineForms.isOffline ) {
					return true;
				}

				// Just do not submit, like ever.
				e.preventDefault();
				e.stopPropagation();
				e.stopImmediatePropagation();

				// Disable certain fields, so they won't be serialized.
				$( '.wpforms-field', $form ).each( function ( index, field ) {
					if (
						$( field ).hasClass( 'wpforms-field-password' ) ||
						$( field ).hasClass( 'wpforms-field-credit-card' ) ||
						$( field ).hasClass( 'wpforms-field-signature' )
					) {
						$( field ).find( 'input, select' ).attr( 'disabled', 'disabled' );
					}
				} );

				// Serialize all the fields, disabled are ignored.
				var form_data = $form.serializeArray();

				// Enable fields back, so user can continue submission.
				$( '.wpforms-field', $form ).each( function ( index, field ) {
					if (
						$( field ).hasClass( 'wpforms-field-password' ) ||
						$( field ).hasClass( 'wpforms-field-credit-card' ) ||
						$( field ).hasClass( 'wpforms-field-signature' )
					) {
						$( field ).find( 'input, select' ).removeAttr( 'disabled' );
					}
				} );

				// Save to the storage.
				WPFormsOfflineForms.storage.set( form_id, form_data );

				// Restore default state of the form.
				WPFormsOfflineForms.processRestoreFormDefault( form_id );
			} );
		},

		/**
		 * Clear all offline records from storage for this form and hide all notifications.
		 *
		 * @since 1.0.0
		 *
		 * @param $form jQuery object of a form.
		 */
		processRestore: function ( $form ) {

			var entry_data = this.storage.get( $form.data( 'formid' ) );

			$form.deserialize( entry_data.entry, {
				change: function () {
					$( this ).trigger( 'change' );
				}
			} );

			this.checkOfflineRecords();
		},

		/**
		 * Clear all offline records from storage for this form and hide all notifications.
		 *
		 * @since 1.0.0
		 *
		 * @param $form jQuery object of a form.
		 */
		processClearAll: function ( $form ) {

			// Clear all offline records, whether it's 1 or 100500.
			this.storage.clearAll( $form.data( 'formid' ) );

			// Hide all notifications.
			this.notifications.setForm( $form ).hideAll();
		},

		/**
		 * What to do when user connectivity state is changed to online or offline.
		 *
		 * @since 1.0.0
		 */
		connectionEvent: function () {

			WPFormsOfflineForms.connectionUpdateStatus();
		},

		/**
		 * Check if we're online, set a class on <body> if not and hide/display a notification to a user.
		 *
		 * @since 1.0.0
		 */
		connectionUpdateStatus: function () {
			if ( typeof navigator.onLine === 'undefined' ) {
				return;
			}

			WPFormsOfflineForms.isOffline = ! navigator.onLine;

			$( 'body' ).toggleClass( 'wpforms-is-offline', WPFormsOfflineForms.isOffline );

			WPFormsOfflineForms.isOffline ? WPFormsOfflineForms.notificationOfflineDisplay() : WPFormsOfflineForms.notificationOfflineHide();
		},

		/**
		 * Display a notification about offline status of a client.
		 * Displayed only for forms with enabled offline functionality.
		 *
		 * @since 1.0.0
		 */
		notificationOfflineDisplay: function () {

			$.each( WPFormsOfflineForms.forms, function ( index, form ) {
				var notifications = WPFormsOfflineForms.notifications.setForm( form );

				notifications.hideAll();
				notifications.display( wpforms_offline_forms.text_offline, true );
			} );
		},

		/**
		 * Hide all offline messages for all forms.
		 *
		 * @since 1.0.0
		 */
		notificationOfflineHide: function ( $form ) {

			this.notifications.setForm( $form ).hide();

			// As we are back online - check whether we have something already saved locally.
			this.checkOfflineRecords();
		},

		/**
		 * Check whether the number is natural.
		 *
		 * @param {Number} number
		 *
		 * @returns {boolean}
		 * @private
		 */
		_isNaturalNumber: function ( number ) {
			return number >= 0 && Math.floor( number ) === + number;
		}
	};

	// Here goes the magic. http://gph.is/1KjihQe
	WPFormsOfflineForms.init();
})(jQuery);
