<?php
/**
 * WP Courseware Quiz Model.
 *
 * @package WPCW
 * @subpackage Models
 * @since 4.2.0
 */
namespace WPCW\Models;

use WPCW\Database\DB_Quizzes;

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Class Quiz.
 *
 * @since 4.2.0
 *
 * @property int $quiz_id
 * @property string $quiz_title
 * @property string $quiz_desc
 * @property int $quiz_author
 * @property int $parent_unit_id
 * @property int $parent_course_id
 * @property string $quiz_type
 * @property int $quiz_pass_mark
 * @property string $quiz_show_answers
 * @property string $quiz_show_survey_responses
 * @property int $quiz_attempts_allowed
 * @property string $show_answers_settings
 * @property string $quiz_paginate_questions
 * @property string $quiz_paginate_questions_settings
 * @property string $quiz_timer_mode
 * @property int $quiz_timer_mode_limit
 * @property string $quiz_results_downloadable
 * @property string $quiz_results_by_tag
 * @property string $quiz_results_by_timer
 * @property string $quiz_recommended_score
 * @property int $show_recommended_percentage
 */
class Quiz extends Model {

	/**
	 * @var DB_Quizzes The quizzes database.
	 * @since 4.3.0
	 */
	protected $db;

	/**
	 * @var int Quiz Id.
	 * @since 4.2.0
	 */
	protected $quiz_id;

	/**
	 * @var string Quiz Title.
	 * @since 4.2.0
	 */
	protected $quiz_title = '';

	/**
	 * @var string Quiz Description.
	 * @since 4.2.0
	 */
	protected $quiz_desc = '';

	/**
	 * @var int Quiz Author.
	 * @since 4.2.0
	 */
	protected $quiz_author = 0;

	/**
	 * @var int Quiz Unit Id.
	 * @since 4.2.0
	 */
	protected $parent_unit_id = 0;

	/**
	 * @var int Quiz Course Id.
	 * @since 4.2.0
	 */
	protected $parent_course_id = 0;

	/**
	 * @var string Quiz Type.
	 * @since 4.2.0
	 */
	protected $quiz_type = '';

	/**
	 * @var int Quiz Pass Mark.
	 * @since 4.2.0
	 */
	protected $quiz_pass_mark = 0;

	/**
	 * @var string Quiz show answers.
	 * @since 4.2.0
	 */
	protected $quiz_show_answers = 'no_answers';

	/**
	 * @var string Quiz show survey responses.
	 * @since 4.2.0
	 */
	protected $quiz_show_survey_responses = 'no_responses';

	/**
	 * @var int Quiz attempts allowed.
	 * @since 4.2.0
	 */
	protected $quiz_attempts_allowed = -1;

	/**
	 * @var string Quiz show answer settings.
	 * @since 4.2.0
	 */
	protected $show_answers_settings = '';

	/**
	 * @var string Quiz paginate quesitons.
	 * @since 4.2.0
	 */
	protected $quiz_paginate_questions = 'no_paging';

	/**
	 * @var string Quiz paginate questions settings.
	 * @since 4.2.0
	 */
	protected $quiz_paginate_questions_settings = '';

	/**
	 * @var string Quiz timer mode.
	 * @since 4.2.0
	 */
	protected $quiz_timer_mode = 'no_timer';

	/**
	 * @var int Quiz timer mode limit.
	 * @since 4.2.0
	 */
	protected $quiz_timer_mode_limit = 15;

	/**
	 * @var string Quiz results downloadable.
	 * @since 4.2.0
	 */
	protected $quiz_results_downloadable = 'on';

	/**
	 * @var string Quiz results by tag.
	 * @since 4.2.0
	 */
	protected $quiz_results_by_tag = 'on';

	/**
	 * @var string Quiz results by timer.
	 * @since 4.2.0
	 */
	protected $quiz_results_by_timer = 'on';

	/**
	 * @var string Quiz recommended score.
	 * @since 4.2.0
	 */
	protected $quiz_recommended_score = 'no_recommended';

	/**
	 * @var int Quiz recommended percentage.
	 * @since 4.2.0
	 */
	protected $show_recommended_percentage = 50;

	/**
	 * Quiz Constructor.
	 *
	 * @since 4.3.0
	 *
	 * @param array|int|Model $data The model data.
	 */
	public function __construct( $data = array() ) {
		$this->db = new DB_Quizzes();
		parent::__construct( $data );
	}

	/**
	 * Get Quiz Id.
	 *
	 * @since 4.3.0
	 *
	 * @return int|void
	 */
	public function get_id() {
		return absint( $this->get_quiz_id() );
	}

	/**
	 * Get quiz id.
	 *
	 * @since 4.2.0
	 *
	 * @return int
	 */
	public function get_quiz_id() {
		return absint( $this->quiz_id );
	}

	/**
	 * Get quiz title.
	 *
	 * @since 4.2.0
	 *
	 * @return string
	 */
	public function get_quiz_title() {
		return $this->quiz_title;
	}

	/**
	 * Get quiz description.
	 *
	 * @since 4.2.0
	 *
	 * @return string
	 */
	public function get_quiz_desc() {
		return wp_kses_post( $this->quiz_desc );
	}

	/**
	 * Get quiz author.
	 *
	 * @since 4.2.0
	 *
	 * @return int
	 */
	public function get_quiz_author() {
		return absint( $this->quiz_author );
	}

	/**
	 * Get parent unit id.
	 *
	 * @since 4.2.0
	 *
	 * @return int
	 */
	public function get_parent_unit_id() {
		return absint( $this->parent_unit_id );
	}

	/**
	 * Get parent course id.
	 *
	 * @since 4.2.0
	 *
	 * @return int
	 */
	public function get_parent_course_id() {
		return absint( $this->parent_course_id );
	}

	/**
	 * Get quiz type.
	 *
	 * @since 4.2.0
	 *
	 * @return string
	 */
	public function get_quiz_type() {
		return $this->quiz_type;
	}

	/**
	 * Get quiz pass mark.
	 *
	 * @since 4.2.0
	 *
	 * @return int
	 */
	public function get_quiz_pass_mark() {
		return $this->quiz_pass_mark;
	}

	/**
	 * Get quiz show answers.
	 *
	 * @since 4.2.0
	 *
	 * @return string
	 */
	public function get_quiz_show_answers() {
		return $this->quiz_show_answers;
	}

	/**
	 * Get quiz show survey responses.
	 *
	 * @since 4.2.0
	 *
	 * @return string
	 */
	public function get_quiz_show_survey_responses() {
		return $this->quiz_show_survey_responses;
	}

	/**
	 * Get quiz attempts allowed.
	 *
	 * @since 4.2.0
	 *
	 * @return int
	 */
	public function get_quiz_attempts_allowed() {
		return $this->quiz_attempts_allowed;
	}

	/**
	 * Get show answers settings.
	 *
	 * @since 4.2.0
	 *
	 * @return string
	 */
	public function get_show_answers_settings() {
		return $this->show_answers_settings;
	}

	/**
	 * Get quiz paginate quesitons.
	 *
	 * @since 4.2.0
	 *
	 * @return string
	 */
	public function get_quiz_paginate_questions() {
		return $this->quiz_paginate_questions;
	}

	/**
	 * Get quiz paginate settings.
	 *
	 * @since 4.2.0
	 *
	 * @return string
	 */
	public function get_quiz_paginate_questions_settings() {
		return $this->quiz_paginate_questions_settings;
	}

	/**
	 * Get quiz timer mode.
	 *
	 * @since 4.2.0
	 *
	 * @return string
	 */
	public function get_quiz_timer_mode() {
		return $this->quiz_timer_mode;
	}

	/**
	 * Get quiz timer mode limit.
	 *
	 * @since 4.2.0
	 *
	 * @return int
	 */
	public function get_quiz_timer_mode_limit() {
		return $this->quiz_timer_mode_limit;
	}

	/**
	 * Get quiz results downloadable.
	 *
	 * @since 4.2.0
	 *
	 * @return string
	 */
	public function get_quiz_results_downloadable() {
		return $this->quiz_results_downloadable;
	}

	/**
	 * Get quiz results by tag.
	 *
	 * @since 4.2.0
	 *
	 * @return string
	 */
	public function get_quiz_results_by_tag() {
		return $this->quiz_results_by_tag;
	}

	/**
	 * Get quiz results by timer.
	 *
	 * @since 4.2.0
	 *
	 * @return string
	 */
	public function get_quiz_results_by_timer() {
		return $this->quiz_results_by_timer;
	}

	/**
	 * Get quiz recommended score.
	 *
	 * @since 4.2.0
	 *
	 * @return string
	 */
	public function get_quiz_recommended_score() {
		return $this->quiz_recommended_score;
	}

	/**
	 * Get show recommended percentage.
	 *
	 * @since 4.2.0
	 *
	 * @return int
	 */
	public function get_show_recommended_percentage() {
		return $this->show_recommended_percentage;
	}
}