<?php
/**
 * WP Courseware Module Model.
 *
 * @package WPCW
 * @subpackage Models
 * @since 4.1.0
 */
namespace WPCW\Models;

use WPCW\Database\DB_Modules;

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Class Module.
 *
 * @since 4.1.0
 *
 * @property int $module_id
 * @property int $parent_course_id
 * @property int $module_author
 * @property string $module_title
 * @property string $module_desc
 * @property int $module_order
 * @property int $module_number
 * @property int $course_id
 * @property int $course_title
 */
class Module extends Model {

	/**
	 * @var DB_Modules The modules database.
	 * @since 4.3.0
	 */
	protected $db;

	/**
	 * @var int Module Id.
	 * @since 4.1.0
	 */
	public $module_id;

	/**
	 * @var int Module Course Id.
	 * @since 4.1.0
	 */
	public $parent_course_id;

	/**
	 * @var int Module Author
	 * @since 4.1.0
	 */
	public $module_author = 0;

	/**
	 * @var string Module Author.
	 * @since 4.1.0
	 */
	public $module_title = '';

	/**
	 * @var string Module Description.
	 * @since 4.1.0
	 */
	public $module_desc = '';

	/**
	 * @var int Module Order.
	 * @since 4.1.0
	 */
	public $module_order = 0;

	/**
	 * @var int Module Number.
	 * @since 4.1.0
	 */
	public $module_number = 0;

	/**
	 * @var int Course Id.
	 * @since 4.1.0
	 */
	public $course_id = 0;

	/**
	 * @var string Course Title.
	 * @since 4.1.0
	 */
	public $course_title = '';

	/**
	 * Module Constructor.
	 *
	 * @since 4.3.0
	 *
	 * @param array|int|Model $data The model data.
	 */
	public function __construct( $data = array() ) {
		$this->db = new DB_Modules();
		parent::__construct( $data );
	}

	/**
	 * Get Module Id.
	 *
	 * @since 4.3.0
	 *
	 * @return int|void
	 */
	public function get_id() {
		return absint( $this->get_module_id() );
	}

	/**
	 * Get Module Id.
	 *
	 * @since 4.1.0
	 *
	 * @return mixed
	 */
	public function get_module_id() {
		return absint( $this->module_id );
	}

	/**
	 * Get Parent Course Id.
	 *
	 * @since 4.1.0
	 *
	 * @return mixed
	 */
	public function get_parent_course_id() {
		return absint( $this->parent_course_id );
	}

	/**
	 * Get Module Author
	 *
	 * @since 4.1.0
	 *
	 * @return int
	 */
	public function get_module_author() {
		return absint( $this->module_author );
	}

	/**
	 * Get Module Title.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_module_title() {
		return esc_attr( $this->module_title );
	}

	/**
	 * Get Module Description.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_module_desc() {
		return wp_kses_post( $this->module_desc );
	}

	/**
	 * Get Module Order.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_module_order() {
		return absint( $this->module_order );
	}

	/**
	 * Get Module Number.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_module_number() {
		return absint( $this->module_number );
	}

	/**
	 * Get Course Id.
	 *
	 * @since 4.1.0
	 *
	 * @return int
	 */
	public function get_course_id() {
		if ( empty( $this->course_id ) ) {
			return $this->get_parent_course_id();
		}

		return absint( $this->course_id );
	}

	/**
	 * Get Course Title.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_title() {
		return esc_attr( $this->course_title );
	}
}