<?php
/**
 * WP Courseware Course Model.
 *
 * @package WPCW
 * @subpackage Models
 * @since 4.1.0
 */
namespace WPCW\Models;

use WPCW\Database\DB_Course_Meta;
use WPCW\Database\DB_Courses;

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Class Course.
 *
 * @since 4.3.0
 *
 * @property int $course_id
 * @property string $course_title
 * @property string $course_desc
 * @property int $course_author
 * @property string $course_opt_completion_wall
 * @property string $course_opt_use_certificate
 * @property string $course_opt_user_access
 * @property int $course_unit_count
 * @property string $course_from_name
 * @property string $course_from_email
 * @property string $course_to_email
 * @property string $course_opt_prerequisites
 * @property string $course_message_unit_complete
 * @property string $course_message_course_complete
 * @property string $course_message_unit_not_logged_in
 * @property string $course_message_unit_pending
 * @property string $course_message_unit_no_access
 * @property string $course_message_prerequisite_not_met
 * @property string $course_message_unit_not_yet
 * @property string $course_message_unit_not_yet_dripfeed
 * @property string $course_message_quiz_open_grading_blocking
 * @property string $course_message_quiz_open_grading_non_blocking
 * @property string $email_complete_module_option_admin
 * @property string $email_complete_module_option
 * @property string $email_complete_module_subject
 * @property string $email_complete_module_body
 * @property string $email_complete_course_option_admin
 * @property string $email_complete_course_option
 * @property string $email_complete_course_subject
 * @property string $email_complete_course_body
 * @property string $email_quiz_grade_option
 * @property string $email_quiz_grade_subject
 * @property string $email_quiz_grade_body
 * @property string $email_complete_course_grade_summary_subject
 * @property string $email_complete_course_grade_summary_body
 * @property string $email_unit_unlocked_subject
 * @property string $email_unit_unlocked_body
 * @property string $cert_signature_type
 * @property string $cert_sig_text
 * @property string $cert_sig_image_url
 * @property string $cert_logo_enabled
 * @property string $cert_logo_url
 * @property string $cert_background_type
 * @property string $cert_background_custom_url
 * @property string $payments_type
 * @property string $payments_price
 * @property string $payments_interval
 */
class Course extends Model {

	/**
	 * @var DB_Courses The courses database.
	 * @since 4.3.0
	 */
	protected $db;

	/**
	 * @var DB_Course_Meta The courses meta database.
	 * @sicn
	 */
	protected $meta_db;

	/**
	 * @var int
	 * @since 4.1.0
	 */
	public $course_id;

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_title = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_desc = '';

	/**
	 * @var int
	 * @since 4.1.0
	 */
	public $course_author = 0;

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_opt_completion_wall = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_opt_use_certificate = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_opt_user_access = '';

	/**
	 * @var int
	 * @since 4.1.0
	 */
	public $course_unit_count = 0;

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_from_name = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_from_email = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_to_email = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_opt_prerequisites = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_message_unit_complete = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_message_course_complete = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_message_unit_not_logged_in = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_message_unit_pending = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_message_unit_no_access = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_message_prerequisite_not_met = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_message_unit_not_yet = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_message_unit_not_yet_dripfeed = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_message_quiz_open_grading_blocking = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $course_message_quiz_open_grading_non_blocking = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $email_complete_module_option_admin = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $email_complete_module_option = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $email_complete_module_subject = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $email_complete_module_body = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $email_complete_course_option_admin = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $email_complete_course_option = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $email_complete_course_subject = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $email_complete_course_body = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $email_quiz_grade_option = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $email_quiz_grade_subject = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $email_quiz_grade_body = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $email_complete_course_grade_summary_subject = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $email_complete_course_grade_summary_body = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $email_unit_unlocked_subject = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $email_unit_unlocked_body = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $cert_signature_type = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $cert_sig_text = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $cert_sig_image_url = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $cert_logo_enabled = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $cert_logo_url = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $cert_background_type = '';

	/**
	 * @var string
	 * @since 4.1.0
	 */
	public $cert_background_custom_url = '';

	/**
	 * @var string
	 * @since 4.3.0
	 */
	public $payments_type = 'free';

	/**
	 * @var string
	 * @since 4.3.0
	 */
	public $payments_price = '0.00';

	/**
	 * @var string
	 * @since 4.3.0
	 */
	public $payments_interval = 'month';

	/**
	 * Course Constructor.
	 *
	 * @since 4.3.0
	 *
	 * @param array|int|Model $data The data to setup the course.
	 */
	public function __construct( $data = array() ) {
		$this->db      = new DB_Courses();
		$this->meta_db = new DB_Course_Meta();
		parent::__construct( $data );
	}

	/**
	 * Get Course Id.
	 *
	 * @since 4.3.0
	 *
	 * @return int|void
	 */
	public function get_id() {
		return absint( $this->get_course_id() );
	}

	/**
	 * Create Course.
	 *
	 * @since 4.3.0
	 *
	 * @param array $data The data to insert upon creation.
	 *
	 * @return int|bool $course_id The course id or false otherwise.
	 */
	public function create( $data = array() ) {
		$defaults = array(
			'status'       => 'create',
			'order_key'    => $this->create_order_key(),
			'created_via'  => esc_attr( $created_via ),
			'date_created' => current_time( 'mysql' ),
		);
		$data     = wp_parse_args( $data, $defaults );

		if ( $course_id = $this->db->insert_course( $data ) ) {
			$this->set_prop( 'course_id', $course_id );
			$this->set_data( $data );
		}

		return $course_id;
	}

	/**
	 * Get course_id
	 *
	 * @since 4.1.0
	 *
	 * @return int
	 */
	public function get_course_id() {
		return absint( $this->course_id );
	}

	/**
	 * Get course_title
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_title() {
		return $this->course_title;
	}

	/**
	 * Get course_desc
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_desc() {
		return $this->course_desc;
	}

	/**
	 * Get course_author
	 *
	 * @since 4.1.0
	 *
	 * @return int
	 */
	public function get_course_author() {
		return absint( $this->course_author );
	}

	/**
	 * Get course_opt_completion_wall
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_opt_completion_wall() {
		return $this->course_opt_completion_wall;
	}

	/**
	 * Get course_opt_use_certificate
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_opt_use_certificate() {
		return $this->course_opt_use_certificate;
	}

	/**
	 * Get course_opt_user_access
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_opt_user_access() {
		return $this->is_purchasable() ? 'default_show' : $this->course_opt_user_access;
	}

	/**
	 * Get Course User Access Message.
	 *
	 * @since 4.3.0
	 *
	 * @return string $message The user access message.
	 */
	public function get_course_opt_user_access_message() {
		$message = esc_html__( 'Give new users access by default', 'wp-courseware' );

		if ( $this->is_purchasable() ) {
			if ( $this->is_subscription() ) {
				$message = esc_html__( 'New users given access if they have an active subscription', 'wp-courseware' );
			} else {
				$message = esc_html__( 'New users given access if they have paid for this course', 'wp-courseware' );
			}
		}

		return apply_filters( 'wpcw_course_user_access_message', $message );
	}

	/**
	 * Get course_unit_count
	 *
	 * @since 4.1.0
	 *
	 * @return int
	 */
	public function get_course_unit_count() {
		return $this->course_unit_count;
	}

	/**
	 * Get course_from_name
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_from_name() {
		return $this->course_from_name;
	}

	/**
	 * Get course_from_email
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_from_email() {
		return $this->course_from_email;
	}

	/**
	 * Get course_to_email
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_to_email() {
		return $this->course_to_email;
	}

	/**
	 * Get course_opt_prerequisites
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_opt_prerequisites() {
		return $this->course_opt_prerequisites;
	}

	/**
	 * Get course_message_unit_complete
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_message_unit_complete() {
		return $this->course_message_unit_complete;
	}

	/**
	 * Get course_message_course_complete
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_message_course_complete() {
		return $this->course_message_course_complete;
	}

	/**
	 * Get course_message_unit_not_logged_in
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_message_unit_not_logged_in() {
		return $this->course_message_unit_not_logged_in;
	}

	/**
	 * Get course_message_unit_pending
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_message_unit_pending() {
		return $this->course_message_unit_pending;
	}

	/**
	 * Get course_message_unit_no_access
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_message_unit_no_access() {
		return $this->course_message_unit_no_access;
	}

	/**
	 * Get course_message_prerequisite_not_met
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_message_prerequisite_not_met() {
		return $this->course_message_prerequisite_not_met;
	}

	/**
	 * Get course_message_unit_not_yet
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_message_unit_not_yet() {
		return $this->course_message_unit_not_yet;
	}

	/**
	 * Get course_message_unit_not_yet_dripfeed
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_message_unit_not_yet_dripfeed() {
		return $this->course_message_unit_not_yet_dripfeed;
	}

	/**
	 * Get course_message_quiz_open_grading_blocking
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_message_quiz_open_grading_blocking() {
		return $this->course_message_quiz_open_grading_blocking;
	}

	/**
	 * Get course_message_quiz_open_grading_non_blocking
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_course_message_quiz_open_grading_non_blocking() {
		return $this->course_message_quiz_open_grading_non_blocking;
	}

	/**
	 * Get email_complete_module_option_admin
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_email_complete_module_option_admin() {
		return $this->email_complete_module_option_admin;
	}

	/**
	 * Get email_complete_module_option
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_email_complete_module_option() {
		return $this->email_complete_module_option;
	}

	/**
	 * Get email_complete_module_subject
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_email_complete_module_subject() {
		return $this->email_complete_module_subject;
	}

	/**
	 * Get email_complete_module_body
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_email_complete_module_body() {
		return $this->email_complete_module_body;
	}

	/**
	 * Get email_complete_course_option_admin
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_email_complete_course_option_admin() {
		return $this->email_complete_course_option_admin;
	}

	/**
	 * Get email_complete_course_option
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_email_complete_course_option() {
		return $this->email_complete_course_option;
	}

	/**
	 * Get email_complete_course_subject
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_email_complete_course_subject() {
		return $this->email_complete_course_subject;
	}

	/**
	 * Get email_complete_course_body
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_email_complete_course_body() {
		return $this->email_complete_course_body;
	}

	/**
	 * Get email_quiz_grade_option
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_email_quiz_grade_option() {
		return $this->email_quiz_grade_option;
	}

	/**
	 * Get email_quiz_grade_subject
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_email_quiz_grade_subject() {
		return $this->email_quiz_grade_subject;
	}

	/**
	 * Get email_quiz_grade_body
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_email_quiz_grade_body() {
		return $this->email_quiz_grade_body;
	}

	/**
	 * Get email_complete_course_grade_summary_subject
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_email_complete_course_grade_summary_subject() {
		return $this->email_complete_course_grade_summary_subject;
	}

	/**
	 * Get email_complete_course_grade_summary_body
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_email_complete_course_grade_summary_body() {
		return $this->email_complete_course_grade_summary_body;
	}

	/**
	 * Get email_unit_unlocked_subject
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_email_unit_unlocked_subject() {
		return $this->email_unit_unlocked_subject;
	}

	/**
	 * Get email_unit_unlocked_body
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_email_unit_unlocked_body() {
		return $this->email_unit_unlocked_body;
	}

	/**
	 * Get cert_signature_type
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_cert_signature_type() {
		return $this->cert_signature_type;
	}

	/**
	 * Get cert_sig_text
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_cert_sig_text() {
		return $this->cert_sig_text;
	}

	/**
	 * Get cert_sig_image_url
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_cert_sig_image_url() {
		return $this->cert_sig_image_url;
	}

	/**
	 * Get cert_logo_enabled
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_cert_logo_enabled() {
		return $this->cert_logo_enabled;
	}

	/**
	 * Get cert_logo_url
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_cert_logo_url() {
		return $this->cert_logo_url;
	}

	/**
	 * Get cert_background_type
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_cert_background_type() {
		return $this->cert_background_type;
	}

	/**
	 * Get cert_background_custom_url
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_cert_background_custom_url() {
		return $this->cert_background_custom_url;
	}

	/**
	 * Get Course Payments Type.
	 *
	 * @since 4.3.0
	 *
	 * @return string The course payments type.
	 */
	public function get_payments_type() {
		return esc_attr( $this->payments_type );
	}

	/**
	 * Get Course Payments Price.
	 *
	 * @since 4.3.0
	 *
	 * @return string The course payments price.
	 */
	public function get_payments_price() {
		return $this->payments_price;
	}

	/**
	 * Get Course Payments Price Refunded.
	 *
	 * @since 4.3.0
	 *
	 * @return string The course payments price.
	 */
	public function get_payments_price_refunded() {
		return $this->payments_price * -1;
	}

	/**
	 * Get Course Payments Interval.
	 *
	 * @since 4.3.0
	 *
	 * @return string The course payments interval.
	 */
	public function get_payments_interval() {
		return $this->payments_interval;
	}

	/**
	 * Is Course a Subscription?
	 *
	 * @since 4.3.0
	 *
	 * @return bool True if it is, false otherwise.
	 */
	public function is_subscription() {
		return 'subscription' === $this->get_payments_type();
	}

	/**
	 * Can Course be Purchased?
	 *
	 * @since 4.3.0
	 *
	 * @return bool True if it can be purchased, false otherwise.
	 */
	public function is_purchasable() {
		$payments_type = $this->get_payments_type();

		return ( ! empty( $payments_type ) && 'free' !== $payments_type ) ? true : false;
	}

	/**
	 * Can User Access Course.
	 *
	 * @since 4.3.0
	 *
	 * @return bool True if the user can acccess a course. False otherwise.
	 */
	public function can_user_access( $user_id = 0 ) {
		global $wpdb;

		if ( empty( $user_id ) ) {
			return false;
		}

		$table_name = wpcw()->database->get_table_name( 'user_courses' );
		$course_id  = $this->get_course_id();

		$result = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table_name WHERE user_id = %d AND course_id = %d LIMIT 1;", absint( $user_id ), absint( $course_id ) ) );

		$can_access = apply_filters( 'wpcw_courses_canuseraccesscourse', $result, $this->get_course_id(), $user_id );

		return apply_filters( 'wpcw_courses_can_user_access', $can_access, $this->get_course_id(), $user_id );
	}

	/**
	 * Get Subscription Interval Price
	 *
	 * @since 4.3.0
	 */
	public function get_subscription_interval() {
		$periods = wpcw()->subscriptions->get_periods();

		$interval = $this->get_payments_interval();

		return isset( $periods[ $interval ] ) ? $periods[ $interval ] : $interval;
	}

	/**
	 * Get Price Label.
	 *
	 * @since 4.3.0
	 *
	 * @return string The price label.
	 */
	public function get_price_label() {
		$price = wpcw_price( $this->get_payments_price() );

		if ( $this->is_subscription() ) {
			$price = sprintf( '%s / %s', $price, $this->get_subscription_interval() );
		}

		return $price;
	}

	/**
	 * Get a Course Meta Field.
	 *
	 * @since 4.3.0
	 *
	 * @param string $meta_key The meta key.
	 * @param bool $single Whether to return a single value.
	 *
	 * @return mixed Will be an array if $single is false. Will be value of meta data field if $single is true.
	 */
	public function get_meta( $meta_key = '', $single = true ) {
		return $this->meta_db->get_meta( $this->get_course_id(), $meta_key, $single );
	}

	/**
	 * Add Course Meta Field.
	 *
	 * @since 4.3.0
	 *
	 * @param string $meta_key Metadata name.
	 * @param mixed $meta_value Metadata value.
	 * @param bool $unique Optional, default is false. Whether the same key should not be added.
	 *
	 * @return bool False for failure. True for success.
	 */
	public function add_meta( $meta_key = '', $meta_value, $unique = false ) {
		return $this->meta_db->add_meta( $this->get_course_id(), $meta_key, $meta_value, $unique );
	}

	/**
	 * Update Course Meta Field.
	 *
	 * @since 4.3.0
	 *
	 * @param string $meta_key Metadata key.
	 * @param mixed $meta_value Metadata value.
	 * @param mixed $prev_value Optional. Previous value to check before removing.
	 *
	 * @return bool False on failure, true if success.
	 */
	public function update_meta( $meta_key = '', $meta_value, $prev_value = '' ) {
		return $this->meta_db->update_meta( $this->get_course_id(), $meta_key, $meta_value, $prev_value );
	}

	/**
	 * Delete Course Meta Field.
	 *
	 * @since 4.3.0
	 *
	 * @param string $meta_key Metadata name.
	 * @param mixed $meta_value Optional. Metadata value.
	 *
	 * @return bool False for failure. True for success.
	 */
	public function delete_meta( $meta_key = '', $meta_value = '' ) {
		return $this->meta_db->delete_meta( $this->get_course_id(), $meta_key, $meta_value );
	}
}