<?php
/**
 * WP Courseware Course Functions.
 *
 * @package WPCW
 * @subpackage Functions
 * @since 4.3.0
 */

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Get Course.
 *
 * @since 4.3.0
 *
 * @param int|bool $course_id The Course Id.
 *
 * @return \WPCW\Models\Course|bool An course object or false.
 */
function wpcw_get_course( $course_id = false ) {
	if ( ! did_action( 'wpcw_loaded' ) ) {
		wpcw_doing_it_wrong( __FUNCTION__, 'wpcw_get_course should not be called the course object is setup.', '4.3.0' );

		return false;
	}

	return new \WPCW\Models\Course( $course_id );
}

/**
 * Get Courses.
 *
 * @since 4.3.0
 *
 * @param array $args The courses query args.
 *
 * @return array The array of Course objects.
 */
function wpcw_get_courses( $args = array() ) {
	if ( isset( $args['student_id'] ) ) {
		return wpcw()->courses->get_courses_by_student( absint( $args['student_id'] ) );
	}

	return wpcw()->courses->get_courses( $args );
}

/**
 * Get Test Course.
 *
 * @since 4.3.0
 *
 * @return \WPCW\Models\Course $test_course The test course object.
 */
function wpcw_get_test_course() {
	$test_course = new \WPCW\Models\Course();

	$test_data = array(
		'course_id'                                     => rand(),
		'course_title'                                  => esc_html__( 'Free Course', 'wp-courseware' ),
		'course_desc'                                   => esc_html__( 'This is a free course.', 'wp-coursewar' ),
		'course_author'                                 => 1,
		'course_opt_completion_wall'                    => 'completion_wall',
		'course_opt_use_certificate'                    => 'use_certs',
		'course_opt_user_access'                        => 'default_show',
		'course_unit_count'                             => '1',
		'course_from_name'                              => esc_html__( 'WP Courseware', 'wp-courseware' ),
		'course_from_email'                             => 'wpcw.test.admin@wpcwtest.com',
		'course_to_email'                               => 'wpcw.test.admin@wpcwtest.com',
		'course_opt_prerequisites'                      => 'a:0:{}',
		'course_message_unit_complete'                  => 'You have now completed this unit.',
		'course_message_course_complete'                => 'You have now completed the whole course. Congratulations!',
		'course_message_unit_not_logged_in'             => 'You cannot view this unit as you\'re not logged in yet.',
		'course_message_unit_pending'                   => 'Have you completed this unit? Then mark this unit as completed.',
		'course_message_unit_no_access'                 => 'Sorry, but you\'re not allowed to access this course.',
		'course_message_prerequisite_not_met'           => 'This course can not be accessed until the prerequisites for this course are complete.',
		'course_message_unit_not_yet'                   => 'You need to complete the previous unit first.',
		'course_message_unit_not_yet_dripfeed'          => 'This unit isn\'t available just yet. Please check back in about {UNIT_UNLOCKED_TIME}.',
		'course_message_quiz_open_grading_blocking'     => 'Your quiz has been submitted for grading by the course instructor. Once your grade has been entered, you will be able to access the next unit.',
		'course_message_quiz_open_grading_non_blocking' => 'Your quiz has been submitted for grading by the course instructor. You have now completed this unit.',
		'email_complete_module_option_admin'            => 'send_email',
		'email_complete_module_option'                  => 'send_email',
		'email_complete_module_subject'                 => 'Module {MODULE_TITLE} - Complete.',
		'email_complete_module_body'                    => 'Hi {USER_NAME}

Great work for completing the "{MODULE_TITLE}" module!

{SITE_NAME}
{SITE_URL}',
		'email_complete_course_option_admin'            => 'send_email',
		'email_complete_course_option'                  => 'send_email',
		'email_complete_course_subject'                 => 'Course {COURSE_TITLE} - Complete',
		'email_complete_course_body'                    => 'Hi {USER_NAME}

Great work for completing the "{COURSE_TITLE}" training course! Fantastic!

{SITE_NAME}
{SITE_URL}',
		'email_quiz_grade_option'                       => 'send_email',
		'email_quiz_grade_subject'                      => '{COURSE_TITLE} - Your Quiz Grade - For "{QUIZ_TITLE}"',
		'email_quiz_grade_body'                         => 'Hi {USER_NAME}

Your grade for the "{QUIZ_TITLE}" quiz is:
{QUIZ_GRADE}

This was for the quiz at the end of this unit:
{UNIT_URL}

{QUIZ_RESULT_DETAIL}

{SITE_NAME}
{SITE_URL}',
		'email_complete_course_grade_summary_subject'   => 'Your final grade summary for "{COURSE_TITLE}"',
		'email_complete_course_grade_summary_body'      => 'Hi {USER_NAME}

Congratulations on completing the "{COURSE_TITLE}" training course! Fantastic!

Your final grade is: {CUMULATIVE_GRADE}

Here is a summary of your quiz results:
{QUIZ_SUMMARY}

You can download your certificate here:
{CERTIFICATE_LINK}

I hope you enjoyed the course!

{SITE_NAME}
{SITE_URL}',
		'email_unit_unlocked_subject'                   => 'Your next unit ({UNIT_TITLE}) is now available!',
		'email_unit_unlocked_body'                      => 'Hi {USER_NAME}

Good news! You can now access the next unit!

The unit called "{UNIT_TITLE}" from "{COURSE_TITLE}" is now available for you to access.

To access the unit, just click on this link:
{UNIT_URL}

Thanks!

{SITE_NAME}
{SITE_URL}',
		'cert_signature_type'                           => 'text',
		'cert_sig_text'                                 => 'WP Courseware',
		'cert_sig_image_url'                            => '',
		'cert_logo_enabled'                             => 'no_cert_logo',
		'cert_logo_url'                                 => '',
		'cert_background_type'                          => 'use_default',
		'cert_background_custom_url'                    => '',
		'payments_type'                                 => 'free',
		'payments_price'                                => '5.00',
		'payments_interval'                             => 'month',
	);

	$test_course->set_data( $test_data );

	return $test_course;
}

/**
 * Get Course Title.
 *
 * @since 4.3.0
 *
 * @param \WPCW\Models\Course The course object.
 *
 * @return string The course title.
 */
function wpcw_course_get_title( $course ) {
	if ( ! $course instanceof \WPCW\Models\Course ) {
		return;
	}

	return esc_html( $course->get_course_title() );
}

/**
 * Get Course Description.
 *
 * @since 4.3.0
 *
 * @param \WPCW\Models\Course The course description.
 *
 * @return string The course description.
 */
function wpcw_course_get_desc( $course ) {
	if ( ! $course instanceof \WPCW\Models\Course ) {
		return;
	}

	return wp_kses_post( $course->get_course_desc() );
}

/**
 * Get Purchase Type.
 *
 * @since 4.3.0
 *
 * @param int $course_id The course id.
 */
function wpcw_course_get_payments_type( $course_id ) {
	return wpcw()->courses->get_course_payments_type( $course_id );
}

/**
 * Get Course Enroll / Purchase button.
 *
 * @since 4.3.0
 *
 * @param \WPCW\Models\Course The course description.
 *
 * @return string The course enroll / purchase button.
 */
function wpcw_course_get_enroll_purchase_button( $course ) {
	if ( ! $course instanceof \WPCW\Models\Course ) {
		return;
	}

	return ( $course->is_purchasable() ) ? wpcw_add_to_cart_link( $course ) : do_shortcode( '[wpcw_course_enroll courses=' . $course->get_course_id() . ']' );
}