<?php
/**
 * WP Courseware Conditional Functions.
 *
 * @package WPCW
 * @subpackage Functions
 * @since 4.3.0
 */

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * What type of request is this?
 *
 * @since 4.3.0
 *
 * @param string $type The string request type.
 *
 * @return bool True on success, false on failure.
 */
function wpcw_is_request( $type ) {
	switch ( $type ) {
		case 'admin':
			return is_admin();
		case 'ajax':
			return defined( 'DOING_AJAX' ) || defined( 'WPCW_DOING_AJAX' );
		case 'cron':
			return defined( 'DOING_CRON' );
		case 'frontend':
			return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
	}
}

/**
 * Is_ajax - Returns true when the page is loaded via ajax.
 *
 * @since 4.3.0
 *
 * @return bool
 */
function wpcw_is_ajax() {
	return defined( 'DOING_AJAX' );
}

/**
 * Is Doing Cron?
 *
 * Returns true when a cron job is running.
 *
 * @since 4.3.0
 *
 * @return boolean True if doing a cron job.
 */
function wpcw_is_doing_cron() {
	if ( function_exists( 'wp_doing_cron' ) && wp_doing_cron() ) {
		return true;
	} elseif ( defined( 'DOING_CRON' ) && ( true === DOING_CRON ) ) {
		return true;
	}

	return false;
}

/**
 * Check if an endpoint is showing.
 *
 * @since 4.3.0
 *
 * @param string $endpoint Which endpoint to check, or any.
 *
 * @return bool True on successful check, false on failure.
 */
function wpcw_is_endpoint_url( $endpoint = false ) {
	global $wp;

	$endpoints = wpcw()->query->get_query_vars();

	if ( false !== $endpoint ) {
		if ( ! isset( $endpoints[ $endpoint ] ) ) {
			return false;
		} else {
			$endpoint_var = $endpoints[ $endpoint ];
		}

		return isset( $wp->query_vars[ $endpoint_var ] );
	}

	foreach ( $endpoints as $key => $value ) {
		if ( isset( $wp->query_vars[ $key ] ) ) {
			return true;
		}
	}

	return false;
}

/**
 * Is Valid url?
 *
 * Simple check for validating a URL, it must start with http:// or https://.
 * and pass FILTER_VALIDATE_URL validation.
 *
 * @since 4.3.0
 *
 * @param string $url The url to check.
 *
 * @return bool True on success, False on failure.
 */
function wpcw_is_valid_url( $url ) {
	// Must start with http:// or https://.
	if ( 0 !== strpos( $url, 'http://' ) && 0 !== strpos( $url, 'https://' ) ) {
		return false;
	}

	// Must pass validation.
	if ( ! filter_var( $url, FILTER_VALIDATE_URL ) ) {
		return false;
	}

	return true;
}

/**
 * Is site secured with https.
 *
 * Check if the home URL is https. If it is, we don't need to do things such as 'force ssl'.
 *
 * @since 4.3.0
 *
 * @return bool
 */
function wpcw_is_site_https() {
	return false !== strstr( get_option( 'home' ), 'https:' );
}

/**
 * Is the Checkout Page secured with https.
 *
 * Look at options, WP HTTPS plugin, or the permalink itself.
 *
 * @since 4.3.0
 *
 * @return bool
 */
function wpcw_is_checkout_https() {
	return wpcw_is_site_https() || 'yes' === wpcw()->settings->get_setting( 'force_ssl' ) || class_exists( 'WordPressHTTPS' ) || strstr( wpcw_get_page_permalink( 'checkout' ), 'https:' );
}

/**
 * Check the content for a specific shortcode.
 *
 * @since 4.3.0
 *
 * @param string $tag The shortcode tag to check.
 *
 * @return bool True on success, False on failure.
 */
function wpcw_post_content_has_shortcode( $tag = '' ) {
	global $post;

	return is_singular() && is_a( $post, 'WP_Post' ) && has_shortcode( $post->post_content, $tag );
}

/**
 * Is Checkout Page?
 *
 * @since 4.3.0
 *
 * @return bool
 */
function wpcw_is_checkout() {
	return is_page( wpcw_get_page_id( 'checkout' ) ) || wpcw_post_content_has_shortcode( 'wpcw_checkout' ) || apply_filters( 'wpcw_is_checkout', false ) || defined( 'WPCW_CHECKOUT' );
}

/**
 * Is Account Page?
 *
 * @since 4.3.0
 *
 * @return bool
 */
function wpcw_is_account_page() {
	return is_page( wpcw_get_page_id( 'account' ) ) || wpcw_post_content_has_shortcode( 'wpcw_account' ) || apply_filters( 'wpcw_is_account_page', false );
}

/**
 * Is Order Received Page?
 *
 * @since 4.3.0
 *
 * @return bool
 */
function wpcw_is_order_received_page() {
	global $wp;

	return apply_filters( 'wpcw_is_order_received_page', ( is_page( wpcw_get_page_id( 'checkout' ) ) && isset( $wp->query_vars['order-received'] ) ) );
}

/**
 * Is view order page?
 *
 * @since 4.3.0
 *
 * @return bool
 */
function wpcw_is_view_order_page() {
	global $wp;

	return ( is_page( wpcw_get_page_id( 'account' ) ) && isset( $wp->query_vars['order'] ) );
}