<?php
/**
 * WP Courseware DB Courses.
 *
 * @package WPCW
 * @subpackage Database
 * @since 4.3.0
 */
namespace WPCW\Database;

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Class DB_Courses.
 *
 * @since 4.3.0
 */
class DB_Courses extends DB {

	/**
	 * Courses Database Constructor.
	 *
	 * Intiate the table name, version, and primary key.
	 *
	 * @since 4.1.0
	 */
	public function __construct() {
		// Define Table Name and Primary Key
		$this->table_name  = wpcw()->database->get_table_name( 'courses' );
		$this->primary_key = 'course_id';
	}

	/**
	 * Get Columns.
	 *
	 * @since 4.1.0
	 *
	 * @return array The array of columns.
	 */
	public function get_columns() {
		return array(
			'course_id'                                     => '%d',
			'course_title'                                  => '%s',
			'course_desc'                                   => '%s',
			'course_author'                                 => '%d',
			'course_opt_completion_wall'                    => '%s',
			'course_opt_use_certificate'                    => '%s',
			'course_opt_user_access'                        => '%s',
			'course_unit_count'                             => '%d',
			'course_from_name'                              => '%s',
			'course_from_email'                             => '%s',
			'course_to_email'                               => '%s',
			'course_opt_prerequisites'                      => '%s',
			'course_message_unit_complete'                  => '%s',
			'course_message_course_complete'                => '%s',
			'course_message_unit_not_logged_in'             => '%s',
			'course_message_unit_pending'                   => '%s',
			'course_message_unit_no_access'                 => '%s',
			'course_message_prerequisite_not_met'           => '%s',
			'course_message_unit_not_yet'                   => '%s',
			'course_message_unit_not_yet_dripfeed'          => '%s',
			'course_message_quiz_open_grading_blocking'     => '%s',
			'course_message_quiz_open_grading_non_blocking' => '%s',
			'email_complete_module_option_admin'            => '%s',
			'email_complete_module_option'                  => '%s',
			'email_complete_module_subject'                 => '%s',
			'email_complete_module_body'                    => '%s',
			'email_complete_course_option_admin'            => '%s',
			'email_complete_course_option'                  => '%s',
			'email_complete_course_subject'                 => '%s',
			'email_complete_course_body'                    => '%s',
			'email_quiz_grade_option'                       => '%s',
			'email_quiz_grade_subject'                      => '%s',
			'email_quiz_grade_body'                         => '%s',
			'email_complete_course_grade_summary_subject'   => '%s',
			'email_complete_course_grade_summary_body'      => '%s',
			'email_unit_unlocked_subject'                   => '%s',
			'email_unit_unlocked_body'                      => '%s',
			'cert_signature_type'                           => '%s',
			'cert_sig_text'                                 => '%s',
			'cert_sig_image_url'                            => '%s',
			'cert_logo_enabled'                             => '%s',
			'cert_logo_url'                                 => '%s',
			'cert_background_type'                          => '%s',
			'cert_background_custom_url'                    => '%s',
			'payments_type'                                 => '%s',
			'payments_price'                                => '%s',
			'payments_interval'                             => '%s',
		);
	}

	/**
	 * Get Column Defaults.
	 *
	 * @since 4.3.0
	 *
	 * @return array
	 */
	public function get_column_defaults() {
		return array(
			'course_author'              => ( is_user_logged_in() && current_user_can( 'manage_wpcw_settings' ) ) ? get_current_user_id() : 0,
			'course_unit_count'          => 0,
			'course_opt_use_certificate' => 'no_certs',
			'cert_signature_type'        => 'text',
			'cert_sig_image_url'         => '',
			'cert_logo_enabled'          => 'no_cert_logo',
			'cert_logo_url'              => '',
			'cert_background_type'       => 'use_default',
			'payments_type'              => 'free',
			'payments_price'             => 0.00,
			'payments_interval'          => 'month',
		);
	}

	/**
	 * Get Courses.
	 *
	 * @since 4.1.0
	 *
	 * @param array $args An array of query arguments.
	 * @param bool $count Optional. Return only the total number of results.
	 *
	 * @return array Array of courses.
	 */
	public function get_courses( $args = array(), $count = false ) {
		global $wpdb;

		$defaults = array(
			'number'        => 10,
			'offset'        => 0,
			'course_author' => '',
			'course_id'     => 0,
			'order'         => 'DESC',
			'orderby'       => 'course_id',
			'search'        => '',
			'fields'        => '',
		);

		$args = wp_parse_args( $args, $defaults );
		
		$courses = array();

		$fields  = '';
		$join    = '';
		$where   = '';
		$orderby = $args['orderby'];
		$order   = strtoupper( $args['order'] );
		$wild    = '%';

		if ( 'ids' === $args['fields'] ) {
			$fields = "{$this->primary_key}";
		} else {
			$fields = $this->parse_fields( $args['fields'] );
		}

		if ( ! empty( $args['course_id'] ) ) {
			$where .= empty( $where ) ? ' WHERE ' : ' AND ';

			if ( is_array( $args['course_id'] ) ) {
				$course_ids = implode( ',', array_map( 'intval', $args['course_id'] ) );
			} else {
				$course_ids = intval( $args['course_id'] );
			}

			$where .= "course_id IN( {$course_ids} )";
		}

		if ( ! empty( $args['course_author'] ) ) {
			$where .= empty( $where ) ? ' WHERE ' : ' AND ';

			$course_author = intval( $args['course_author'] );

			$where .= "course_author = {$course_author}";
		}

		if ( ! empty( $args['search'] ) ) {
			$search_value = $args['search'];

			if ( is_numeric( $search_value ) ) {
				$search = $wpdb->prepare( "course_id IN( %s )", $search_value );
			} elseif ( is_string( $search_value ) ) {
				$search_value = $wild . $wpdb->esc_like( stripslashes( $search_value ) ) . $wild;
				$search       = $wpdb->prepare( "course_title LIKE %s", $search_value );
			}

			if ( ! empty( $search ) ) {
				$where .= empty( $where ) ? ' WHERE ' : ' AND ';
				$where .= $search;
			}
		}

		switch ( $args['orderby'] ) {
			case 'id' :
				$orderby = 'course_id';
				break;

			case 'title' :
				$orderby = 'course_title';
				break;

			case 'course_author' :
				$orderby = 'u.user_login';
				$join    = "c INNER JOIN {$wpdb->users} u ON c.course_author = u.ID";
				break;

			default :
				$orderby = array_key_exists( $args['orderby'], $this->get_columns() ) ? $args['orderby'] : $this->primary_key;
				break;
		}

		$args['orderby'] = $orderby;
		$args['order']   = $order;

		$clauses = compact( 'fields', 'join', 'where', 'orderby', 'order', 'count' );

		$results = $this->get_results( $clauses, $args );

		return $results;
	}

	/**
	 * Insert Course.
	 *
	 * @since 4.3.0
	 *
	 * @param array $data The courose data.
	 *
	 * @return int|bool The course id or false if an error occurred.
	 */
	public function insert_course( $data = array() ) {
		if ( empty( $data ) ) {
			return false;
		}

		$data = $this->sanitize_columns( $data );

		return $this->insert( $data, 'course' );
	}
}