<?php
/**
 * WP Courseware Install.
 *
 * @package WPCW
 * @subpackage Core
 * @since 4.3.0
 */
namespace WPCW\Core;

use WPCW_queue_dripfeed;

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Class Install
 *
 * @since 4.3.0
 */
final class Install {

	/**
	 * Load Activate Class.
	 *
	 * @since 4.3.0
	 */
	public function load() {
		register_activation_hook( WPCW_FILE, array( $this, 'install' ) );
		add_action( 'plugins_loaded', array( $this, 'upgrade' ), 0 );
		add_action( 'shutdown', array( $this, 'maybe_install_ecommerce' ) );
	}

	/**
	 * Install Plugin.
	 *
	 * @since 4.3.0
	 */
	public function install() {
		// Register Settings.
		wpcw()->settings->register_settings();

		// Remove flush rules flage.
		wpcw_remove_flush_rules_flag();

		// Create upload directory
		wpcw_create_upload_directory();

		// Clean up tags
		WPCW_tag_cleanup();

		// Setup or clear the scheduler for the notifications timers. If the setting is 'never'
		// then clear the scheduler. If it's anything else, then add it.
		$dripfeedSetting = WPCW_TidySettings_getSettingSingle( WPCW_DATABASE_SETTINGS_KEY, 'cron_time_dripfeed' );
		WPCW_queue_dripfeed::installNotificationHook_dripfeed( ( ! empty( $dripfeedSetting ) ) ? $dripfeedSetting : 'twicedaily' );

		// Create Roles & Capabilities
		$roles = new Roles();
		$roles->add_roles();
		$roles->add_caps();

		// Flush Rewrite Rules.
		flush_rewrite_rules();
	}

	/**
	 * Upgrade Plugin.
	 *
	 * @since 4.3.0
	 */
	public function upgrade() {
		$installed_db_version = get_option( 'wpcw_db_version' );
		$current_db_version   = WPCW_DB_VERSION;

		$installed_version = get_option( 'wpcw_plugin_version' );
		$current_version   = WPCW_VERSION;

		if ( $current_version != $installed_version ) {
			wpcw_remove_flush_rules_flag();
			update_option( 'wpcw_plugin_version', WPCW_VERSION );
			update_option( 'wpcw_db_version', WPCW_DB_VERSION );
		}

		wpcw_create_upload_directory();
	}

	/**
	 * Maybe Install E-Commerce Features.
	 *
	 * @since 4.3.0
	 */
	public function maybe_install_ecommerce() {
		if ( ! is_admin() ) {
			return;
		}

		if ( get_option( 'wpcw_install_ecommerce' ) ) {
			return;
		}

		if ( $this->can_install_ecommerce() ) {
			$pages = array( 'courses_page', 'checkout_page', 'order_received_page', 'order_failed_page', 'terms_page', 'account_page' );

			foreach ( $pages as $page ) {
				$page_id = wpcw_get_setting( $page );

				if ( 0 !== absint( $page_id ) ) {
					continue;
				}

				switch ( $page ) {
					case 'courses_page' :
						$page_id = wpcw()->courses->create_courses_page();
						break;
					case 'checkout_page' :
						$page_id = wpcw()->checkout->create_checkout_page();
						break;
					case 'order_received_page' :
						$page_id = wpcw()->checkout->create_checkout_order_received_page();
						break;
					case 'order_failed_page' :
						$page_id = wpcw()->checkout->create_checkout_order_failed_page();
						break;
					case 'terms_page' :
						$page_id = wpcw()->checkout->create_checkout_terms_page();
						break;
					case 'account_page' :
						$page_id = wpcw()->students->create_students_account_page();
						break;
				}

				if ( 0 !== absint( $page_id ) ) {
					wpcw_update_setting( $page, absint( $page_id ) );
				}
			}
		}

		update_option( 'wpcw_install_ecommerce', true );
		flush_rewrite_rules();
	}

	/**
	 * Can Install E-Commerce?
	 *
	 * @since 4.3.0
	 *
	 * @return bool True if the pages can be installed.
	 */
	public function can_install_ecommerce() {
		$can_install = true;

		if ( $can_install ) {
			$ecommerce_plugins = apply_filters( 'wpcw_install_ecommerce_plugins', array(
				'easy-digital-downloads/easy-digital-downloads.php',
				'woocommerce/woocommerce.php',
				'learnpress/learnpress.php',
				'memberpress/memberpress.php',
				'paid-memberships-pro/paid-memberships-pro.php',
				'easy-digital-downloads-addon-for-wp-courseware/wp-courseware-edd.php',
				'magic-member-addon-for-wp-courseware/wp-courseware-magic-members.php',
				'membermouse-addon-for-wp-courseware/wp-courseware-member-mouse.php',
				'memberpress-addon-for-wp-courseware/wp-courseware-memberpress.php',
				'membersonic-addon-for-wp-courseware/wp-courseware-membersonic.php',
				'om-addon-for-wp-courseware/wp-courseware-optimizemember.php',
				'paid-memberships-pro-for-wp-courseware/wp-courseware-pmpro.php',
				'premise-addon-for-wp-courseware/wp-courseware-premise.php',
				's2member-addon-for-wp-courseware/wp-courseware-s2-member.php',
				'wishlist-member-addon-for-wp-courseware/wp-courseware-wishlist-member.php',
				'woo-commerce-addon-for-wp-courseware/wp-courseware-woo-commerce.php',
			) );

			if ( ! function_exists( 'is_plugin_active' ) ) {
				include_once( ABSPATH . 'wp-admin/includes/plugin.php' );
			}

			foreach ( $ecommerce_plugins as $plugin ) {
				if ( is_plugin_active( $plugin ) ) {
					$can_install = false;
					break;
				}
			}
		}

		return $can_install;
	}
}