<?php
/**
 * WP Courseware Units Controller.
 *
 * @package WPCW
 * @subpackage Controllers
 * @since 4.3.0
 */
namespace WPCW\Controllers;

use WPCW\Core\Api;
use WPCW_queue_dripfeed;
use WPCW_UnitFrontend;
use WP_Query;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Class Units.
 *
 * @since 4.3.0
 */
class Units extends Controller {

	/**
	 * @var string The post type slug.
	 * @since 4.3.0
	 */
	protected $post_type_slug = 'course_unit';

	/**
	 * @var string The category slug.
	 * @since 4.3.0
	 */
	protected $taxonomy_category_slug = 'course_unit_category';

	/**
	 * @var string The tag slug.
	 * @since 4.3.0
	 */
	protected $taxonomy_tag_slug = 'course_unit_tag';

	/**
	 * Units Load.
	 *
	 * @since 4.3.0
	 */
	public function load() {
		// Course Unit Post Type.
		add_action( 'init', array( $this, 'post_type' ), 0 );
		add_filter( 'post_updated_messages', array( $this, 'post_type_updated_messages' ) );
		add_action( 'pre_get_posts', array( $this, 'post_type_exclude_from_search' ) );
		add_filter( 'the_content', array( $this, 'post_type_remove_content' ) );

		// Course Unit Taxonomies.
		add_action( 'init', array( $this, 'taxonomy_category' ), 0 );
		add_action( 'init', array( $this, 'taxonomy_tag' ), 0 );
		add_filter( 'admin_head', array( $this, 'taxonomy_menu_fix' ) );
		add_action( 'admin_head-edit-tags.php', array( $this, 'taxonomy_action_buttons' ) );
		add_action( 'admin_head-edit-tags.php', array( $this, 'taxonomy_title_icon' ) );

		// Course Unit Comments & Rewrite Support.
		add_action( 'init', array( $this, 'post_type_comments_support' ), 0 );
		add_action( 'init', array( $this, 'post_type_rewrites_support' ), 0 );

		// Add Taxonomy Buttons
		add_filter( 'wpcw_admin_page_units_action_buttons', array( $this, 'add_taxonomy_buttons' ) );
		add_filter( 'wpcw_admin_page_units_single_action_buttons', array( $this, 'add_taxonomy_buttons' ) );

		// Update Drip Feed Cron.
		add_action( 'wpcw_settings_after_save', array( $this, 'update_drip_feed_cron' ) );

		// Api Endpoints
		add_filter( 'wpcw_api_endoints', array( $this, 'register_api_endpoints' ), 10, 2 );
	}

	/**
	 * Get Units Settings Fields.
	 *
	 * @since 4.3.0
	 *
	 * @return array The array of unit settings fields.
	 */
	public function get_settings_fields() {
		return apply_filters( 'wpcw_unit_settings_fields', array(
			array(
				'type'  => 'heading',
				'key'   => 'units_section_heading',
				'title' => esc_html__( 'Units', 'wp-courseware' ),
				'desc'  => esc_html__( 'Below are settings that determine functionality for the units.', 'wp-courseware' ),
			),
			array(
				'type'     => 'radio',
				'title'    => esc_html__( 'Unit Comments', 'wp-courseware' ),
				'key'      => 'enable_unit_comments',
				'default'  => 'yes',
				'desc_tip' => esc_html__( 'If you enable comments, you will have the ability to disable comments for individual units.', 'wp-courseware' ),
				'options'  => array(
					'yes' => esc_html__( 'Yes', 'wp-courseware' ),
					'no'  => esc_html__( 'No', 'wp-courseware' ),
				),
			),
			array(
				'type'     => 'heading',
				'key'      => 'drip_feed_section_heading',
				'title'    => esc_html__( 'Drip Feed', 'wp-courseware' ),
				'desc_tip' => esc_html__( 'Below are settings that determine functionality for the drip feed.', 'wp-courseware' ),
			),
			array(
				'type'         => 'select',
				'key'          => 'cron_time_dripfeed',
				'title'        => esc_html__( 'Notifications Interval', 'wp-courseware' ),
				'placeholder'  => esc_html__( 'Select an Interval', 'wp-courseware' ),
				'desc_tip'     => esc_html__( 'How frequently should the system check if there are any notifications to send out to trainees? When a unit that is locked by a drip feed setting becomes available, the system sends them an email. This setting determines how frequently the system should check for possible notifications.', 'wp-courseware' ),
				'options'      => $this->get_drip_feed_intervals(),
				'blank_option' => esc_html__( 'Select Interval', 'wp-courseware' ),
				'default'      => 'twicedaily',
			),
		) );
	}

	/**
	 * Register Post Type Course Units.
	 *
	 * @since 4.3.0
	 */
	public function post_type() {
		register_post_type( $this->post_type_slug, apply_filters( 'wpcw_units_post_type_args', array(
			'labels'                => array(
				'name'               => __( 'Units', 'wp-courseware' ),
				'singular_name'      => __( 'Unit', 'wp-courseware' ),
				'all_items'          => __( 'All Units', 'wp-courseware' ),
				'new_item'           => __( 'New Unit', 'wp-courseware' ),
				'add_new'            => __( 'Add New', 'wp-courseware' ),
				'add_new_item'       => __( 'Add New Unit', 'wp-courseware' ),
				'edit_item'          => __( 'Edit Unit', 'wp-courseware' ),
				'view_item'          => __( 'View Unit', 'wp-courseware' ),
				'search_items'       => __( 'Search Units', 'wp-courseware' ),
				'not_found'          => __( 'No units found', 'wp-courseware' ),
				'not_found_in_trash' => __( 'No units found in trash', 'wp-courseware' ),
				'parent_item_colon'  => __( 'Parent Unit:', 'wp-courseware' ),
				'menu_name'          => __( 'Units', 'wp-courseware' ),
			),
			'public'                => true,
			'hierarchical'          => false,
			'show_ui'               => true,
			'show_in_nav_menus'     => false,
			'supports'              => array( 'title', 'editor', 'revisions' ),
			'has_archive'           => false,
			'rewrite'               => false,
			'query_var'             => true,
			'map_meta_cap'          => true,
			'menu_position'         => WPCW_MENU_POSITION + 1,
			'can_export'            => true,
			'show_in_rest'          => true,
			'capability_type'       => 'wpcw_course_unit',
			'rest_base'             => $this->post_type_slug,
			'rest_controller_class' => 'WP_REST_Posts_Controller',
		) ) );
	}

	/**
	 * Register Post Type Course Unit Updated Messages.
	 *
	 * @since 4.3.0
	 *
	 * @param array $messages The updated messages.
	 *
	 * @return array $messages The updated messages.
	 */
	public function post_type_updated_messages( $messages ) {
		global $post;

		$permalink = get_permalink( $post );

		$messages[ $this->post_type_slug ] = apply_filters( 'wpcw_units_post_type_updated_messages', array(
			0  => '', // Unused. Messages start at index 1.
			1  => sprintf( __( 'Unit updated. <a target="_blank" href="%s">View Unit</a>', 'wp-courseware' ), esc_url( $permalink ) ),
			2  => __( 'Custom field updated.', 'wp-courseware' ),
			3  => __( 'Custom field deleted.', 'wp-courseware' ),
			4  => __( 'Unit updated.', 'wp-courseware' ),
			/* translators: %s: date and time of the revision */
			5  => isset( $_GET['revision'] ) ? sprintf( __( 'Unit restored to revision from %s', 'wp-courseware' ), wp_post_revision_title( (int) $_GET['revision'], false ) ) : false,
			6  => sprintf( __( 'Unit published. <a href="%s">View Unit</a>', 'wp-courseware' ), esc_url( $permalink ) ),
			7  => __( 'Unit saved.', 'wp-courseware' ),
			8  => sprintf( __( 'Unit submitted. <a target="_blank" href="%s">Preview Unit</a>', 'wp-courseware' ), esc_url( add_query_arg( 'preview', 'true', $permalink ) ) ),
			9  => sprintf( __( 'Unit scheduled for: <strong>%1$s</strong>. <a target="_blank" href="%2$s">Preview Unit</a>', 'wp-courseware' ),
				// translators: Publish box date format, see http://php.net/date
				date_i18n( __( 'M j, Y @ G:i' ), strtotime( $post->post_date ) ), esc_url( $permalink ) ),
			10 => sprintf( __( 'Unit draft updated. <a target="_blank" href="%s">Preview Unit</a>', 'wp-courseware' ), esc_url( add_query_arg( 'preview', 'true', $permalink ) ) ),
		) );

		return $messages;
	}

	/**
	 * Exclude Post Type Course Units from Search.
	 *
	 * @since 4.3.0
	 *
	 * @param WP_Query $query The current query.
	 */
	public function post_type_exclude_from_search( $query ) {
		if ( is_admin() || ! $query->is_main_query() ) {
			return;
		}

		/**
		 * Filter: Units Exclude From Search
		 *
		 * @since 4.3.0
		 *
		 * @param bool True if excluded, False otherwise.
		 *
		 * @return bool True if you want to exclude units from search.
		 */
		if ( $query->is_search() && apply_filters( 'wpcw_units_exclude_from_search', true ) ) {
			$post_type_to_remove   = $this->post_type_slug;
			$searchable_post_types = get_post_types( array( 'exclude_from_search' => false ) );
			if ( is_array( $searchable_post_types ) && in_array( $post_type_to_remove, $searchable_post_types ) ) {
				unset( $searchable_post_types[ $post_type_to_remove ] );
				$query->set( 'post_type', $searchable_post_types );
			}
		}
	}

	/**
	 * Remove Unit Content on Post Type Archives.
	 *
	 * @since 4.3.0
	 */
	public function post_type_remove_content( $content ) {
		if ( ! is_main_query() ) {
			return $content;
		}

		/**
		 * Filter: Units Enable Taxonomy Archive Content.
		 *
		 * @since 4.3.0
		 *
		 * @param bool True to enable unit content to be displayed on archive pages. Default is False.
		 *
		 * @return bool The boolean paramater to enable or disable content on archive pages.
		 */
		if ( apply_filters( 'wpcw_units_enable_taxonomy_archive_content', false ) ) {
			return $content;
		}

		if ( is_archive() && ( is_tax( $this->taxonomy_tag_slug ) || is_tax( $this->taxonomy_category_slug ) ) ) {
			if ( ! wpcw_can_student_access_unit( get_the_ID() ) ) {
				$message = WPCW_UnitFrontend::message_createMessage_error( esc_html__( 'Sorry, but you\'re not allowed to access this unit.', 'wp-courseware' ) );

				/**
				 * Filter: Units Taxonomy Archive Content - No Access Message.
				 *
				 * @since 4.3.0
				 *
				 * @param string $message The no access message.
				 *
				 * @return string $message The no access message.
				 */
				$content = apply_filters( 'wpcw_units_taxonomy_archive_content_no_access_message', $message );
			}
		}

		return $content;
	}

	/**
	 * Register Course Unit Category.
	 *
	 * @since 4.3.0
	 */
	public function taxonomy_category() {
		register_taxonomy( $this->taxonomy_category_slug, array( $this->post_type_slug ), apply_filters( 'wpcw_units_category_args', array(
			'hierarchical'          => true,
			'public'                => true,
			'show_in_nav_menus'     => true,
			'show_ui'               => true,
			'show_admin_column'     => true,
			'query_var'             => true,
			'rewrite'               => array( 'slug' => 'unit-category' ),
			'capabilities'          => array(
				'manage_terms' => 'edit_posts',
				'edit_terms'   => 'edit_posts',
				'delete_terms' => 'edit_posts',
				'assign_terms' => 'edit_posts',
			),
			'labels'                => array(
				'name'                       => __( 'Categories', 'wp-courseware' ),
				'singular_name'              => _x( 'Category', 'taxonomy general name', 'wp-courseware' ),
				'search_items'               => __( 'Search categories', 'wp-courseware' ),
				'popular_items'              => __( 'Popular categories', 'wp-courseware' ),
				'all_items'                  => __( 'All categories', 'wp-courseware' ),
				'parent_item'                => __( 'Parent category', 'wp-courseware' ),
				'parent_item_colon'          => __( 'Parent category:', 'wp-courseware' ),
				'edit_item'                  => __( 'Edit category', 'wp-courseware' ),
				'update_item'                => __( 'Update category', 'wp-courseware' ),
				'add_new_item'               => __( 'New category', 'wp-courseware' ),
				'new_item_name'              => __( 'New category', 'wp-courseware' ),
				'separate_items_with_commas' => __( 'Separate categories with commas', 'wp-courseware' ),
				'add_or_remove_items'        => __( 'Add or remove categories', 'wp-courseware' ),
				'choose_from_most_used'      => __( 'Choose from the most used categories', 'wp-courseware' ),
				'not_found'                  => __( 'No categories found.', 'wp-courseware' ),
				'menu_name'                  => __( 'Categories', 'wp-courseware' ),
			),
			'show_in_rest'          => true,
			'rest_base'             => $this->taxonomy_category_slug,
			'rest_controller_class' => 'WP_REST_Terms_Controller',
		) ) );
	}

	/**
	 * Register Course Unit Tag.
	 *
	 * @since 4.3.0
	 */
	public function taxonomy_tag() {
		register_taxonomy( $this->taxonomy_tag_slug, array( $this->post_type_slug ), apply_filters( 'wpcw_units_tag_args', array(
			'hierarchical'          => false,
			'public'                => true,
			'show_in_nav_menus'     => true,
			'show_ui'               => true,
			'show_admin_column'     => true,
			'query_var'             => true,
			'rewrite'               => array( 'slug' => 'unit-tag' ),
			'capabilities'          => array(
				'manage_terms' => 'edit_posts',
				'edit_terms'   => 'edit_posts',
				'delete_terms' => 'edit_posts',
				'assign_terms' => 'edit_posts',
			),
			'labels'                => array(
				'name'                       => __( 'Tags', 'wp-courseware' ),
				'singular_name'              => _x( 'Tag', 'taxonomy general name', 'wp-courseware' ),
				'search_items'               => __( 'Search tags', 'wp-courseware' ),
				'popular_items'              => __( 'Popular tags', 'wp-courseware' ),
				'all_items'                  => __( 'All tags', 'wp-courseware' ),
				'parent_item'                => __( 'Parent tag', 'wp-courseware' ),
				'parent_item_colon'          => __( 'Parent tag:', 'wp-courseware' ),
				'edit_item'                  => __( 'Edit tag', 'wp-courseware' ),
				'update_item'                => __( 'Update tag', 'wp-courseware' ),
				'add_new_item'               => __( 'New tag', 'wp-courseware' ),
				'new_item_name'              => __( 'New tag', 'wp-courseware' ),
				'separate_items_with_commas' => __( 'Separate tags with commas', 'wp-courseware' ),
				'add_or_remove_items'        => __( 'Add or remove tags', 'wp-courseware' ),
				'choose_from_most_used'      => __( 'Choose from the most used tags', 'wp-courseware' ),
				'not_found'                  => __( 'No tags found.', 'wp-courseware' ),
				'menu_name'                  => __( 'Tags', 'wp-courseware' ),
			),
			'show_in_rest'          => true,
			'rest_base'             => $this->taxonomy_tag_slug,
			'rest_controller_class' => 'WP_REST_Terms_Controller',
		) ) );
	}

	/**
	 * Taxonomy Menu Fix.
	 *
	 * @since 4.3.0
	 */
	public function taxonomy_menu_fix() {
		global $parent_file, $submenu_file;

		$current_screen = get_current_screen();

		if ( empty( $current_screen->taxonomy ) || ! in_array( $current_screen->taxonomy, array( $this->taxonomy_tag_slug, $this->taxonomy_category_slug ) ) ) {
			return;
		}

		$parent_file  = 'wpcw';
		$submenu_file = esc_url( add_query_arg( array( 'post_type' => 'course_unit' ), 'edit.php' ) );
	}

	/**
	 * Taxonomy Action Buttons.
	 *
	 * @since 4.3.0
	 */
	public function taxonomy_action_buttons() {
		global $current_screen;

		if ( empty( $current_screen->taxonomy ) || ! in_array( $current_screen->taxonomy, array( $this->taxonomy_tag_slug, $this->taxonomy_category_slug ) ) ) {
			return;
		}

		if ( $this->taxonomy_tag_slug === $current_screen->taxonomy ) {
			$action_buttons = $this->get_taxonomy_tag_action_buttons();
		} else {
			$action_buttons = $this->get_taxonomy_category_action_buttons();
		}

		if ( empty( $action_buttons ) ) {
			return;
		}
		?>
        <script type="text/javascript">
			(function( $ ) {
				$( document ).ready( function() {
					$( '<?php echo $action_buttons; ?>' ).insertAfter( '.wrap h1.wp-heading-inline' );
				} )
			})( jQuery );
        </script>
		<?php

	}

	/**
	 * Add Icon to Taxonomy Title.
	 *
	 * @since 4.3.0
	 */
	public function taxonomy_title_icon() {
		global $current_screen;

		if ( empty( $current_screen->taxonomy ) || ! in_array( $current_screen->taxonomy, array( $this->taxonomy_tag_slug, $this->taxonomy_category_slug ) ) ) {
			return;
		}

		echo
			'<style type="text/css">
                .wrap h1.wp-heading-inline {
                    position: relative;
                    padding-top: 5px;
                    padding-left: 50px;
                    margin-right: 10px;
                }
                .wrap h1.wp-heading-inline:before {
                    background-image: url("' . wpcw_image_file( 'wp-courseware-icon.svg' ) . '");
                    background-size: 40px 40px;
                    content: "";
                    display: inline-block;
                    position: absolute;
                    top: -2px;
                    left: 0;
                    width: 40px;
                    height: 40px;
                }
            </style>';
	}

	/**
	 * Add Taxonomy Buttons.
	 *
	 * @since 4.3.0
	 *
	 * @param string $action_buttons The action buttons html.
	 *
	 * @return string $action_buttons The actions buttons html.
	 */
	public function add_taxonomy_buttons( $action_buttons ) {
		$taxonomy_buttons = sprintf(
			'<a class="page-title-action" href="%s">%s</a>',
			add_query_arg( array( 'taxonomy' => $this->taxonomy_category_slug ), admin_url( 'edit-tags.php' ) ),
			esc_html__( 'View Categories', 'wp-courseware' )
		);

		$taxonomy_buttons .= sprintf(
			'<a class="page-title-action" href="%s">%s</a>',
			add_query_arg( array( 'taxonomy' => $this->taxonomy_tag_slug ), admin_url( 'edit-tags.php' ) ),
			esc_html__( 'View Tags', 'wp-courseware' )
		);

		return $action_buttons . $taxonomy_buttons;
	}

	/**
	 * Get Taxonomy Tag Action Buttons.
	 *
	 * @since 4.3.0
	 *
	 * @return array $action_buttons The action buttons that will go on the taxonomy pages.
	 */
	public function get_taxonomy_category_action_buttons() {
		$action_buttons = sprintf(
			'<a class="page-title-action" href="%s">%s</a>',
			add_query_arg( array( 'post_type' => $this->post_type_slug ), admin_url( 'edit.php' ) ),
			esc_html__( 'View Units', 'wp-courseware' )
		);

		$action_buttons .= sprintf(
			'<a class="page-title-action" href="%s">%s</a>',
			add_query_arg( array( 'taxonomy' => $this->taxonomy_tag_slug ), admin_url( 'edit-tags.php' ) ),
			esc_html__( 'View Tags', 'wp-courseware' )
		);

		return $action_buttons;
	}

	/**
	 * Get Taxonomy Tag Action Buttons.
	 *
	 * @since 4.3.0
	 *
	 * @return string $action_buttons The action buttons that will go on the taxonomy tag page.
	 */
	public function get_taxonomy_tag_action_buttons() {
		$action_buttons = sprintf(
			'<a class="page-title-action" href="%s">%s</a>',
			add_query_arg( array( 'post_type' => $this->post_type_slug ), admin_url( 'edit.php' ) ),
			esc_html__( 'View Units', 'wp-courseware' )
		);

		$action_buttons .= sprintf(
			'<a class="page-title-action" href="%s">%s</a>',
			add_query_arg( array( 'taxonomy' => $this->taxonomy_category_slug ), admin_url( 'edit-tags.php' ) ),
			esc_html__( 'View Categories', 'wp-courseware' )
		);

		return $action_buttons;
	}

	/**
	 * Register Course Unit Comments Support.
	 *
	 * This will allow comments to be supported on units
	 * if the setting is set otherwise it will remove support.
	 *
	 * @since 4.3.0
	 */
	public function post_type_comments_support() {
		$unit_comments_setting = wpcw_get_setting( 'enable_unit_comments' );

		if ( 'enable_comments' === $unit_comments_setting || 'yes' === $unit_comments_setting ) {
			add_post_type_support( 'course_unit', 'comments' );
		} else {
			remove_post_type_support( 'course_unit', 'comments' );
		}
	}

	/**
	 * Course Units Rewrites Support.
	 *
	 * Creates a custom permalink structure
	 * to support Units & Modules.
	 *
	 * @since 4.3.0
	 */
	public function post_type_rewrites_support() {
		global $wp_rewrite;

		// Permalink Structure for Units.
		$unit_permalink_structure = apply_filters( 'wpcw_units_permalink_structure', '/%module_number%/%course_unit%/' );

		// Rewrite Module.
		$module_rewrite_tag   = apply_filters( 'wpcw_units_rewrite_tag_module', '%module_number%' );
		$module_rewrite_regex = apply_filters( 'wpcw_permalinks_urlmatch_module', '(module-[^/]+)' );
		$module_rewrite_regex = apply_filters( 'wpcw_units_rewrite_regex_module', $module_rewrite_regex );
		$module_rewrite_query = apply_filters( 'wpcw_units_rewrite_query_module', 'module_number=' );

		// Rewrite Unit.
		$unit_rewrite_tag   = apply_filters( 'wpcw_units_rewrite_tag_unit', '%course_unit%' );
		$unit_rewrite_regex = apply_filters( 'wpcw_permalinks_urlmatch_unit', '([^/]+)' );
		$unit_rewrite_regex = apply_filters( 'wpcw_units_rewrite_regex_unit', $unit_rewrite_regex );
		$unit_rewrite_query = apply_filters( 'wpcw_units_rewrite_query_unit', 'course_unit=' );

		// Handle Module and Unit Tags.
		$wp_rewrite->add_rewrite_tag( $module_rewrite_tag, $module_rewrite_regex, $module_rewrite_query );
		$wp_rewrite->add_rewrite_tag( $unit_rewrite_tag, $unit_rewrite_regex, $unit_rewrite_query );

		// Add Permalink Structure for Units.
		$wp_rewrite->add_permastruct( 'course_unit', $unit_permalink_structure, array( 'ep_mask' => EP_PAGES ) );

		// Get Flushed Rewrite Rules Flag
		$unit_rules_flushed = get_option( 'wpcw_flush_rules' );

		// Flush the Permalink Structure Rules - Should only happen once.
		if ( ! $unit_rules_flushed ) {
			update_option( 'wpcw_flush_rules', 'done' );
			$wp_rewrite->flush_rules();
		}
	}

	/**
	 * Get Drip Feed Intervals.
	 *
	 * @since 4.1.0
	 *
	 * @return array The drip feed intervals.
	 */
	public function get_drip_feed_intervals() {
		return apply_filters( 'wpcw_units_drip_feed_intervals', array(
			'never'      => esc_html__( 'Never check', 'wp-courseware' ),
			'hourly'     => esc_html__( 'Every hour', 'wp-courseware' ),
			'twicedaily' => esc_html__( 'Twice a day', 'wp-courseware' ),
			'daily'      => esc_html__( 'Daily', 'wp-courseware' ),
		) );
	}

	/**
	 * Update Drip Feed Cron Interval.
	 *
	 * @since 4.1.0
	 *
	 * @param array $settings The settings array.
	 */
	public function update_drip_feed_cron( $settings ) {
		wpcw()->settings->set_settings( $settings );

		$drip_feed = wpcw()->settings->get_setting( 'cron_time_dripfeed' );

		WPCW_queue_dripfeed::installNotificationHook_dripfeed( $drip_feed );
	}

	/**
	 * Get Units Filter By Dropdown.
	 *
	 * @since 4.1.0
	 *
	 * @param array $args The dropdown args. Can be used to customize the dropdown.
	 */
	public function get_units_filter_by_dropdown( array $args = array() ) {
		$defaults = array(
			'label'       => false,
			'placeholder' => esc_html__( 'All Units', 'wp-courseware' ),
			'name'        => 'unit_id',
			'classes'     => array(),
			'selected'    => isset( $_GET['unit_id'] ) ? absint( $_GET['unit_id'] ) : 0,
			'orderby'     => 'post_title',
		);

		$args = wp_parse_args( $args, $defaults );

		$form = '';

		// Check if admin
		if ( ! current_user_can( 'manage_wpcw_settings' ) ) {
			$args['post_author'] = get_current_user_id();
		}

		$units = new WP_Query( array(
			'post_type'              => 'course_unit',
			'posts_per_page'         => 1000,
			'update_post_meta_cache' => false,
			'update_post_term_cache' => false,
			'no_found_rows'          => true,
		) );

		if ( $units->have_posts() ) {
			$classes = ! empty( $args['classes'] ) ? sprintf( ' class="%s"', implode( ' ', $args['classes'] ) ) : '';

			if ( $args['label'] ) {
				$form .= sprintf( '<label for="%s">%s:</label> ', $args['name'], $args['label'] );
			}

			$form .= sprintf(
				'<select name="%s" placeholder="%s" %s>',
				$args['name'],
				$args['placeholder'],
				$classes
			);

			$form .= sprintf( '<option value="">%s</option>', $args['placeholder'] );

			while ( $units->have_posts() ) {
				$units->the_post();
				$selected = ( get_the_ID() === absint( $args['selected'] ) ) ? ' selected="selected"' : '';

				$form .= sprintf( '<option value="%s" %s>%s</option>', get_the_ID(), $selected, get_the_title() );
			}
			wp_reset_postdata();
			wp_reset_query();

			$form .= '</select>';
		}

		return $form;
	}

	/**
	 * Get Courses Filter Dropdown.
	 *
	 * @since 4.3.0
	 *
	 * @return string The html for the courses filter dropdown.
	 */
	public function get_units_filter_dropdown() {
		$unit_id = isset( $_GET['unit_id'] ) ? absint( $_GET['unit_id'] ) : 0;

		ob_start();

		printf( '<span class="wpcw-filter-wrapper">' );
		printf( '<select id="wpcw-units-filter" class="select-field-select2-filter" name="unit_id" data-placeholder="%s">', esc_html__( 'All Units', 'wp-courseware' ) );

		if ( $unit_id ) {
			printf( '<option value="%s">%s</option>', $unit_id, get_the_title( $unit_id ) );
		}

		printf( '</select>' );
		printf( '</span>' );

		return ob_get_clean();
	}

	/**
	 * Register Units Api Endpoints.
	 *
	 * @since 4.3.0
	 *
	 * @param array $endpoints The endpoints to filter.
	 * @param Api The api object.
	 *
	 * @return array $endpoints The modified array of endpoints.
	 */
	public function register_api_endpoints( $endpoints, Api $api ) {
		$endpoints[] = array( 'endpoint' => 'units', 'method' => 'GET', 'callback' => array( $this, 'api_get_units' ) );
		$endpoints[] = array( 'endpoint' => 'units-filtered', 'method' => 'GET', 'callback' => array( $this, 'api_get_units_filtered' ) );

		return $endpoints;
	}

	/**
	 * Api: Get Units.
	 *
	 * @since 4.3.0
	 *
	 * @param object \WP_REST_Request The api request.
	 *
	 * @return object \WP_REST_Response The api response.
	 */
	public function api_get_units( WP_REST_Request $request ) {
		$search = $request->get_param( 'search' );
		$number = $request->get_param( 'number' );
		$order  = $request->get_param( 'order' );
		$author = $request->get_param( 'author' );

		if ( ! $search ) {
			$search = '';
		}

		if ( ! $number ) {
			$number = 1000;
		}

		if ( ! $order ) {
			$order = 'DESC';
		}

		if ( ! $author ) {
			$author = '';
		}

		$results    = array();
		$query_args = array(
			'post_type'              => 'course_unit',
			'posts_per_page'         => $number,
			'update_post_meta_cache' => false,
			'update_post_term_cache' => false,
			'no_found_rows'          => true,
			's'                      => $search,
			'order'                  => $order,
			'author'                 => $author,
		);

		$units = new WP_Query( $query_args );

		if ( $units->have_posts() ) {
			while ( $units->have_posts() ) {
				$units->the_post();
				$results[] = array(
					'id'    => get_the_ID(),
					'title' => get_the_title(),
				);
			}

			wp_reset_postdata();
			wp_reset_query();
		}

		return rest_ensure_response( array( 'units' => $results ) );
	}

	/**
	 * Api: Get Units Fifltered.
	 *
	 * @since 4.3.0
	 *
	 * @param object \WP_REST_Request The api request.
	 *
	 * @return object \WP_REST_Response The api response.
	 */
	public function api_get_units_filtered( WP_REST_Request $request ) {
		$search = $request->get_param( 'search' );
		$number = $request->get_param( 'number' );
		$order  = $request->get_param( 'order' );
		$author = $request->get_param( 'author' );

		if ( ! $search ) {
			$search = '';
		}

		if ( ! $number ) {
			$number = 100;
		}

		if ( ! $order ) {
			$order = 'DESC';
		}

		$results    = array();
		$query_args = array(
			'post_type'              => 'course_unit',
			'posts_per_page'         => $number,
			'update_post_meta_cache' => false,
			'update_post_term_cache' => false,
			'no_found_rows'          => true,
			's'                      => $search,
			'order'                  => $order,
		);

		if ( is_user_logged_in() && ! $author && ! current_user_can( 'manage_wpcw_settings' ) ) {
			$query_args['author'] = get_current_user_id();
		}

		$units = new WP_Query( $query_args );

		if ( $units->have_posts() ) {
			while ( $units->have_posts() ) {
				$units->the_post();
				$results[] = array(
					'id'    => get_the_ID(),
					'title' => html_entity_decode( get_the_title(), ENT_QUOTES, get_bloginfo( 'charset' ) ),
				);
			}

			wp_reset_postdata();
			wp_reset_query();
		}

		return rest_ensure_response( array( 'units' => $results ) );
	}
}