<?php
/**
 * WP Courseware Modules Controller.
 *
 * @package WPCW
 * @subpackage Controllers
 * @since 4.3.0
 */
namespace WPCW\Controllers;

use WPCW\Database\DB_Modules;
use WPCW\Models\Module;

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Class Modules.
 *
 * @since 4.1.0
 */
class Modules extends Controller {

	/**
	 * @var DB_Modules The modules db object.
	 * @since 4.3.0
	 */
	protected $db;

	/**
	 * Modules constructor.
	 *
	 * @since 4.3.0
	 */
	public function __construct() {
		$this->db = new DB_Modules();
	}

	/**
	 * Modules Load.
	 *
	 * @since 4.3.0
	 */
	public function load() { /* Do Nothing for now */ }

	/**
	 * Get Module by Id.
	 *
	 * @since 4.1.0
	 *
	 * @param int $module_id The module id.
	 *
	 * @return bool|Module The module object.
	 */
	public function get_module( $module_id ) {
		if ( 0 === absint( $module_id ) ) {
			return false;
		}

		$row = $this->db->get( $module_id );

		if ( ! $row ) {
			return false;
		}

		return new Module( $row );
	}

	/**
	 * Get Modules.
	 *
	 * @param array $args Optional. Valid Query Arguments.
	 * @param bool $raw Optional. Retrieve the raw database data.
	 *
	 * @return array Array of module objects.
	 */
	public function get_modules( $args = array(), $raw = false ) {
		$modules = array();
		$results = $this->db->get_modules( $args );

		if ( $raw ) {
			return $results;
		}

		foreach ( $results as $result ) {
			$modules[] = new Module( $result );
		}

		return $modules;
	}

	/**
	 * Get Number of Modules.
	 *
	 * @since 4.1.0
	 *
	 * @param array $args Optional. Valid Query Arguments.
	 *
	 * @return int The number of modules.
	 */
	public function get_modules_count( $args = array() ) {
		return $this->db->get_modules( $args, true );
	}

	/**
	 * Get Module Course Title.
	 *
	 * @since 4.1.0
	 *
	 * @param int $course_id The course id.
	 *
	 * @return string $course_title The course title.
	 */
	public function get_module_course_title( $course_id ) {
		return $this->db->get_module_course_title( $course_id );
	}

	/**
	 * Delete Module.
	 *
	 * @since 4.1.0
	 *
	 * @param int $id The module id.
	 */
	public function delete_module( $module_id ) {
		if ( ! is_admin() || ! current_user_can( 'view_wpcw_courses' ) ) {
			return false;
		}

		if ( $module = $this->get_module( $module_id ) ) {
			if ( ! current_user_can( 'manage_wpcw_settings' ) ) {
				if ( $module->get_module_author() !== get_current_user_id() ) {
					return false;
				}
			}

			if ( WPCW_modules_deleteModule( $module ) ) {
				return $module;
			}
		}

		return false;
	}
}