<?php
/**
 * WP Courseware Courses Controller.
 *
 * @package WPCW
 * @subpackage Controllers
 * @since 4.1.0
 */
namespace WPCW\Controllers;

use WPCW\Database\DB_Courses;
use WPCW\Core\Api;
use WPCW\Models\Course;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Class Courses.
 *
 * @since 4.3.0
 */
class Courses extends Controller {

	/**
	 * @var DB_Courses The courses database.
	 * @since 4.3.0
	 */
	protected $db;

	/**
	 * Courses constructor.
	 *
	 * @since 4.3.0
	 */
	public function __construct() {
		$this->db = new DB_Courses();
	}

	/**
	 * Load Courses Controller.
	 *
	 * @since 4.1.0
	 */
	public function load() {
		add_filter( 'wpcw_api_endoints', array( $this, 'register_api_endpoints' ), 10, 2 );
		add_filter( 'wpcw_course_automatic_enrollment_disable', array( $this, 'maybe_disable_autommatic_enrollment' ), 10, 2 );
	}

	/**
	 * Get Course Settings Fields.
	 *
	 * @since 4.3.0
	 *
	 * @return array The array of course settings fields.
	 */
	public function get_settings_fields() {
		return apply_filters( 'wpcw_course_settings_fields', array(
			array(
				'type'  => 'heading',
				'key'   => 'courses_heading',
				'title' => esc_html__( 'Courses', 'wp-courseware' ),
				'desc'  => esc_html__( 'Below are settings that control how courses should function.', 'wp-courseware' ),
			),
			array(
				'type'    => 'hidden',
				'key'     => 'course_enrollment_method',
				'default' => 'sync',
			),
			array(
				'type'     => 'page',
				'key'      => 'courses_page',
				'title'    => esc_html__( 'Courses Page', 'wp-courseware' ),
				'desc_tip' => esc_html__( 'The main course index page.', 'wp-courseware' ),
				'default'  => '',
			),
			array(
				'type'  => 'heading',
				'key'   => 'course_outlines_section_heading',
				'title' => esc_html__( 'Course Outline', 'wp-courseware' ),
				'desc'  => esc_html__( 'Below are settings related to the display of a course outline.', 'wp-courseware' ),
			),
			array(
				'type'      => 'affiliates',
				'key'       => 'affiliates',
				'component' => true,
				'views'     => array( 'settings/settings-field-affiliates' ),
				'settings'  => array(
					array(
						'key'     => 'show_powered_by',
						'type'    => 'radio',
						'default' => 'yes',
					),
					array(
						'key'     => 'affiliate_id',
						'type'    => 'text',
						'default' => '',
					),
				),
			),
		) );
	}

	/**
	 * Maybe Disable Automatic Enrollment.
	 *
	 * If a course is paid, either a one-time or subscription
	 * then disabled the automatic enrollment that happens
	 * when a user is registered.
	 *
	 * @since 4.3.0
	 *
	 * @param bool $disable Boolean to disable automatic enrollment. Default is false.
	 * @param int $course_id The course id.
	 *
	 * @return bool $disable Boolean value to disable automatic enrollment.
	 */
	public function maybe_disable_autommatic_enrollment( $disable, $course_id ) {
		$payments_type = $this->get_course_payments_type( $course_id );

		if ( $payments_type && 'free' !== $payments_type ) {
			$disable = true;
		}

		return $disable;
	}

	/**
	 * Register Course Api Endpoints.
	 *
	 * @since 4.1.0
	 *
	 * @param array $endpoints The endpoints to filter.
	 * @param Api The api object.
	 *
	 * @return array $endpoints The modified array of endpoints.
	 */
	public function register_api_endpoints( $endpoints, Api $api ) {
		$endpoints[] = array( 'endpoint' => 'courses', 'method' => 'GET', 'callback' => array( $this, 'api_get_courses' ) );
		$endpoints[] = array( 'endpoint' => 'courses-filtered', 'method' => 'GET', 'callback' => array( $this, 'api_get_courses_filtered' ) );

		return $endpoints;
	}

	/**
	 * Api: Get Courses.
	 *
	 * @since 4.1.0
	 *
	 * @param object \WP_REST_Request The api request.
	 *
	 * @return object \WP_REST_Response The api response.
	 */
	public function api_get_courses( WP_REST_Request $request ) {
		$search = $request->get_param( 'search' );
		$number = $request->get_param( 'number' );
		$order  = $request->get_param( 'order' );
		$author = $request->get_param( 'author' );

		if ( ! $search ) {
			$search = '';
		}

		if ( ! $number ) {
			$number = 1000;
		}

		if ( ! $order ) {
			$order = 'DESC';
		}

		if ( ! $author ) {
			$author = '';
		}

		$results    = array();
		$query_args = array( 'search' => $search, 'number' => $number, 'order' => $order, 'course_author' => $author );

		$courses = $this->get_courses( $query_args );
		$count   = $this->get_courses_count( $query_args );

		foreach ( $courses as $course ) {
			if ( ! $course instanceof Course ) {
				continue;
			}

			$results[] = array(
				'id'        => $course->get_course_id(),
				'title'     => $course->get_course_title(),
				'price'     => $course->get_payments_price(),
				'recurring' => ( 'subscription' === $course->get_payments_type() ) ? true : false,
			);
		}

		return rest_ensure_response( array( 'courses' => $results ) );
	}

	/**
	 * Api: Get Courses Filtered.
	 *
	 * @since 4.1.0
	 *
	 * @param object \WP_REST_Request The api request.
	 *
	 * @return object \WP_REST_Response The api response.
	 */
	public function api_get_courses_filtered( WP_REST_Request $request ) {
		$search = $request->get_param( 'search' );
		$number = $request->get_param( 'number' );
		$order  = $request->get_param( 'order' );
		$author = $request->get_param( 'author' );

		if ( ! $search ) {
			$search = '';
		}

		if ( ! $number ) {
			$number = 100;
		}

		if ( ! $order ) {
			$order = 'DESC';
		}

		if ( ! $author ) {
			$author = '';
		}

		$results    = array();
		$query_args = array(
			'search' => $search,
			'number' => $number,
			'order'  => $order,
		);

		// Check if admin
		if ( is_user_logged_in() && ! $author && ! current_user_can( 'manage_wpcw_settings' ) ) {
			$query_args['course_author'] = get_current_user_id();
		}

		$courses = $this->get_courses( $query_args );
		$count   = $this->get_courses_count( $query_args );

		foreach ( $courses as $course ) {
			if ( ! $course instanceof Course ) {
				continue;
			}

			$results[] = array(
				'id'        => $course->get_course_id(),
				'title'     => $course->get_course_title(),
				'price'     => $course->get_payments_price(),
				'recurring' => ( 'subscription' === $course->get_payments_type() ) ? true : false,
			);
		}

		return rest_ensure_response( array( 'courses' => $results ) );
	}

	/**
	 * Get Course Payments Type.
	 *
	 * @since 4.3.0
	 *
	 * @param int $course_id The course id.
	 */
	public function get_course_payments_type( $course_id = 0 ) {
		if ( ! $course_id ) {
			return '';
		}

		return $this->db->get_column( 'payments_type', $course_id );
	}

	/**
	 * Get Courses.
	 *
	 * @param array $args Optional. Valid Query Arguments.
	 * @param bool $raw Optional. Retrieve the raw database data.
	 *
	 * @return array Array of course objects.
	 */
	public function get_courses( $args = array(), $raw = false ) {
		$courses = array();
		$results = $this->db->get_courses( $args );

		if ( $raw ) {
			return $results;
		}

		foreach ( $results as $result ) {
			$courses[] = new Course( $result );
		}

		return $courses;
	}

	/**
	 * Get Number of Courses.
	 *
	 * @since 4.1.0
	 *
	 * @param array $args Optional. Valid Query Arguments.
	 *
	 * @return int The number of courses.
	 */
	public function get_courses_count( $args = array() ) {
		return $this->db->get_courses( $args, true );
	}

	/**
	 * Get Courses by Student.
	 *
	 * @since 4.3.0
	 *
	 * @param int $student_id The student id.
	 *
	 * @return array An array of student courses.
	 */
	public function get_courses_by_student( $student_id ) {
		global $wpdb, $wpcwdb;
		$courses = array();

		if ( ! $student_id ) {
			return $courses;
		}

		$results = $wpdb->get_results( $wpdb->prepare(
			"SELECT * FROM $wpcwdb->user_courses uc 
			 LEFT JOIN  $wpcwdb->courses c ON c.course_id = uc.course_id 
			 WHERE user_id = %d 
			 ORDER BY course_title ASC",
			$student_id
		) );

		foreach ( $results as $result ) {
			$courses[] = new Course( $result );
		}

		return $courses;
	}

	/**
	 * Delete Course.
	 *
	 * @since 4.1.0
	 *
	 * @param int $id The module id.
	 */
	public function delete_course( $id, $method = 'complete' ) {
		if ( ! is_admin() || ! current_user_can( 'view_wpcw_courses' ) ) {
			return false;
		}

		$course = new Course( $id );

		if ( $course->get_course_id() ) {
			if ( ! current_user_can( 'manage_wpcw_settings' ) ) {
				if ( $course->get_course_author() !== get_current_user_id() ) {
					return false;
				}
			}

			if ( WPCW_modules_deleteCourse( $course, $method ) ) {
				return $course;
			}
		}

		return false;
	}

	/**
	 * Get Course Dropdown Html.
	 *
	 * @since 4.1.0
	 *
	 * @param array $courses The courses in which will populate the dropdown.
	 * @param string $bulk For the bulk dropdown.
	 *
	 * @return string
	 */
	public function get_courses_reset_dropdown( array $courses, $bulk = false, $query = true ) {
		if ( $query ) {
			$results = $this->get_courses( array(
				'course_id' => $courses,
				'fields'    => array( 'course_id', 'course_title' ),
				'orderby'   => 'course_title',
			) );
		} else {
			$results = $courses;
		}

		$count         = 1;
		$blank_message = esc_html__( 'No courses available.', 'wp-courseware' );

		if ( $bulk ) {
			$course_dropdown = array( '' => esc_html__( 'Reset student(s) to beginning of...', 'wp-courseware' ) );
		} else {
			$course_dropdown = array( '' => esc_html__( 'Reset student to beginning of...', 'wp-courseware' ) );
		}

		$current_user = wp_get_current_user();

		if ( ! empty( $results ) ) {
			foreach ( $results as $course ) {
				if ( $query ) {
					if ( ! $course instanceof Course ) {
						continue;
					}

					$course_id     = $course->get_course_id();
					$course_title  = $course->get_course_title();
					$course_author = $course->get_course_author();
				} else {
					$course_id     = $course['id'];
					$course_title  = $course['title'];
					$course_author = $course['author'];
				}

				if ( empty( $course_id ) || empty( $course_title ) ) {
					continue;
				}

				if ( ! user_can( $current_user, 'manage_wpcw_settings' ) ) {
					if ( $current_user->ID !== absint( $course_author ) ) {
						continue;
					}
				}

				$course_dropdown[ 'course_' . $course_id ] = $course_title;

				$course_modules = WPCW_courses_getModuleDetailsList( $course_id );

				if ( empty( $course_modules ) ) {
					$count++;
					continue;
				}

				foreach ( $course_modules as $module_id => $module ) {
					$units = WPCW_units_getListOfUnits( $module_id );

					if ( ! empty( $units ) ) {
						$course_dropdown[ 'module_' . $module_id ] = sprintf( '&nbsp;&nbsp;- %s %d: %s',
							__( 'Module', 'wp-courseware' ),
							$module->module_number,
							$module->module_title
						);

						foreach ( $units as $unit_id => $unit ) {
							$course_dropdown[ 'unit_' . $unit_id ] = sprintf( '&nbsp;&nbsp;-- %s %d: %s',
								__( 'Unit', 'wp-courseware' ),
								$unit->unit_meta->unit_number,
								$unit->post_title
							);
						}
					}
				}

				if ( $count !== count( $courses ) ) {
					$padding = str_pad( false, $count++, ' ' );

					$course_dropdown[ $padding ] = '&nbsp;';
				}
			}
		}

		// No Courses.
		if ( count( $course_dropdown ) === 0 ) {
			$course_dropdown[' '] = $blankMessage;
		}

		if ( $bulk ) {
			$field_name    = 'wpcw_user_progress_reset_point_bulk';
			$field_classes = 'wpcw_user_progress_reset_select';
		} else {
			$field_name    = 'wpcw_user_progress_reset_point_single';
			$field_classes = 'wpcw_user_progress_reset_select wpcw_user_progress_reset_point_single';
		}

		return WPCW_forms_createDropdown( $field_name, $course_dropdown, false, false, $field_classes );
	}

	/**
	 * Get Courses Dropdown.
	 *
	 * @since 4.1.0
	 *
	 * @param array $args The dropdown args. Can be used to customize the dropdown.
	 */
	public function get_courses_switch_dropdown( array $args = array() ) {
		$defaults = array(
			'label'       => esc_html__( 'Switch Courses', 'wp-courseware' ),
			'placeholder' => esc_html__( 'Switch Courses', 'wp-courseware' ),
			'name'        => 'course_id',
			'classes'     => array(),
			'selected'    => 0,
			'action'      => '',
			'page'        => '',
			'fields'      => array( 'course_id', 'course_title' ),
			'orderby'     => 'course_title',
		);

		$args = wp_parse_args( $args, $defaults );

		$form = '';

		if ( ! current_user_can( 'manage_wpcw_settings' ) ) {
			$args['course_author'] = get_current_user_id();
		}

		$results = $this->get_courses( $args );

		if ( ! empty( $results ) && count( $results ) > 1 ) {
			$classes = ! empty( $args['classes'] ) ? sprintf( ' class="%s"', implode( ' ', $args['classes'] ) ) : '';

			$form = sprintf( '<form id="wpcw-courses-switch-dropdown" class="wpcw-courses-switch-dropdown" method="get" action="%s">', $args['action'] );
			if ( $args['label'] ) {
				$form .= sprintf( '<label for="%s">%s:</label> ', $args['name'], $args['label'] );
			}

			if ( $args['page'] ) {
				$form .= sprintf( '<input name="page" type="hidden" value="%s" />', $args['page'] );
			}

			$form .= sprintf(
				'<select name="%s" placeholder="%s" %s>',
				$args['name'],
				$args['placeholder'],
				$classes
			);

			$form .= sprintf( '<option value="">%s</option>', $args['placeholder'] );

			foreach ( $results as $course ) {
				if ( ! $course instanceof Course ) {
					continue;
				}

				$selected = ( $course->get_course_id() === absint( $args['selected'] ) ) ? ' selected="selected"' : '';

				$form .= sprintf( '<option value="%s" %s>%s</option>', $course->get_course_id(), $selected, $course->get_course_title() );
			}

			$form .= '</select>';
			$form .= '</form>';
		}

		return $form;
	}

	/**
	 * Get Courses Filter By Dropdown.
	 *
	 * @since 4.1.0
	 *
	 * @param array $args The dropdown args. Can be used to customize the dropdown.
	 *
	 * @return stirng $form The filter by courses dropdown form.
	 */
	public function get_courses_filter_by_dropdown( array $args = array() ) {
		$defaults = array(
			'label'       => false,
			'placeholder' => esc_html__( 'All Courses', 'wp-courseware' ),
			'name'        => 'course_id',
			'classes'     => array(),
			'selected'    => isset( $_GET['course_id'] ) ? absint( $_GET['course_id'] ) : 0,
			'fields'      => array( 'course_id', 'course_title' ),
			'orderby'     => 'course_title',
			'order'       => 'ASC',
		);

		$args = wp_parse_args( $args, $defaults );

		$form = '';

		// Check if admin
		if ( ! current_user_can( 'manage_wpcw_settings' ) ) {
			$args['course_author'] = get_current_user_id();
		}

		$results = $this->get_courses( $args );

		if ( ! empty( $results ) && count( $results ) > 0 ) {
			$classes = ! empty( $args['classes'] ) ? sprintf( ' class="%s"', implode( ' ', $args['classes'] ) ) : '';

			if ( $args['label'] ) {
				$form .= sprintf( '<label for="%s">%s:</label> ', $args['name'], $args['label'] );
			}

			$form .= sprintf(
				'<select name="%s" placeholder="%s" %s>',
				$args['name'],
				$args['placeholder'],
				$classes
			);

			$form .= sprintf( '<option value="">%s</option>', $args['placeholder'] );

			foreach ( $results as $course ) {
				if ( ! $course instanceof Course ) {
					continue;
				}

				$selected = ( $course->get_course_id() === absint( $args['selected'] ) ) ? ' selected="selected"' : '';

				$form .= sprintf( '<option value="%s" %s>%s</option>', $course->get_course_id(), $selected, $course->get_course_title() );
			}

			$form .= '</select>';
		}

		return $form;
	}

	/**
	 * Get Courses Filter Dropdown.
	 *
	 * @since 4.3.0
	 *
	 * @return string The html for the courses filter dropdown.
	 */
	public function get_courses_filter_dropdown() {
		$course_id = isset( $_GET['course_id'] ) ? absint( $_GET['course_id'] ) : 0;

		ob_start();

		printf( '<span class="wpcw-filter-wrapper">' );
		printf( '<select id="wpcw-courses-filter" class="select-field-select2-filter" name="course_id" data-placeholder="%s">', esc_html__( 'All Courses', 'wp-courseware' ) );

		if ( $course_id && ( $course = new Course( $course_id ) ) ) {
			printf( '<option value="%s">%s</option>', $course->get_id(), $course->get_course_title() );
		}

		printf( '</select>' );
		printf( '</span>' );

		return ob_get_clean();
	}

	/**
	 * Get Course Quizzes Count that need grading.
	 *
	 * @since 4.1.0
	 *
	 * @return int $id The grading course id.
	 */
	public function get_course_quizzes_that_need_grading_count( $id = 0 ) {
		if ( ! is_admin() || ! current_user_can( 'manage_wpcw_settings' ) ) {
			return;
		}

		$grading_show = get_user_meta( get_current_user_id(), 'wpcw_course_dashboard_quiz_notification_hide', true );

		if ( ! $grading_show || 'hide' === $grading_show ) {
			return false;
		}

		if ( $id ) {
			global $wpdb, $wpcwdb;

			$course_quizzes = $wpdb->get_col( $wpdb->prepare( "
				SELECT quiz_id
				FROM $wpcwdb->quiz
				WHERE parent_course_id = %d
			", $id ) );

			$need_grading     = 0;
			$need_manual_help = 0;

			if ( ! empty( $course_quizzes ) ) {
				$ids = '(' . implode( ',', $course_quizzes ) . ')';

				$need_grading = $wpdb->get_var( "
					SELECT COUNT(*)
					FROM $wpcwdb->user_progress_quiz
					WHERE quiz_id IN $ids
					  AND quiz_needs_marking > 0
					  AND quiz_is_latest = 'latest'
				" );

				$need_manual_help = $wpdb->get_var( "
					SELECT COUNT(*)
					FROM $wpcwdb->user_progress_quiz
					WHERE quiz_id IN $ids
					  AND quiz_next_step_type = 'quiz_fail_no_retakes'
					  AND quiz_is_latest = 'latest'
				" );
			}

			return absint( $need_grading ) + absint( $need_manual_help );
		}

		return WPCW_quizzes_getCoursesNeedingAttentionCount();
	}

	/**
	 * Create Courses Page.
	 *
	 * @since 4.3.0
	 *
	 * @return bool|int $page_id The page to be created.
	 */
	public function create_courses_page() {
		return wp_insert_post(
			array(
				'post_title'     => esc_html__( 'Courses', 'wp-courseware' ),
				'post_content'   => '[wpcw_courses]',
				'post_status'    => 'publish',
				'post_author'    => 1,
				'post_type'      => 'page',
				'comment_status' => 'closed',
				'menu_order'     => 10,
			)
		);
	}

	/**
	 * Get Payment Types.
	 *
	 * @since 4.3.0
	 *
	 * @return array The array of payment types.
	 */
	public function get_payment_types() {
		return apply_filters( 'wpcw_payment_types', array(
			'free'         => __( '<strong>Free</strong> - No payment to enroll in course.', 'wp-courseware' ),
			'one-time'     => __( '<strong>One-Time Purchase</strong> - A single payment to enroll in course.', 'wp-courseware' ),
			'subscription' => __( '<strong>Subscription</strong> - Monthly or Annual billing interval for continued enrollment in course.', 'wp-courseware' ),
		) );
	}

	/**
	 * Get Payment Intervals.
	 *
	 * @since 4.3.0
	 *
	 * @return array The array of payment intervals.
	 */
	public function get_payment_intervals() {
		$intervals = array();
		$periods   = wpcw()->subscriptions->get_periods();

		foreach ( $periods as $period => $period_label ) {
			/* translators: %1$s - The subscription period capitalized. %2$s - The subscription period lowercase. */
			$intervals[ $period ] = sprintf( __( '<strong>%1$s</strong> - Subscription is billed %2$s until cancelled.', 'wp-courseware' ), $period_label, strtolower( $period_label ) );
		}

		return apply_filters( 'wpcw_payment_intervals', $intervals );
	}

	/**
	 * Get Payment Feature Label.
	 *
	 * @since 4.3.0
	 *
	 * @param Course $course The course object.
	 */
	public function get_payments_feature_label( Course $course ) {
		// Get Types & Intervals.
		$payment_types     = $this->get_payment_types();
		$payment_intervals = $this->get_payment_intervals();

		// Get Course Payments Details.
		$payments_type     = $course->get_payments_type();
		$payments_price    = $course->get_payments_price();
		$payments_interval = $course->get_subscription_interval();

		// Default.
		$payments_feature_label = '';

		// Define the labels internally.
		switch ( $payments_type ) {
			case 'one-time' :
				$payments_feature_label = sprintf( __( 'One-Time Payment - %s', 'wp-courseware' ), wpcw_price( $payments_price ) );
				break;
			case 'subscription' :
				/* translators: %1$s - Payments Price, %2$s - Payments Interval. */
				$payments_feature_label = sprintf( __( 'Subscription - %1$s / %2$s', 'wp-courseware' ), wpcw_price( $payments_price ), $payments_interval ? $payments_interval : esc_html__( 'Monthly', 'wp-courseware' ) );
				break;
			case 'free' :
				$payments_feature_label = __( 'Free', 'wp-courseware' );
				break;
			default:
				break;
		}

		/**
		 * Filter: Course Payments Feature Label.
		 *
		 * @since 4.3.0
		 *
		 * @param Course The Course model object.
		 * @param array The valid payment types array.
		 * @param array The valid payment intervals array.
		 *
		 * @return string The Course Payments Feature Label.
		 */
		return apply_filters( "wpcw_course_payments_feature_label", $payments_feature_label, $course, $payment_types, $payment_intervals );
	}

	/**
	 * Shortcode Courses Display.
	 *
	 * @since 4.3.0
	 *
	 * @param array $atts The array of shortcode attributes.
	 */
	public function courses_display( $atts = array() ) {
		$query_args = array(
			'number'        => 10,
			'order'         => 'ASC',
			'orderby'       => 'course_id',
			'course_author' => '',
			'course_id'     => 0,
			'search'        => '',
		);

		$query_args = wp_parse_args( $atts, $query_args );
		$courses    = $this->get_courses( $query_args );

		if ( empty( $courses ) ) {
			wpcw_print_notice( esc_html__( 'There are currently no courses available to display.', 'wp-courseware' ), 'info' );

			return;
		}

		wpcw_get_template( 'courses.php', array( 'courses' => $courses ) );
	}
}