<?php
/**
 * WP Courseware Classroom Table.
 *
 * @package WPCW
 * @subpackage Admin\Tables
 * @since 4.3.0
 */
namespace WPCW\Admin\Tables;

use WPCW\Admin\Pages\Page_Course_Classroom;
use WPCW\Models\Course;
use WPCW\Models\Student;

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Class Classroom Table.
 *
 * @since 4.1.0
 */
class Table_Classroom extends Table_Students {

	/**
	 * @var Page_Course_Classroom The Course Classroom Admin Page.
	 * @since 4.1.0
	 */
	protected $page;

	/**
	 * @var Course The course object.
	 * @since 4.1.0
	 */
	protected $course;

	/**
	 * Table_Classroom Constructor.
	 *
	 * @param array $args
	 */
	public function __construct( array $args = array() ) {
		if ( isset( $args['course'] ) ) {
			$this->course = $args['course'];
			unset( $args['course'] );
		} else {
			$this->course = new Course( 0 );
		}

		parent::__construct( $args );
	}

	/**
	 * Get Table Columns
	 *
	 * @since 4.1.0
	 *
	 * @return array $columns An array of columns displayed in the table.
	 */
	public function get_columns() {
		$columns = array(
			'cb'       => '<input type="checkbox" />',
			'name'     => esc_html__( 'Name', 'wp-courseware' ),
			'email'    => esc_html__( 'Email', 'wp-courseware' ),
			'percent'  => esc_html__( 'Progress %', 'wp-courseware' ),
			'progress' => esc_html__( 'Detailed Progress', 'wp-courseware' ),
		);

		return $columns;
	}

	/**
	 * Column Percent.
	 *
	 * @since 4.1.0
	 *
	 * @param Student $student The current student.
	 *
	 * @return string
	 */
	public function column_percent( $student ) {
		$student_id = $student->get_ID();
		$course_id  = $this->course->get_course_id();

		$user_progress = wpcw()->students->get_student_progress_bar( $student_id, $course_id );

		return $user_progress;
	}

	/**
	 * Column Reset.
	 *
	 * @since 4.1.0
	 *
	 * @param Student $student The current student.
	 *
	 * @return string
	 */
	public function column_reset( $student ) {
		$student_id = $student->get_ID();
		$course_id  = $this->course->get_course_id();

		ob_start();
		?>
        <form method="get" action="<?php echo $this->page->get_url(); ?>">
            <input type="hidden" name="wpcw_users_single" value="<?php echo $student_id; ?>"/>
            <input type="hidden" name="course_id" value="<?php echo $course_id; ?>"/>
			<?php echo $this->get_single_reset_dropdown( array( $course_id ) ); ?>
        </form>
		<?php
		return ob_get_clean();
	}

	/**
	 * Get Course Reset Dropdown.
	 *
	 * @since 4.1.0
	 *
	 * @return string Course Reset Dropdown html.
	 */
	public function get_bulk_reset_dropdown() {
		$course_id = $this->course->get_course_id();

		if ( ! $course_id ) {
			return;
		}

		ob_start();
		?>
        <input type="hidden" name="bulk_course_id" value="<?php echo $course_id; ?>"/>
		<?php echo wpcw()->courses->get_courses_reset_dropdown( array( $course_id ), true ); ?>
        <button type="submit" name="wpcw_user_bulk_progress_reset" id="wpcw_user_progress_reset_point_bulk_btn" class="button" value="Reset"><i class="wpcw-fas wpcw-fa-retweet"></i> <?php esc_html_e( 'Reset', 'wp-courseware' ); ?></button>
		<?php
		return ob_get_clean();
	}

	/**
	 * Get Bulk Actions.
	 *
	 * @since 4.1.0
	 *
	 * @return array $actions The bulk actions
	 */
	public function get_bulk_actions() {
		return array();
	}

	/**
	 * Generate the table navigation above or below the table.
	 *
	 * @since 4.1.0
	 *
	 * @param string $which
	 */
	protected function display_tablenav( $which ) {
		if ( 'top' === $which ) {
			wp_nonce_field( 'bulk-' . $this->_args['plural'] );
		}

		$bulk_actions = $this->get_bulk_actions();
		?>
        <div class="tablenav <?php echo esc_attr( $which ); ?>">
			<?php if ( $this->has_items() && ! empty( $bulk_actions ) ) : ?>
                <div class="alignleft actions bulkactions">
					<?php $this->bulk_actions( $which ); ?>
                </div>
			<?php endif;

			$this->extra_tablenav( $which );
			$this->pagination( $which );
			?>

            <br class="clear"/>
        </div>
		<?php
	}

	/**
	 * Extra controls to be displayed between bulk actions and pagination
	 *
	 * @since 3.1.0
	 * @access protected
	 *
	 * @param string $which
	 */
	protected function extra_tablenav( $which ) {
		if ( 'top' !== $which ) {
			return;
		}
		?>
        <div class="alignleft wpcw-custom-actions wpcw-custom-bulkactions">
			<?php echo $this->get_bulk_reset_dropdown(); ?>
        </div>
		<?php
	}

	/**
	 * Get Classroom Student Args.
	 *
	 * @since 4.3.0
	 */
	public function get_classroom_students_args() {
		global $wpdb;

		$page      = isset( $_GET['paged'] ) ? absint( $_GET['paged'] ) : 1;
		$search    = isset( $_GET['s'] ) ? $_GET['s'] : '';
		$order     = isset( $_GET['order'] ) ? $_GET['order'] : 'DESC';
		$orderby   = isset( $_GET['orderby'] ) ? $_GET['orderby'] : 'user_id';
		$course_id = isset( $_GET['course_id'] ) ? $_GET['course_id'] : 0;

		$per_page = $this->get_items_per_page( $this->per_page_option, $this->per_page );

		$args = array(
			'number'    => $per_page,
			'offset'    => $per_page * ( $page - 1 ),
			'search'    => $search,
			'orderby'   => sanitize_text_field( $orderby ),
			'order'     => sanitize_text_field( $order ),
			'course_id' => $course_id,
		);

		return $args;
	}

	/**
	 * Get Students Count.
	 *
	 * @since 4.1.0
	 *
	 * @return int The students count.
	 */
	public function get_students_count() {
		$this->count = wpcw()->students->get_students_count( $this->get_classroom_students_args() );

		return $this->count;
	}

	/**
	 * Get Students Data.
	 *
	 * @since 4.1.0
	 *
	 * @return array $data Modules data.
	 */
	public function get_students_data() {
		return wpcw()->students->get_students( $this->get_classroom_students_args() );
	}
}