<?php
/**
 * WP Courseware Units Page.
 *
 * @package WPCW
 * @subpackage Admin\Pages
 * @since 4.1.0
 */
namespace WPCW\Admin\Pages;

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Class Page_Units.
 *
 * @since 4.1.0
 */
class Page_Units extends Page {

	/**
	 * Units - Setup.
	 *
	 * @since 4.1.0
	 */
	protected function setup() {
		add_action( 'admin_head', array( $this, 'hightlight_submenu_add_edit' ) );
		add_action( 'admin_head', array( $this, 'add_icon_to_title' ) );
		add_action( 'admin_head-edit.php', array( $this, 'add_action_buttons' ) );
		add_action( 'admin_head-post.php', array( $this, 'add_single_action_buttons' ) );
		add_action( 'admin_head-post-new.php', array( $this, 'add_new_action_buttons' ) );
	}

	/**
	 * Highlight Submenu on Post Type Add / Edit
	 *
	 * @since 4.1.0
	 */
	public function hightlight_submenu_add_edit() {
		global $current_screen, $parent_file, $submenu_file;

		if ( empty( $current_screen->post_type ) ) {
			return;
		}

		if ( $current_screen->post_type !== 'course_unit' ) {
			return;
		}

		$parent_file  = $this->admin->get_slug();
		$submenu_file = $this->get_slug();
	}

	/**
	 * Add Icon to Title.
	 *
	 * @since 4.3.0
	 */
	public function add_icon_to_title() {
		global $current_screen;

		if ( empty( $current_screen->post_type ) ) {
			return;
		}

		if ( $current_screen->post_type !== 'course_unit' ) {
			return;
		}

		echo
			'<style type="text/css">
                .wrap h1.wp-heading-inline {
                    position: relative;
                    padding-top: 5px;
                    padding-left: 50px;
                }
                .wrap h1.wp-heading-inline:before {
                    background-image: url("' . wpcw_image_file( 'wp-courseware-icon.svg' ) . '");
                    background-size: 40px 40px;
                    content: "";
                    display: inline-block;
                    position: absolute;
                    top: -2px;
                    left: 0;
                    width: 40px;
                    height: 40px;
                }
            </style>';
	}

	/**
	 * Add Action Buttons.
	 *
	 * @since 4.3.0
	 */
	public function add_action_buttons() {
		global $current_screen;

		if ( 'course_unit' !== $current_screen->post_type ) {
			return;
		}

		$action_buttons = $this->get_action_buttons();

		if ( empty( $action_buttons ) ) {
			return;
		}
		?>
        <script type="text/javascript">
			(function( $ ) {
				$( document ).ready( function() {
					$( '<?php echo $action_buttons; ?>' ).insertAfter( '.wrap a.page-title-action' );
				} )
			})( jQuery );
        </script>
		<?php
	}

	/**
	 * Add Single Action Buttons.
	 *
	 * @since 4.3.0
	 */
	public function add_single_action_buttons() {
		global $current_screen;

		if ( 'course_unit' !== $current_screen->post_type ) {
			return;
		}

		$single_action_buttons = $this->get_single_action_buttons();

		if ( empty( $single_action_buttons ) ) {
			return;
		}
		?>
        <script type="text/javascript">
			(function( $ ) {
				$( document ).ready( function() {
					$( '<?php echo $single_action_buttons; ?>' ).insertAfter( '.wrap a.page-title-action' );
				} )
			})( jQuery );
        </script>
		<?php
	}

	/**
	 * Add New Action Buttons.
	 *
	 * @since 4.3.0
	 */
	public function add_new_action_buttons() {
		global $current_screen;

		if ( 'course_unit' !== $current_screen->post_type ) {
			return;
		}

		$single_action_buttons = $this->get_single_action_buttons();

		if ( empty( $single_action_buttons ) ) {
			return;
		}

		$single_action_buttons = sprintf( '<span class="wpcw-single-action-buttons" style="display: inline-block;margin-left: 5px;">%s</span>', $single_action_buttons );
		?>
        <script type="text/javascript">
			(function( $ ) {
				$( document ).ready( function() {
					$( '<?php echo $single_action_buttons; ?>' ).insertAfter( '.wrap h1.wp-heading-inline' );
				} )
			})( jQuery );
        </script>
		<?php
	}

	/**
	 * Get Unit Page Action Buttons.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	protected function get_action_buttons() {
		$actions = sprintf(
			'<a class="page-title-action" href="%s">%s</a>',
			add_query_arg( array( 'page' => 'wpcw-modules' ), admin_url( 'admin.php' ) ),
			esc_html__( 'View Modules', 'wp-courseware' )
		);

		$actions .= sprintf(
			'<a class="page-title-action" href="%s">%s</a>',
			add_query_arg( array( 'page' => 'wpcw-courses' ), admin_url( 'admin.php' ) ),
			esc_html__( 'View Courses', 'wp-courseware' )
		);

		return apply_filters( 'wpcw_admin_page_units_action_buttons', $actions );
	}

	/**
	 * Get Sinle Unit Page Action Buttons.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	protected function get_single_action_buttons() {
		$actions = sprintf(
			'<a class="page-title-action" href="%s">%s</a>',
			$this->get_slug(),
			esc_html__( 'Back to Units', 'wp-courseware' )
		);

		return apply_filters( 'wpcw_admin_page_units_single_action_buttons', $actions );
	}

	/**
	 * Get Units Page Menu Title.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_menu_title() {
		return esc_html__( 'Units', 'wp-courseware' );
	}

	/**
	 * Get Units Page Title.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_page_title() {
		return esc_html__( 'Units', 'wp-courseware' );
	}

	/**
	 * Get Units Page Capability.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_capability() {
		return apply_filters( 'wpcw_admin_page_units_capability', 'view_wpcw_courses' );
	}

	/**
	 * Get Units Page Slug.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_slug() {
		return esc_url( add_query_arg( array( 'post_type' => 'course_unit' ), 'edit.php' ) );
	}

	/**
	 * Get Admin Url.
	 *
	 * @since 4.1.0
	 *
	 * @return string The admin url.
	 */
	public function get_url() {
		return admin_url( $this->get_slug() );
	}

	/**
	 * Get Units Page Callback.
	 *
	 * @since 4.1.0
	 *
	 * @return null
	 */
	protected function get_callback() {
		return null;
	}

	/**
	 * Get Units Page hook.
	 */
	public function get_hook() {
		return '';
	}
}