<?php
/**
 * WP Courseware Courses Page.
 *
 * @package WPCW
 * @subpackage Admin\Pages
 * @since 4.1.0
 */
namespace WPCW\Admin\Pages;

use WPCW\Admin\Tables\Table_Courses;

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Class Page_Courses.
 *
 * @since 4.1.0
 */
class Page_Courses extends Page {

	/**
	 * @var string Screen Option Courses per page option.
	 * @since 4.1.0
	 */
	protected $screen_per_page_option = 'courses_per_page';

	/**
	 * @var int Screen Option courses per page default.
	 * @since 4.1.0
	 */
	protected $screen_per_page_default = 20;

	/**
	 * Setup Page.
	 *
	 * @since 4.1.0
	 */
	protected function setup() {
		add_action( 'init', array( $this, 'save_custom_screen_options' ) );
		add_filter( 'screen_settings', array( $this, 'set_screen_custom' ), 10, 2 );
	}

	/**
	 * Load Courses Page.
	 *
	 * @since 4.1.0
	 */
	public function load() {
		$this->table = new Table_Courses( array(
			'page'            => $this,
			'per_page_option' => $this->screen_per_page_option,
			'per_page'        => $this->screen_per_page_default,
		) );
		$this->table->prepare_items();
	}

	/**
	 * Courses Menu Settings.
	 *
	 * @since 4.1.0
	 */
	public function menu_settings() {
		if ( $this->is_current_page() ) {
			$this->add_icon_count();
		}
	}

	/**
	 * Get Courses Action Buttons.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	protected function get_action_buttons() {
		$actions = sprintf(
			'<a class="page-title-action" href="%s">%s</a>',
			add_query_arg( array( 'page' => 'WPCW_showPage_ModifyCourse' ), admin_url( 'admin.php' ) ),
			esc_html__( 'Add New', 'wp-courseware' )
		);

		$actions .= sprintf(
			'<a id="wpcw-enroll-bulk-students" class="page-title-action" href="#">
                <i class="wpcw-fa wpcw-fa-user-plus left" aria-hidden="true"></i> 
                %s
            </a>',
			esc_html__( 'Bulk Enroll Students', 'wp-courseware' )
		);

		return $actions;
	}

	/**
	 * Course Actions.
	 *
	 * @since 4.1.0
	 */
	public function actions() {
		$action = isset( $_GET['action'] ) ? $_GET['action'] : '';

		if ( ! $action ) {
			return;
		}

		switch ( $action ) {
			case 'delete':
				$course_id = wpcw_get_var( 'course_id' );
				$method    = wpcw_post_var( 'delete_course_type' );
				$method    = ( ! $method ) ? wpcw_get_var( 'delete_course_type' ) : '';
				if ( $course = wpcw()->courses->delete_course( $course_id, $method ) ) {
					wpcw_add_admin_notice_success( esc_html__( 'Course deleted successfully.', 'wp-courseware' ) );
				}
				wp_safe_redirect( $this->get_url() );
				exit;
				break;
			default:
				break;
		}
	}

	/**
	 * Get Courses Page Screen Options.
	 *
	 * @since 4.1.0
	 *
	 * @return array
	 */
	protected function get_screen_options() {
		return array(
			'per_page' => array(
				'label'   => esc_html__( 'Number of courses per page', 'wp-courseware' ),
				'default' => $this->screen_per_page_default,
				'option'  => $this->screen_per_page_option,
			),
		);
	}

	/**
	 * Admin Page Custom Screen Settings.
	 *
	 * @since 4.1.0
	 *
	 * @param array $settings The screen settings.
	 * @param object \WP_Screen The screen object.
	 *
	 * @return array $settings The screen settings.
	 */
	public function set_screen_custom( $settings, $screen ) {
		if ( $screen->base !== 'wp-courseware_page_wpcw-courses' ) {
			return $settings;
		}

		$hide_quiz_notifiations = get_user_meta( get_current_user_id(), 'wpcw_course_dashboard_quiz_notification_hide', true );

		if ( ! $hide_quiz_notifiations ) {
			$hide_quiz_notifiations = 'show';
		}

		$expand = '<fieldset class="course-quiz-notifications"><legend>' . esc_html__( 'Additional Course Settings', 'wp-courseware' ) . '</legend><label for="course-quiz-notifications">';
		$expand .= '<input type="hidden" name="quiz_notification_hide_submit" value="yes" />';
		$expand .= '<input type="checkbox" name="quiz_notification_hide" id="course-quiz-notifications" value="show" ' . checked( $hide_quiz_notifiations, 'show', false ) . ' />';
		$expand .= esc_html__( 'Enable / Disable - Quiz Notifications.', 'wp-courseware' ) . '</label></fieldset>';

		$settings .= $expand;

		return $settings;
	}

	/**
	 * Save Custom Screen Options.
	 *
	 * @since 4.1.0
	 *
	 * @return mixed
	 */
	public function save_custom_screen_options() {
		if ( ! is_admin() ) {
			return;
		}

		if ( wpcw_get_var( 'page' ) !== $this->get_slug() ) {
			return;
		}

		if ( isset( $_POST['quiz_notification_hide_submit'] ) && 'yes' === $_POST['quiz_notification_hide_submit'] ) {
			if ( isset( $_POST['quiz_notification_hide'] ) && $_POST['quiz_notification_hide'] == 'show' ) {
				update_user_meta( get_current_user_id(), 'wpcw_course_dashboard_quiz_notification_hide', 'show', 'hide' );
			} else {
				update_user_meta( get_current_user_id(), 'wpcw_course_dashboard_quiz_notification_hide', 'hide', 'show' );
			}
		}
	}

	/**
	 * Get Courses Menu Title.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_menu_title() {
		return esc_html__( 'Courses', 'wp-courseware' );
	}

	/**
	 * Get Courses Page Title.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_page_title() {
		return esc_html__( 'Courses', 'wp-courseware' );
	}

	/**
	 * Get Courses Page Capability.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_capability() {
		return apply_filters( 'wpcw_admin_page_courses_capability', 'view_wpcw_courses' );
	}

	/**
	 * Get Courses Page Slug.
	 *
	 * @since 4.1.0
	 *
	 * @return string
	 */
	public function get_slug() {
		return 'wpcw-courses';
	}

	/**
	 * Display Courses Page.
	 *
	 * @since 4.1.0
	 *
	 * @return void
	 */
	protected function display() {
		do_action( 'wpcw_admin_page_courses_display_top', $this );
		?>
        <form id="wpcw-admin-page-courses-form" method="get" action="<?php echo $this->get_url(); ?>">
            <input type="hidden" name="page" value="<?php echo $this->get_slug(); ?>"/>
			<?php $this->table->search_box( esc_html__( 'Search Courses', 'wp-courseware' ), 'wpcw-courses' ); ?>
			<?php $this->table->views(); ?>
			<?php $this->table->display(); ?>
        </form>
		<?php
		do_action( 'wpcw_admin_page_courses_display_bottom', $this );
	}

	/**
	 * Course Page Views.
	 *
	 * @since 4.1.0
	 */
	public function views() {
		$views = array(
			'common/notices',
			'common/form-field',
			'enrollment/enroll-bulk-students',
		);

		foreach ( $views as $view ) {
			echo $this->get_view( $view );
		}

		?>
        <div id="wpcw-enroll-bulk">
            <wpcw-enroll-bulk-students></wpcw-enroll-bulk-students>
        </div>
		<?php
	}

	/**
	 * Add Quiz Needs Grading Icon Count.
	 *
	 * @since 4.1.0
	 */
	public function add_icon_count() {
		global $submenu;

		if ( ! isset( $submenu[ $this->admin->get_slug() ] ) ) {
			return false;
		}

		foreach ( $submenu[ $this->admin->get_slug() ] as $index => $details ) {
			if ( $details[2] == $this->get_slug() ) {
				$course_id = wpcw_get_var( 'course_id' );

				if ( empty( $course_id ) ) {
					$course_id = 0;
				}

				$quiz_count = wpcw()->courses->get_course_quizzes_that_need_grading_count( $course_id );

				if ( $quiz_count > 0 ) {
					$submenu[ $this->admin->get_slug() ][ $index ][0] .= sprintf( '&nbsp;&nbsp;<span class="update-plugins count-%d"><span class="update-count">%s</span></span>', $quiz_count, $quiz_count );
				}

				return;
			}
		}
	}
}