<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       http://plugins.db-dzine.com
 * @since      1.0.0
 *
 * @package    WooCommerce_Quick_Order
 * @subpackage WooCommerce_Quick_Order/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    WooCommerce_Quick_Order
 * @subpackage WooCommerce_Quick_Order/public
 * @author     Daniel Barenkamp <support@db-dzine.com>
 */
class WooCommerce_Quick_Order_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * options of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      array    $options
	 */
	private $options;

	/**
	 * if true this product will be excluded
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      array    $options
	 */
	private $exclude;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) 
	{
		$this->plugin_name = $plugin_name;
		$this->version = $version;
	}

	/**
	 * Enqueu Styles
	 * @author Daniel Barenkamp
	 * @version 1.0.0
	 * @since   1.0.0
	 * @link    https://plugins.db-dzine.com
	 * @return  [type]                       [description]
	 */
	public function enqueue_styles() 
	{
		global $woocommerce_quick_order_options;

		$this->options = $woocommerce_quick_order_options;

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/woocommerce-quick-order-public.css', array(), $this->version, 'all' );
		if($this->get_option('datatablesEnable')) {
			wp_enqueue_style( 'jquery-datatables', 'https://cdn.datatables.net/1.10.16/css/jquery.dataTables.min.css', array(), '1.10.16', 'all' );
			wp_enqueue_style( 'jquery-datatables-responsive', 'https://cdn.datatables.net/responsive/2.2.1/css/responsive.dataTables.min.css', array(), '2.2.1', 'all' );
			wp_enqueue_style( 'jquery-datatables-buttons', 'https://cdn.datatables.net/buttons/1.5.1/css/buttons.dataTables.min.css', array(), '1.5.1', 'all' );
		}
	}

	/**
	 * Enque JS SCripts
	 * @author Daniel Barenkamp
	 * @version 1.0.0
	 * @since   1.0.0
	 * @link    https://plugins.db-dzine.com
	 * @return  [type]                       [description]
	 */
	public function enqueue_scripts() 
	{
		global $woocommerce_quick_order_options;

		$this->options = $woocommerce_quick_order_options;

		if($this->get_option('datatablesEnable')) {
			wp_enqueue_script( 'jquery-datatables', 'https://cdn.datatables.net/1.10.16/js/jquery.dataTables.min.js', array( 'jquery' ), '1.10.16', true );

			$jss = array(
				'datatables-responsive' => 'https://cdn.datatables.net/responsive/2.2.1/js/dataTables.responsive.min.js',
				'datatables-buttons' => 'https://cdn.datatables.net/buttons/1.5.1/js/dataTables.buttons.min.js',
				'datatables-jszip' => 'https://cdnjs.cloudflare.com/ajax/libs/jszip/3.1.3/jszip.min.js',
				'datatables-pdfmake' => 'https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.1.32/pdfmake.min.js',
				'datatables-vfs_fonts' => 'https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.1.32/vfs_fonts.js',
				'datatables-html5' => 'https://cdn.datatables.net/buttons/1.5.1/js/buttons.html5.min.js',
				'datatables-print' => 'https://cdn.datatables.net/buttons/1.5.1/js/buttons.print.min.js',
			);

			foreach ($jss as $key => $js) {
				wp_enqueue_script( $key, $js, array( 'jquery', 'jquery-datatables' ), '1.10.16', true );
			}
		}

		wp_enqueue_script( $this->plugin_name . '-public', plugin_dir_url( __FILE__ ) . 'js/woocommerce-quick-order-public.js', array( 'jquery'), $this->version, true );
        $forJS['ajax_url'] = admin_url('admin-ajax.php');
        $forJS['trans'] = array(
        	'showAll' => __('Show All', 'woocommerce-quick-order'),
        	'sku' => __('SKU: ', 'woocommerce-quick-order', 'woocommerce-quick-order'),
        	'placeholder' => __('Enter SKU or Product Name ...', 'woocommerce-quick-order'),
    	);
    	$forJS = apply_filters('woocommerce_quick_order_js_settings', $forJS);
        wp_localize_script($this->plugin_name . '-public', 'woocommerce_quick_order_options', $forJS);
	}

	/**
	 * Get Options
	 * @author Daniel Barenkamp
	 * @version 1.0.0
	 * @since   1.0.0
	 * @link    https://plugins.db-dzine.com
	 * @param   [type]                       $option [description]
	 * @return  [type]                               [description]
	 */
    private function get_option($option)
    {
    	if(!is_array($this->options)) {
    		return false;
    	}
    	
    	if(!array_key_exists($option, $this->options))
    	{
    		return false;
    	}
    	return $this->options[$option];
    }
	
    /**
     * Inits the Quick Order
     * @author Daniel Barenkamp
     * @version 1.0.0
     * @since   1.0.0
     * @link    https://plugins.db-dzine.com
     * @return  [type]                       [description]
     */
    public function init()
    {

		global $woocommerce_quick_order_options;
		$this->options = $woocommerce_quick_order_options;
		if (!$this->get_option('enable'))
		{
			return false;
		}

		$excludeUserRoles = $this->get_option('applyForExcludeUserRoles');
		if(is_array($excludeUserRoles) && !empty($excludeUserRoles)) {

			$currentUserRole = $this->get_user_role();
			if(in_array($currentUserRole, $excludeUserRoles))
			{
				return false;
			}
		}

		// Only not logged in
		if($this->get_option('applyForUserGroup') == 2) {
			if(is_user_logged_in())
			{
				return false;
			}
		}

		// Only logged in
		if($this->get_option('applyForUserGroup') == 3) {
			if(!is_user_logged_in())
			{
				return false;
			}
		}


		add_shortcode( 'woocommerce_quick_order_search', array($this, 'get_search'));
		add_shortcode( 'woocommerce_quick_order_table', array($this, 'get_table'));
    }

    /**
     * Check if apply on single Product
     * @author Daniel Barenkamp
     * @version 1.0.0
     * @since   1.0.0
     * @link    https://plugins.db-dzine.com
     * @return  [type]                       [description]
     */
    public function get_search()
    {
    	ob_start();
    	?>
    	<table id="woocommerce-quick-order-search" class="woocommerce-quick-order-search nowrap rwd-table" width="100%">
	 		<thead>
	            <tr>
					<th><?php echo __('Product', 'woocommerce-quick-order') ?></th>
					<th><?php echo __('Price', 'woocommerce-quick-order') ?></th>
					<th><?php echo __('Quantity', 'woocommerce-quick-order') ?></th>
					<th></th>
	            </tr>
	        </thead>
			<tbody>
				<tr>
					<td class="woocommerce-quick-order-search-input">
						<div class="woocommerce-quick-order-search-input-indication"></div>
						<input type="text" name="woocommerce-quick-order-search-input-field" class="woocommerce-quick-order-search-input-field" 
						placeholder="<?php echo __('Enter SKU or Product Name ...', 'woocommerce-quick-order') ?>">
					</td>
					<td class="woocommerce-quick-order-search-price">
						–
					</td>
					<td class="woocommerce-quick-order-search-quantity">
						<input type="number" name="woocommerce-quick-order-quantity-field" class="woocommerce-quick-order-search-quantity-field" value="1">
					</td>
					<td class="woocommerce-quick-order-search-actions">
					</td>
				</tr>
	        </tbody>
	    </table>
	    <a href="#" class="woocommerce-quick-order-search-add-row">
	    	<i class="fa fa-plus"></i> <?php echo __('Add another Row', 'woocommerce-quick-order') ?>
	    </a>
	    <div class="woocommerce-quick-order-search-add-to-cart">
	    	<a href="#" class="woocommerce-quick-order-search-add-to-cart-btn button product_type_simple add_to_cart_button ajax_add_to_cart btn theme-button"><?php echo __('Add All to Cart', 'woocommerce-quick-order') ?></a>
	    </div>
		<?php
	   	$html = ob_get_contents();
		ob_end_clean();
		return $html;
    }

    public function get_table($atts)
	{
		$this->products_in = "";
		$this->categories_in = "";

		$args = shortcode_atts( array(
	        'products' => '',
	        'categories' => '',
	    ), $atts );

		if(!empty($args['products'])) {
	    	$products_in = explode(',', $args['products']);
	    	$this->products_in = $products_in;
	    } elseif(!empty($args['categories'])) {
	    	$categories_in = explode(',', $args['categories']);
	    	$this->categories_in = $categories_in;
	    }

	    $products = $this->get_products();

	    if(empty($products)) {
	    	return _e('No products found', 'woocommerce-quick-order');
	    }

	    $this->products = $products;
	    
	    ob_start();

	    $this->render_table();
	    
	   	$html = ob_get_contents();
		ob_end_clean();
		return $html;
    }

    public function render_table()
    {
  		$this->dataToShowData = apply_filters('woocommerce_quick_order_table_data', $this->get_option('dataToShowData') );

  		$this->dataToShowData = $this->dataToShowData['enabled'];
  		unset($this->dataToShowData['placebo']);

  		$this->first = true;

  		$this->get_table_start();
		$this->get_table_header();
		$this->get_table_rows();
		$this->get_table_footer();

		do_action('woocommerce_quick_order_before_table');

		echo $this->table_start;
		echo $this->header;
		echo $this->rows;
		echo $this->tfooter;

		do_action('woocommerce_quick_order_after_table');

    }

    public function get_table_footer()
    {
    	$this->tfooter = "";
    	if(array_key_exists('mc', $this->dataToShowData)) {

    		$this->tfooter .= '<tfoot>';
    			$this->tfooter .= '<tr id="quick-order-table-multiple-add-to-cart" class="quick-order-table-multiple-add-to-cart">';
    				$this->tfooter .= '<td colspan="' . $this->columnCount . '">';
						$this->tfooter .= '<button type="button" disabled href="#" id="quick-order-table-multiple-add-to-cart-btn" class="quick-order-table-multiple-add-to-cart-btn btn button btn-primary btn-lg">' . __('Add all To Cart', 'woocommerce-quick-order') . '</button>';
					$this->tfooter .= '</td>';
				$this->tfooter .= '</tr>';
			$this->tfooter .= '</tfoot>';
    	}
    	$this->tfooter .= '</table>';
    	return $this->tfooter;
    }

    public function get_table_start()
    {
		$datatablesEnabled = apply_filters('woocommerce_quick_order_datatables_enabled', $this->get_option('datatablesEnable'));

		if($datatablesEnabled == "1") {

			$datatablesOptions = array(
				'paging' => $this->get_option('datatablesPaging') == "1" ? true : false,
				'ordering' => $this->get_option('datatablesOrdering') == "1" ? true : false,
				'info' => $this->get_option('datatablesInfo') == "1" ? true : false,
				'stateSave' => $this->get_option('datatablesStateSave') == "1" ? true : false,
				'searching' => $this->get_option('datatablesSearching') == "1" ? true : false,
				'responsive' => $this->get_option('datatablesResponsive') == "1" ? true : false,
				'filtering' => $this->get_option('datatablesFiltering'),
				'sDom' => $this->get_option('datatablesSDom'),
				'language' => array(
					'url' => '//cdn.datatables.net/plug-ins/9dcbecd42ad/i18n/' . $this->get_option('datatablesLanguage') . '.json',
					// 'url' => '//cdn.datatables.net/plug-ins/9dcbecd42ad/i18n/German.json',
					// 'url' => '//cdn.datatables.net/plug-ins/1.10.16/i18n/' . $this->get_option('datatablesLanguage') . '.json',
				)
			);

			if($this->get_option('datatablesScrollCollapse') == "1") {
				$datatablesOptions['scrollCollapse'] = true;
				$datatablesOptions['scrollY'] = $this->get_option('datatablesScrollY') . 'px';
			}

			$buttons = array();
			if($this->get_option('datatablesPrint') == "1") {
				$buttons[] = array(
					'extend' => 'print',
					'exportOptions' => array(
						'stripHtml' => false,
					)
				);
			}
			if($this->get_option('datatablesCopyHtml5') == "1") {
				$buttons[] = array(
					'extend' => 'copyHtml5',
					'exportOptions' => array(
						'stripHtml' => false,
					)
				);
			}
			if($this->get_option('datatablesExcelHtml5') == "1") {
				$buttons[] = array(
					'extend' => 'excelHtml5',
					'exportOptions' => array(
						'stripHtml' => true,
					)
				);
			}
			if($this->get_option('datatablesCsvHtml5') == "1") {
				$buttons[] = array(
					'extend' => 'csvHtml5',
					'exportOptions' => array(
						'stripHtml' => true,
					)
				);
			}
			if($this->get_option('datatablesPdfHtml5') == "1") {
				$buttons[] = array(
					'extend' => 'pdfHtml5',
					'exportOptions' => array(
						'stripHtml' => true,
					)
				);
			}

			if(!empty($buttons)) {
				$datatablesOptions['buttons'] = $buttons;
			}

			$datatablesOptions = apply_filters('woocommerce_quick_order_datatables_options', $datatablesOptions);

			$this->table_start = '<table id="woocommerce-quick-order" class="woocommerce-quick-order datatables nowrap" data-options=' . json_encode($datatablesOptions, true) . ' width="100%">';
		} else {
			$this->table_start = '<table id="woocommerce-quick-order" class="woocommerce-quick-order nowrap rwd-table" width="100%">';
		}

		return $this->table_start;
    }

    public function get_table_rows()
    {
    	global $product;
    	$this->rows = "";

        foreach ($this->products as $product) {

           	$product = wc_get_product($product->ID);

           	$this->rows .= '<tr id="quick-order-table-row-' . $product->get_id() . '" class="quick-order-table-row">';
           	foreach($this->dataToShowData as $dataToShowKey => $dataToShowValue) {

        		$this->rows .=  '<td data-th="' . __( $dataToShowValue, 'woocommerce-quick-order' ) . '" class="quick-order-table-value-' . $dataToShowKey . ' quick-order-table-value">' . $this->get_data($dataToShowKey, $product) . '</td>';
        	}
        	$this->rows .= '</tr>';
        	
        	$this->first = false;
		}
		wp_reset_postdata();
	}

	public function get_table_header()
	{
		$this->header = "<thead>";
		$this->columnCount = 0;
		foreach($this->dataToShowData as $dataToShowKey => $dataToShowValue) {
           		
			if($dataToShowKey == "mc") {
				$dataToShowValue = "";
			}
			$this->header .= '<th id="quick-order-table-header-' . $dataToShowKey . '" class="quick-order-table-header">' . __( $dataToShowValue, 'woocommerce-quick-order' ) . '</th>';
			$this->columnCount++;
    	}
    	$this->header .= '</thead>';
	}

	/**
	 * Get Variation data by Key
	 * @author Daniel Barenkamp
	 * @version 1.0.0
	 * @since   1.0.0
	 * @link    https://plugins.db-dzine.com
	 * @param   [type]                       $key       [description]
	 * @param   [type]                       $variation [description]
	 * @return  [type]                                  [description]
	 */
	public function get_data($key, $product)
	{
		if(empty($key)) {
			return;
		}

		$html = "";
		switch ($key) {
			case 'im':
				$imageId = $product->get_image_id();
				$image = wp_get_attachment_image($imageId);
				$html = $image; 
				break;
			case 'sk':
				$html = $product->get_sku();
				break;
			case 'pr':
				$html = $product->get_price_html();
				break;
			case 'pt':
				$html = $product->get_name();
				break;
			case 'ct':
				$html = wc_get_product_category_list($product->get_id());
				break;
			case 'st':

				$availability = $product->get_availability();
				if($product->managing_stock()) {
					$html = $availability['availability'];
				} else {
					if($product->is_in_stock()) {
						$html = __('In Stock', 'woocommerce-quick-order');
					} else {
						$html = __('Out of Stock', 'woocommerce-quick-order');
					}
				}
				break;
			case 'de':
				$html = $product->get_short_description();
				break;
			case 'di':
				$html = $product->get_dimensions();
				break;
			case 'we':
				$html = $product->get_weight();
				break;
			case 'rm':
				$link = get_permalink($product->get_id());
				$html = '<a class="button product_type_simple add_to_cart_button ajax_add_to_cart btn theme-button" href="' . $link . '">' . __('Read More', 'woocommerce-quick-order') . '</a>';
				break;
			case 'ca':
				ob_start();
				do_action('woocommerce_simple_add_to_cart');
				$html = ob_get_contents();
				ob_end_clean();
				break;
			case 'qt':
				$html = $this->get_quantity($product);
				break;
			case 'mc':
				global $product;
				$product_id = $product->get_id();
				$html = '<input type="checkbox" class="quick-order-table-multiple-add-to-cart-products" name="quick-order-table-multiple-add-to-cart-products" data-product_id="' . $product_id . '">';
				break;
			default:
				
				break;
		}

		return $html;
	}

    public function get_products()
    {
	    $args = array(
		    'post_type'             => 'product',
		    'post_status'           => 'publish',
		    'ignore_sticky_posts'   => 1,
		    'posts_per_page'        => -1,
		    'post__in'				=> $this->products_in,
		);
		if(!empty($this->categories_in)) {
			$args['tax_query'] = array(
		        array(
		            'taxonomy'      => 'product_cat',
		            'field' 		=> 'term_id', //This is optional, as it defaults to 'term_id'
		            'terms'         => $this->categories_in,
		            'operator'      => 'IN' // Possible values are 'IN', 'NOT IN', 'AND'.
		        ),
	        );
		}
	    $products = new WP_Query($args);

	    return $products->posts;
    }


   	public function search_products()
   	{
   		$response = array(
   			'message' => __('No Products found ...', 'woocommerce-quick-order'),
   			'products' => array(),
   		);

   		$skuOrProduct = $_POST['term'];

   		if(empty($skuOrProduct)) {
   			die(json_encode($response));
   		}

   		$bySKU = wc_get_product_id_by_sku($skuOrProduct);

   		if(!empty($bySKU)) {
			$product = array(
    			'ID' => $bySKU->get_id(),
    			'name' => $bySKU->get_name(),
				'img' => get_the_post_thumbnail_url($bySKU->get_id()),
				'sku' => $bySKU->get_sku(),
				'price' => $product->get_price() . ' ' . get_woocommerce_currency_symbol(),
			);

   			$response['message'] = sprintf( __('%d products found!', 'woocommerce-quick-order'), count($product));
   			$response['products'][] = $product;
   		} else {
   			$byName = $this->search_product_by_name($skuOrProduct);
	   		if(!empty($byName)) {
	   			$response['message'] = sprintf( __('%d products found!', 'woocommerce-quick-order'), count($byName));
	   			$response['products'] = $byName;
	   		}
   		}
   		
   		die(json_encode($response));
   	}

   	protected function search_product_by_name($title)
   	{
	    global $wpdb;
	    $title = esc_sql($title);

	    if(!$title) return;
	    
	    $products = $wpdb->get_results("
	        SELECT ID
	        FROM $wpdb->posts
	        WHERE post_title LIKE '%$title%'
	        AND post_type = 'product' 
	        AND post_status = 'publish'
	    ");
	    if(!empty($products)) {
	    	$temp = array();
	    	foreach ($products as $product_id) {

	    		$product = wc_get_product($product_id);
	    		$temp[] = array(
	    			'ID' => $product->get_id(),
	    			'name' => $product->get_name(),
					'img' => get_the_post_thumbnail_url($product->get_id()),
					'sku' => $product->get_sku(),
					'price' => $product->get_price() . ' ' . get_woocommerce_currency_symbol(),
	    		);
	    	}
	    	return $temp;
	    } else {
	    	return false;
	    }
   	}

	public function add_to_cart()
	{
		ob_start();

		$products = isset( $_POST['products'] ) ? $_POST['products'] : '';
		if(!empty($products) && is_array($products)) {
			foreach ($products as $product) {

			    $product_id        = apply_filters( 'woocommerce_add_to_cart_product_id', absint( $product['id']) );
			    if(isset($product['quantity']) && !empty($product['quantity'])) {
			    	$quantity = absint($product['quantity']);
			    } else {
			    	$quantity = 1;
			    }

			    $passed_validation = apply_filters( 'woocommerce_add_to_cart_validation', true, $product_id );

				if ( $passed_validation) {
					WC()->cart->add_to_cart( $product_id, $quantity);
				}
			}
		} else {
			return false;
		}

        do_action( 'woocommerce_ajax_added_to_cart', $product_id );

        if ( get_option( 'woocommerce_cart_redirect_after_add' ) == 'yes' ) {
            wc_add_to_cart_message( $product_id );
        }

        // Return fragments
        WC_AJAX::get_refreshed_fragments();

	    die();
	}

	/**
	 * Get Variation Add to Cart
	 * @author Daniel Barenkamp
	 * @version 1.0.0
	 * @since   1.0.0
	 * @link    https://plugins.db-dzine.com
	 * @param   [type]                       $variation [description]
	 * @return  [type]                                  [description]
	 */
	public function get_quantity($product)
	{
		ob_start();
	
		do_action( 'woocommerce_before_add_to_cart_quantity' );
		woocommerce_quantity_input( array(
			'min_value'   => apply_filters( 'woocommerce_quantity_input_min', $product->get_min_purchase_quantity(), $product ),
			'max_value'   => apply_filters( 'woocommerce_quantity_input_max', $product->get_max_purchase_quantity(), $product ),
			'input_value' => isset( $_POST['quantity'] ) ? wc_stock_amount( $_POST['quantity'] ) : $product->get_min_purchase_quantity(),
		) );
		/**
		 * @since 3.0.0.
		 */
		do_action( 'woocommerce_after_add_to_cart_quantity' );

		$html = ob_get_contents();
		ob_end_clean();
		return $html;
	}

	/**
	 * Return the current user role
	 * @author Daniel Barenkamp
	 * @version 1.0.0
	 * @since   1.0.0
	 * @link    https://plugins.db-dzine.com
	 * @return  [type]                       [description]
	 */
	private function get_user_role()
	{
		global $current_user;

		$user_roles = $current_user->roles;
		$user_role = array_shift($user_roles);

		return $user_role;
	}
}