<?php
/**
 * Plugin Name: Soliloquy - Instagram Addon
 * Plugin URI:  http://soliloquywp.com
 * Description: Enables Instagram sliders in Soliloquy.
 * Author:      Thomas Griffin
 * Author URI:  http://thomasgriffinmedia.com
 * Version:     2.1.2
 * Text Domain: soliloquy-instagram
 * Domain Path: languages
 *
 * Soliloquy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * Soliloquy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Soliloquy. If not, see <http://www.gnu.org/licenses/>.
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Define necessary addon constants.
define( 'SOLILOQUY_INSTAGRAM_PLUGIN_NAME', 'Soliloquy - Instagram Addon' );
define( 'SOLILOQUY_INSTAGRAM_PLUGIN_VERSION', '2.1.2' );
define( 'SOLILOQUY_INSTAGRAM_PLUGIN_SLUG', 'soliloquy-instagram' );

// Register activation and uninstall hooks.
register_activation_hook( __FILE__, 'soliloquy_instagram_activation_hook' );
/**
 * Fired when the plugin is activated.
 *
 * @since 1.0.0
 *
 * @global object $wpdb         The WordPress database object.
 * @param boolean $network_wide True if WPMU superadmin uses "Network Activate" action, false otherwise.
 */
function soliloquy_instagram_activation_hook( $network_wide ) {

    $instance = Soliloquy::get_instance();

    if ( is_multisite() && $network_wide ) {
        global $wpdb;
        $site_list = $wpdb->get_results( "SELECT * FROM $wpdb->blogs ORDER BY blog_id" );
        foreach ( (array) $site_list as $site ) {
            switch_to_blog( $site->blog_id );

            // Set default option.
            $option = get_option( 'soliloquy_instagram' );
            if ( ! $option || empty( $option ) ) {
                update_option( 'soliloquy_instagram', soliloquy_instagram_default_options() );
            }

            restore_current_blog();
        }
    } else {
        // Set default option.
        $option = get_option( 'soliloquy_instagram' );
        if ( ! $option || empty( $option ) ) {
            update_option( 'soliloquy_instagram', soliloquy_instagram_default_options() );
        }
    }

}

register_uninstall_hook( __FILE__, 'soliloquy_instagram_uninstall_hook' );
/**
 * Fired when the plugin is uninstalled.
 *
 * @since 1.0.0
 *
 * @global object $wpdb The WordPress database object.
 */
function soliloquy_instagram_uninstall_hook() {

    $instance = Soliloquy::get_instance();

    if ( is_multisite() ) {
        global $wpdb;
        $site_list = $wpdb->get_results( "SELECT * FROM $wpdb->blogs ORDER BY blog_id" );
        foreach ( (array) $site_list as $site ) {
            switch_to_blog( $site->blog_id );
            delete_option( 'soliloquy_instagram' );
            restore_current_blog();
        }
    } else {
        delete_option( 'soliloquy_instagram' );
    }

}

add_action( 'plugins_loaded', 'soliloquy_instagram_plugins_loaded' );
/**
 * Ensures the full Soliloquy plugin is active before proceeding.
 *
 * @since 1.0.0
 *
 * @return null Return early if Soliloquy is not active.
 */
function soliloquy_instagram_plugins_loaded() {

    // Bail if the main class does not exist.
    if ( ! class_exists( 'Soliloquy' ) ) {
        return;
    }

    // Fire up the addon.
    add_action( 'soliloquy_init', 'soliloquy_instagram_plugin_init' );

}

/**
 * Loads all of the addon hooks and filters.
 *
 * @since 1.0.0
 */
function soliloquy_instagram_plugin_init() {

    // Run a verifcation and removal check for Instagram auth data.
    soliloquy_instagram_verify_auth();
    soliloquy_instagram_remove_auth();

    // Add hooks and filters.
    add_action( 'soliloquy_updater', 'soliloquy_instagram_updater' );
    add_filter( 'soliloquy_defaults', 'soliloquy_instagram_defaults', 10, 2 );
    add_action( 'soliloquy_metabox_scripts', 'soliloquy_instagram_metabox_scripts' );
    add_filter( 'soliloquy_slider_types', 'soliloquy_instagram_type' );
    add_action( 'soliloquy_display_instagram', 'soliloquy_instagram_settings' );
    add_filter( 'soliloquy_save_settings', 'soliloquy_instagram_save', 10, 2 );
    add_action( 'soliloquy_flush_caches', 'soliloquy_instagram_flush_caches', 10, 2 );
    add_filter( 'soliloquy_settings_tab_nav', 'soliloquy_instagram_settings_tab' );
    add_action( 'soliloquy_tab_settings_instagram', 'soliloquy_instagram_settings_output' );
    add_filter( 'soliloquy_pre_data', 'soliloquy_instagram_data', 10, 2 );
    add_filter( 'soliloquy_output_classes', 'soliloquy_instagram_class', 10, 2 );

}

/**
 * Checks for and saves Instagram auth data.
 *
 * @since 1.0.0
 *
 * @retun null Return early if no auth is found.
 */
function soliloquy_instagram_verify_auth() {

    // Return early if no auth data is present.
    if ( ! soliloquy_instagram_has_auth() ) {
        return;
    }

    // Save the auth access token.
    $data = json_decode( stripslashes_deep( $_GET['soliloquy-instagram'] ) );
    if ( empty( $data->access_token ) || empty( $data->user->id ) ) {
        return;
    }

    // Update the option with the Instagram access token and user ID.
    $auth          = soliloquy_instagram_get_auth();
    $auth['token'] = $data->access_token;
    $auth['id']    = $data->user->id;
    update_option( 'soliloquy_instagram', $auth );

}

/**
 * Removes Instagram auth data from the site.
 *
 * @since 1.0.0
 *
 * @retun null Return early if no auth is found.
 */
function soliloquy_instagram_remove_auth() {

    // Return early if not removing the auth data.
    if ( ! soliloquy_instagram_removing_auth() ) {
        return;
    }

    // Return early if no auth data is present.
    $auth = soliloquy_instagram_get_auth();
    if ( ! $auth ) {
        return;
    }

    // Update the option to remove the auth.
    update_option( 'soliloquy_instagram', soliloquy_instagram_default_options() );

}

/**
 * Initializes the addon updater.
 *
 * @since 1.0.0
 *
 * @param string $key The user license key.
 */
function soliloquy_instagram_updater( $key ) {

    $args = array(
        'plugin_name' => SOLILOQUY_INSTAGRAM_PLUGIN_NAME,
        'plugin_slug' => SOLILOQUY_INSTAGRAM_PLUGIN_SLUG,
        'plugin_path' => plugin_basename( __FILE__ ),
        'plugin_url'  => trailingslashit( WP_PLUGIN_URL ) . SOLILOQUY_INSTAGRAM_PLUGIN_SLUG,
        'remote_url'  => 'http://soliloquywp.com/',
        'version'     => SOLILOQUY_INSTAGRAM_PLUGIN_VERSION,
        'key'         => $key
    );
    $soliloquy_instagram_updater = new Soliloquy_Updater( $args );

}

/**
 * Applies a default to the addon setting.
 *
 * @since 1.0.0
 *
 * @param array $defaults  Array of default config values.
 * @param int $post_id     The current post ID.
 * @return array $defaults Amended array of default config values.
 */
function soliloquy_instagram_defaults( $defaults, $post_id ) {

    $defaults['instagram_type']    = 'users_self_media_recent';
    $defaults['instagram_tag']     = '';
    $defaults['instagram_number']  = 5;
    $defaults['instagram_res']     = 'standard_resolution';
    $defaults['instagram_link']    = 0;
    $defaults['instagram_caption'] = 1;
    $defaults['instagram_random']  = 0;
    $defaults['instagram_cache']   = 1;
    return $defaults;

}

/**
 * Enqueues scripts for metabox
 *
 * @since 2.1.2
 */
function soliloquy_instagram_metabox_scripts() {

    wp_enqueue_script( SOLILOQUY_INSTAGRAM_PLUGIN_SLUG . '-metabox-script', plugins_url( 'assets/js/metabox.js', __FILE__ ), array( 'jquery' ), SOLILOQUY_INSTAGRAM_PLUGIN_VERSION, true );

}

/**
 * Adds the "Instagram" slider type to the list of available options.
 *
 * @since 1.0.0
 *
 * @param array $types  Types of sliders to select.
 * @return array $types Amended types of sliders to select.
 */
function soliloquy_instagram_type( $types ) {

    $types['instagram'] = __( 'Instagram', 'soliloquy-instagram' );
    return $types;

}

/**
 * Callback for displaying the UI for setting instagram options.
 *
 * @since 1.0.0
 *
 * @param object $post The current post object.
 */
function soliloquy_instagram_settings( $post ) {

    $instance = Soliloquy_Metaboxes::get_instance();
    $auth     = soliloquy_instagram_get_auth();

    if ( empty( $auth['token'] ) ) : ?>
    <div class="error below-h2">
        <p><?php _e( 'Oops! It looks like you have not authenticated Soliloquy with your Instagram account yet.', 'soliloquy-instagram' ); ?></p>
    </div>
    <p><a href="<?php echo admin_url( 'edit.php?post_type=soliloquy&page=soliloquy-settings#soliloquy-tab-instagram' ); ?>" class="button button-primary"><?php _e( 'Click Here to Get Started with the Authentication Process', 'soliloquy-instagram' ); ?></a></p>
    <?php else : ?>
    <div id="soliloquy-instagram">
        <p class="soliloquy-intro"><?php _e( 'The settings below adjust the Instagram settings for the slider.', 'soliloquy-instagram' ); ?></p>
        <table class="form-table">
            <tbody>
                <tr id="soliloquy-config-instagram-type-box">
                    <th scope="row">
                        <label for="soliloquy-config-instagram-type"><?php _e( 'Feed Type', 'soliloquy-instagram' ); ?></label>
                    </th>
                    <td>
                        <select id="soliloquy-config-instagram-type" name="_soliloquy[instagram_type]">
                            <?php foreach ( (array) soliloquy_instagram_types() as $i => $data ) : ?>
                                <option value="<?php echo $data['value']; ?>"<?php selected( $data['value'], $instance->get_config( 'instagram_type', $instance->get_config_default( 'instagram_type' ) ) ); ?>><?php echo $data['name']; ?></option>
                            <?php endforeach; ?>
                        </select>
                        <p class="description"><?php _e( 'The type of images to pull from Instagram.', 'soliloquy-instagram' ); ?></p>
                    </td>
                </tr>
                <tr id="soliloquy-config-instagram-tag-box">
                    <th scope="row">
                        <label for="soliloquy-config-instagram-tag"><?php _e( 'Tag', 'soliloquy-instagram' ); ?></label>
                    </th>
                    <td>
                        <input id="soliloquy-config-instagram-tag" type="text" name="_soliloquy[instagram_tag]" value="<?php echo $instance->get_config( 'instagram_tag', $instance->get_config_default( 'instagram_tag' ) ); ?>" />
                        <p class="description"><?php _e( 'Pulls images that match the given tag.', 'soliloquy-instagram' ); ?></p>
                    </td>
                </tr>
                <tr id="soliloquy-config-instagram-number-box">
                    <th scope="row">
                        <label for="soliloquy-config-instagram-number"><?php _e( 'Number of Instagram Photos', 'soliloquy-instagram' ); ?></label>
                    </th>
                    <td>
                        <input id="soliloquy-config-instagram-number" type="number" name="_soliloquy[instagram_number]" value="<?php echo $instance->get_config( 'instagram_number', $instance->get_config_default( 'instagram_number' ) ); ?>" />
                        <p class="description"><?php _e( 'The number of images to pull from your Instagram feed.', 'soliloquy-instagram' ); ?></p>
                    </td>
                </tr>
                <tr id="soliloquy-config-instagram-res-box">
                        <th scope="row">
                            <label for="soliloquy-config-instagram-res"><?php _e( 'Image Resolution', 'soliloquy' ); ?></label>
                        </th>
                        <td>
                            <select id="soliloquy-config-instagram-res" name="_soliloquy[instagram_res]">
                                <?php foreach ( (array) soliloquy_instagram_resolutions() as $i => $data ) : ?>
                                    <option value="<?php echo $data['value']; ?>"<?php selected( $data['value'], $instance->get_config( 'instagram_res', $instance->get_config_default( 'instagram_res' ) ) ); ?>><?php echo $data['name']; ?></option>
                                <?php endforeach; ?>
                            </select>
                            <p class="description"><?php _e( 'Determines the image resolution and size to use from Instagram.', 'soliloquy' ); ?></p>
                        </td>
                    </tr>
                <tr id="soliloquy-config-instagram-link-box">
                    <th scope="row">
                        <label for="soliloquy-config-instagram-link"><?php _e( 'Link to Instagram Location?', 'soliloquy' ); ?></label>
                    </th>
                    <td>
                        <input id="soliloquy-config-instagram-link" type="checkbox" name="_soliloquy[instagram_link]" value="<?php echo $instance->get_config( 'instagram_link', $instance->get_config_default( 'instagram_link' ) ); ?>" <?php checked( $instance->get_config( 'instagram_link', $instance->get_config_default( 'instagram_link' ) ), 1 ); ?> />
                        <span class="description"><?php _e( 'Links the photo to its original location on Instagram.', 'soliloquy' ); ?></span>
                    </td>
                </tr>
                <tr id="soliloquy-config-instagram-caption-box">
                    <th scope="row">
                        <label for="soliloquy-config-instagram-caption"><?php _e( 'Use Photo Caption?', 'soliloquy' ); ?></label>
                    </th>
                    <td>
                        <input id="soliloquy-config-instagram-caption" type="checkbox" name="_soliloquy[instagram_caption]" value="<?php echo $instance->get_config( 'instagram_caption', $instance->get_config_default( 'instagram_caption' ) ); ?>" <?php checked( $instance->get_config( 'instagram_caption', $instance->get_config_default( 'instagram_caption' ) ), 1 ); ?> />
                        <span class="description"><?php _e( 'Displays the photo caption from Instagram on the slide.', 'soliloquy' ); ?></span>
                    </td>
                </tr>
                <tr id="soliloquy-config-instagram-cache-box">
                    <th scope="row">
                        <label for="soliloquy-config-instagram-cache"><?php _e( 'Cache Data from Instagram?', 'soliloquy' ); ?></label>
                    </th>
                    <td>
                        <input id="soliloquy-config-instagram-cache" type="checkbox" name="_soliloquy[instagram_cache]" value="<?php echo $instance->get_config( 'instagram_cache', $instance->get_config_default( 'instagram_cache' ) ); ?>" <?php checked( $instance->get_config( 'instagram_cache', $instance->get_config_default( 'instagram_cache' ) ), 1 ); ?> />
                        <span class="description"><?php _e( 'Caches the data from Instagram to improve performance (recommended).', 'soliloquy' ); ?></span>
                    </td>
                </tr>
                <?php do_action( 'soliloquy_instagram_box', $post ); ?>
            </tbody>
        </table>
    </div>
    <?php endif;

}

/**
 * Saves the addon settings.
 *
 * @since 1.0.0
 *
 * @param array $settings  Array of settings to be saved.
 * @param int $post_id     The current post ID.
 * @return array $settings Amended array of settings to be saved.
 */
function soliloquy_instagram_save( $settings, $post_id ) {

    // If not saving a featured content slider, do nothing.
    if ( ! isset( $_POST['_soliloquy']['type_instagram'] ) ) {
        return $settings;
    }

    // Save the settings.
    $settings['config']['instagram_type']    = esc_attr( $_POST['_soliloquy']['instagram_type'] );
    $settings['config']['instagram_tag']     = sanitize_text_field( $_POST['_soliloquy']['instagram_tag'] );
    $settings['config']['instagram_number']  = absint( $_POST['_soliloquy']['instagram_number'] );
    $settings['config']['instagram_res']     = esc_attr( $_POST['_soliloquy']['instagram_res'] );
    $settings['config']['instagram_link']    = isset( $_POST['_soliloquy']['instagram_link'] ) ? 1 : 0;
    $settings['config']['instagram_caption'] = isset( $_POST['_soliloquy']['instagram_caption'] ) ? 1 : 0;
    $settings['config']['instagram_random']  = isset( $_POST['_soliloquy']['instagram_random'] ) ? 1 : 0;
    $settings['config']['instagram_cache']   = isset( $_POST['_soliloquy']['instagram_cache'] ) ? 1 : 0;
    return $settings;

}

/**
 * Flushes the Instagram data caches on save.
 *
 * @since 1.0.0
 *
 * @param int $post_id The current post ID.
 * @param string $slug The current slider slug.
 */
function soliloquy_instagram_flush_caches( $post_id, $slug ) {

    delete_transient( '_sol_instagram_' . $post_id );
    delete_transient( '_sol_instagram_' . $slug );

}

/**
 * Adds a custom settings tab to the Settings screen.
 *
 * @since 1.0.0
 *
 * @param array $tabs  Array of default settings tabs.
 * @return array $tabs Amended array of default settings tabs.
 */
function soliloquy_instagram_settings_tab( $tabs ) {

    $tabs['instagram'] = __( 'Instagram', 'soliloquy-instagram' );
    return $tabs;

}

/**
 * Outputs the custom settings HTML.
 *
 * @since 1.0.0
 */
function soliloquy_instagram_settings_output() {

    ?>
    <div id="soliloquy-settings-instagram">
        <?php $auth = soliloquy_instagram_get_auth(); if ( ! empty( $auth['token'] ) ) : ?>
        <div class="updated below-h2">
            <p><?php _e( 'Your Instagram account has been authenticated for use with Soliloquy!', 'soliloquy-instagram' ); ?></p>
        </div>
        <p><a href="<?php echo add_query_arg( array( 'post_type' => 'soliloquy', 'soliloquy-instagram-remove' => 'true', 'page' => 'soliloquy-settings#soliloquy-tab-instagram' ), admin_url( 'edit.php' ) ); ?>" class="button button-primary soliloquy-oauth-remove"><?php _e( 'Click Here to Remove Instagram Authentication from Soliloquy', 'soliloquy' ); ?></a></p>
        <?php else : ?>
        <div class="error below-h2">
            <p><?php _e( 'Before you can create Instagram sliders, you need to authenticate Soliloquy with your Instagram account.', 'soliloquy-instagram' ); ?></p>
        </div>
        <p><a href="<?php echo add_query_arg( array( 'client_id' => 'c85ddd3230e1419ea8c9a62169cae60b', 'response_type' => 'code', 'redirect_uri' => 'http://soliloquywp.com/instagram?return_to=' . urlencode( admin_url( 'edit.php?post_type=soliloquy&page=soliloquy-settings' ) ) ), 'https://api.instagram.com/oauth/authorize/' ); ?>" class="button button-primary soliloquy-oauth-authorize"><?php _e( 'Click Here to Authenticate Soliloquy with Instagram', 'soliloquy-instagram' ); ?></a></p>
        <?php endif; ?>
        <?php do_action( 'soliloquy_settings_instagram_box' ); ?>
    </div>
    <?php

}

/**
 * Filters the data to pull images from Instagram for Instagram sliders.
 *
 * @since 1.0.0
 *
 * @param array $data  Array of slider data.
 * @param int $id      The slider ID.
 * @return array $data Amended array of slider data.
 */
function soliloquy_instagram_data( $data, $id ) {

    // Return early if not an Instagram slider.
    $instance = Soliloquy_Shortcode::get_instance();
    if ( 'instagram' !== $instance->get_config( 'type', $data ) ) {
        return $data;
    }

    // Grab the Instagram data.
    $instagram = $instance->get_config( 'instagram_cache', $data ) ? soliloquy_instagram_get_data( $id, $data ) : _soliloquy_instagram_get_data( $id, $data );
    if ( ! $instagram ) {
        return $data;
    }

    // Now that we have the proper data, insert the Instagram data into the slider data.
    $data = soliloquy_instagram_insert_data( $data, $instagram );

    // Return the modified data.
    return apply_filters( 'soliloquy_instagram_data', $data, $id );

}

/**
 * Grabs the appropriate data from Instagram.
 *
 * @since 1.0.0
 *
 * @param mixed $id   The current slider ID.
 * @param array $data Array of slider data.
 * @return array|bool Array of data on success, false on failure.
 */
function soliloquy_instagram_get_data( $id, $data ) {

    // Attempt to return the transient first, otherwise generate the new query to retrieve the data.
    if ( false === ( $instagram_data = get_transient( '_sol_instagram_' . $id ) ) ) {
        $instagram_data = _soliloquy_instagram_get_data( $id, $data );
        if ( $instagram_data ) {
            set_transient( '_sol_instagram_' . $id, maybe_serialize( $instagram_data ), DAY_IN_SECONDS );
        }
    }

    // Return the slider data.
    return maybe_unserialize( $instagram_data );

}

/**
 * Grabs the appropriate data from Instagram if the transient doesn't exist.
 *
 * @since 1.0.0
 *
 * @param mixed $id   The current slider ID.
 * @param array $data Array of slider data.
 * @return array|bool Array of data on success, false on failure.
 */
function _soliloquy_instagram_get_data( $id, $data ) {

    // Grab the Instagram auth data.
    $auth = soliloquy_instagram_get_auth();
    if ( empty( $auth['token'] ) || empty( $auth['id'] ) ) {
        return false;
    }

    // Ping Instagram to retrieve the proper data.
    $instance = Soliloquy_Shortcode::get_instance();
    switch ( $instance->get_config( 'instagram_type', $data ) ) {
        case 'users_self_media_recent':
        default:
            $response = wp_remote_get( esc_url_raw( 'https://api.instagram.com/v1/users/' . $auth['id'] . '/media/recent/?access_token=' . $auth['token'] . '&count=' . $instance->get_config( 'instagram_number', $data ) ) );
            break;
        case 'users_self_media_liked':
            $response = wp_remote_get( esc_url_raw( 'https://api.instagram.com/v1/users/self/media/liked/?access_token=' . $auth['token'] . '&count=' . $instance->get_config( 'instagram_number', $data ) ) );
            break;
        case 'tags_tag_media_recent':
            $response = wp_remote_get( esc_url_raw( 'https://api.instagram.com/v1/tags/' . urlencode( $instance->get_config( 'instagram_tag', $data ) ) . '/media/recent/?access_token=' . $auth['token'] . '&count=' . $instance->get_config( 'instagram_number', $data ) ) );
            break;
    }
    
    // If there is an error with the request, return false.
    if ( is_wp_error( $response ) ) {
        return false;
    }

    // Parse and decode the response body. If there is an error or no response body, return false
    $body = json_decode( wp_remote_retrieve_body( $response ), true );
    if ( is_null( $body ) || empty( $body['data'] ) ) {
        return false;
    }

    // Loop through the response data and remove any emoticons that can't be stored in the DB.
    $sanitized_data = array();
    $res            = $instance->get_config( 'instagram_res', $data );
    foreach ( $body['data'] as $i => $image ) {
        $sanitized_data[$i]['id']      = $image['id'];
        $sanitized_data[$i]['status']  = 'active';
        $sanitized_data[$i]['src']     = ! empty( $image['images'][$res]['url'] ) ? esc_url( $image['images'][$res]['url'] ) : '';
        $sanitized_data[$i]['caption'] = $instance->get_config( 'instagram_caption', $data ) && ! empty( $image['caption']['text'] ) ? esc_attr( soliloquy_instagram_clean_caption( $image['caption']['text'] ) ) : '';
        $sanitized_data[$i]['link']    = $instance->get_config( 'instagram_link', $data ) && ! empty( $image['link'] ) ? esc_url( $image['link'] ) : '';
        $sanitized_data[$i]['type']    = 'image';
    }

    // Return the sanitized data.
    return apply_filters( 'soliloquy_instagram_sanitized_data', $sanitized_data, $body['data'], $id, $data );

}

/**
 * Sanitizes a caption and removes emoticons.
 *
 * @since 1.0.0
 *
 * @param string $caption  An image caption.
 * @return string $caption Sanizited image caption.
 */
function soliloquy_instagram_clean_caption( $caption ) {

    return preg_replace( '/([0-9|#][\x{20E3}])|[\x{00ae}|\x{00a9}|\x{203C}|\x{2047}|\x{2048}|\x{2049}|\x{3030}|\x{303D}|\x{2139}|\x{2122}|\x{3297}|\x{3299}][\x{FE00}-\x{FEFF}]?|[\x{2190}-\x{21FF}][\x{FE00}-\x{FEFF}]?|[\x{2300}-\x{23FF}][\x{FE00}-\x{FEFF}]?|[\x{2460}-\x{24FF}][\x{FE00}-\x{FEFF}]?|[\x{25A0}-\x{25FF}][\x{FE00}-\x{FEFF}]?|[\x{2600}-\x{27BF}][\x{FE00}-\x{FEFF}]?|[\x{2900}-\x{297F}][\x{FE00}-\x{FEFF}]?|[\x{2B00}-\x{2BF0}][\x{FE00}-\x{FEFF}]?|[\x{1F000}-\x{1F6FF}][\x{FE00}-\x{FEFF}]?/u', '', $caption );

}

/**
 * Inserts the Instagram data into the slider.
 *
 * @since 1.0.0
 *
 * @param array $data      Array of slider data.
 * @param array $instagram Array of Instagram image data objects.
 * @return array $data     Amended array of slider data.
 */
function soliloquy_instagram_insert_data( $data, $instagram ) {

    // Empty out the current slider data.
    $data['slider'] = array();

    // Loop through and insert the Instagram data.
    $instance = Soliloquy_Shortcode::get_instance();
    foreach ( $instagram as $i => $image ) {
        // Prepare variables.
        $id             = ! empty( $image['id'] ) ? $image['id'] : $i;
        $prep           = array();
        $prep['status'] = 'active';
        $prep['src']    = ! empty( $image['src'] ) ? esc_url( $image['src'] ) : '';
        $prep['title']  = $prep['alt'] = $prep['caption'] = ! empty( $image['caption'] ) ? esc_attr( $image['caption'] ) : '';
        $prep['link']   = ! empty( $image['link'] ) ? esc_url( $image['link'] ) : '';
        $prep['type']   = 'image';

        // Allow image to be filtered for each image.
        $prep = apply_filters( 'soliloquy_instagram_image', $prep, $instagram, $data );

        // Insert the image into the slider.
        $data['slider'][$id] = $prep;
    }

    // Return and allow filtering of final data.
    return apply_filters( 'soliloquy_instagram_slider_data', $data, $instagram );

}

/**
 * Adds a custom slider class to denote an Instagram slider.
 *
 * @since 1.0.0
 *
 * @param array $classes  Array of slider classes.
 * @param array $data     Array of slider data.
 * @return array $classes Amended array of slider classes.
 */
function soliloquy_instagram_class( $classes, $data ) {

    // Return early if not an Instagram slider.
    $instance = Soliloquy_Shortcode::get_instance();
    if ( 'instagram' !== $instance->get_config( 'type', $data ) ) {
        return $classes;
    }

    // Add custom Instagram class.
    $classes[] = 'soliloquy-instagram-slider';
    return $classes;

}

/**
 * Returns the available Instagram query types.
 *
 * @since 2.1.2
 *
 * @return array Array of Instagram query types.
 */
function soliloquy_instagram_types() {

    $types = array(
        array(
            'value' => 'users_self_media_recent',
            'name'  => __( 'My Instagram Photos', 'soliloquy-instagram' )
        ),
        array(
            'value' => 'users_self_media_liked',
            'name'  => __( 'Instagram Photos I\'ve Liked', 'soliloquy-instagram' )
        ),
        array(
            'value' => 'tags_tag_media_recent',
            'name'  => __( 'Instagram Photos by Tag', 'soliloquy-instagram' )
        ),
    );

    return apply_filters( 'soliloquy_instagram_types', $types );

}

/**
 * Returns the available Instagram image resolutions.
 *
 * @since 1.0.0
 *
 * @return array Array of Instagram image resolutions.
 */
function soliloquy_instagram_resolutions() {

    $resolutions = array(
        array(
            'value' => 'thumbnail',
            'name'  => __( 'Thumbnail (150x150)', 'soliloquy-instagram' )
        ),
        array(
            'value' => 'low_resolution',
            'name'  => __( 'Low Resolution (306x306)', 'soliloquy-instagram' )
        ),
        array(
            'value' => 'standard_resolution',
            'name'  => __( 'Standard Resolution (640x640)', 'soliloquy-instagram' )
        )
    );

    return apply_filters( 'soliloquy_instagram_resolutions', $resolutions );

}

/**
 * Returns Instagram auth data.
 *
 * @since 1.0.0
 *
 * @return string|bool Access token on success, false on failure.
 */
function soliloquy_instagram_get_auth() {

    return get_option( 'soliloquy_instagram' );

}

/**
 * Flag to check for Instagram auth data.
 *
 * @since 1.0.0
 *
 * @return bool True if auth data is available, false otherwise.
 */
function soliloquy_instagram_has_auth() {

    return isset( $_GET['soliloquy-instagram'] ) && ! empty( $_GET['soliloquy-instagram'] );

}

/**
 * Flag to check for removing Instagram auth data.
 *
 * @since 1.0.0
 *
 * @return bool True if auth data is being removed, false otherwise.
 */
function soliloquy_instagram_removing_auth() {

    return isset( $_GET['soliloquy-instagram-remove'] ) && $_GET['soliloquy-instagram-remove'];

}

/**
 * Returns the default options for the Instagram addon.
 *
 * @since 1.0.0
 *
 * @return array Array of default options.
 */
function soliloquy_instagram_default_options() {

    return array(
        'token' => '',
        'id'    => 0
    );

}