<?php

class MediaController extends BaseController {

	/**
	 * Media Repository
	 *
	 * @var Meda
	 */
	protected $media;

	public function __construct(Media $media)
	{
		$this->media = $media;
	}

	/**
	 * Display a listing of the resource.
	 *
	 * @return Response
	 */
	public function index()
	{
		$media = $this->media->all();

		return View::make('media.index', compact('media'));
	}

	/**
	 * Show the form for creating a new resource.
	 *
	 * @return Response
	 */
	public function create()
	{
		$data = array(
			'categories' => Category::orderBy('order', 'ASC')->get()
			);

		return View::make('media.create', $data);
	}

	/**
	 * Store a newly created resource in storage.
	 *
	 * @return Response
	 */
	public function store()
	{
		if(!Auth::guest()){

			$input = Input::all();

			$validation = Validator::make($input, Media::$rules);

			$valid_media = false;
			if(isset($input['pic_url']) && !empty($input['pic_url'])){
				$valid_media = true;
			} else if(isset($input['img_url']) && $input['img_url'] != ''){
				$valid_media = true;
			} else if(isset($input['vid']) && $input['vid'] != ''){
				$valid_media = true;
			}


			if ($validation->passes() && $valid_media)
			{

				if(isset($input['pic'])){
					if(isset($input['img_url']) && $input['img_url'] != ''){
						$input['pic_url'] = $this->uploadImageFromURL($input['img_url'], $this->sanitize($input['title']));
					} else if(isset($input['pic_url'])){

						$input['pic_url'] = $this->uploadImage(Input::file('pic_url'));
					}
					$input['pic'] = 1;
				}

				unset($input['img_url']);

				if(isset($input['vid'])){
					if(isset($input['vid_url'])){
						unset($input['vid']);
						
						if (strpos($input['vid_url'], 'youtube') > 0 || strpos($input['vid_url'], 'youtu.be') > 0) {
							$video_id = explode('?v=', $input['vid_url']);
							if(isset($video_id[1])){
								$img_url = 'http://img.youtube.com/vi/'. $video_id[1] . '/0.jpg';
								$input['pic_url'] = $this->uploadImageFromURL($img_url, $this->sanitize($input['title']));
							} else {
								unset($input['vid_url']);
							}
							$input['vid'] = 1;
						} elseif(strpos($input['vid_url'], 'vimeo') > 0) {
							$vimeo_id = (int)substr(parse_url($input['vid_url'], PHP_URL_PATH), 1);
							$link = unserialize(file_get_contents("http://vimeo.com/api/v2/video/$vimeo_id.php"));
							$image = $link[0]['thumbnail_large'];  
							
							$input['pic_url'] = $this->uploadImageFromURL($image, $this->sanitize($input['title']));
							$input['vid'] = 1;
						} elseif(strpos($input['vid_url'], 'vine') > 0){
							$video_id = explode('/v/', $input['vid_url']);
							$video_id = str_replace('/embed', '', $video_id[1]);
							$video_id = str_replace('/', '', $video_id);
							if(isset($video_id)){
								ini_set('default_socket_timeout', 120);
								$vine = file_get_contents("http://vine.co/v/$video_id");
								preg_match('/property="og:image" content="(.*?)"/', $vine, $matches);

								$image = ($matches[1]) ? $matches[1] : '';
								$input['pic_url'] = $this->uploadImageFromURL($image, $this->sanitize($input['title']));
							} else {
								unset($input['vid_url']);
							}
							$input['vid'] = 1;
						}
					}
				}

				$this->add_daily_media_points();

				$setting = Setting::first();
				if(!$setting->auto_approve_posts){
					$input['active'] = 0;
				}

				$input['title'] = htmlspecialchars(stripslashes($input['title']));

				$input['slug'] = $this->slugify($input['title']);
					
				$slugexist = Media::where('slug', '=', $input['slug'])->first();
				$increment = 1;
				while(isset($slugexist->id)){
					$input['slug'] = $input['slug'] . '-' . $increment;
					$slugexist = Media::where('slug', '=', $input['slug'])->first();
					$increment += 1;
				}

				$new_media = $this->media->create($input);

				return Redirect::to('media' . '/' . $new_media->slug)->with(array('note' => 'Successfully Uploaded Media!', 'note_type' => 'success') );
			}

			return Redirect::to('/upload')->with(array('note' => 'Error uploading media. Please make sure all the fields were filled out correctly', 'note_type' => 'error') );

		} else {
			return Redirect::to('/')->with(array('note' => 'You must be logged in to upload media.', 'note_type' => 'error') );
		}
	}

	// When user submits media award them 1 point, max 5 per day

	private function add_daily_media_points(){
		$user_id = Auth::user()->id;

		$LastQuestionPoints = Point::where('user_id', '=', $user_id)->where('description', '=', 'Daily Upload')->orderBy('created_at', 'desc')->take(5)->get();
		
		$total_daily_questions = 0;
		foreach($LastQuestionPoints as $QuestionPoint){
			if( date('Ymd') ==  date('Ymd', strtotime($QuestionPoint->created_at)) ){
				$total_daily_questions += 1;
			}
		}

		if($total_daily_questions < 5){
			$point = new Point;
			$point->user_id = $user_id;
			$point->description = 'Daily Upload';
			$point->points = 1;
			$point->save();
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Display the specified resource.
	 *
	 * @param  int  $id
	 * @return Response
	 */

	public function show($slug)
	{
		$media = $this->media->where('slug', '=', $slug)->first();
		
		$next = Media::where('active', '=', 1)->where('created_at', '>', date("Y-m-d H:i:s", strtotime($media->created_at)) )->first();
		$previous = Media::where('active', '=', 1)->where('created_at', '<', date("Y-m-d H:i:s", strtotime($media->created_at)) )->orderBy('created_at', 'desc')->first();
		$media_next = Media::where('active', '=', 1)->where('created_at', '>=', date("Y-m-d H:i:s", strtotime($media->created_at)) )->take(6)->get();

		$next_media_count = $media_next->count();

		if($next_media_count < 6){
			$get_prev_results = 6 - $next_media_count;
			$media_prev = Media::where('active', '=', 1)->where('created_at', '<', date("Y-m-d H:i:s", strtotime($media->created_at)) )->orderBy('created_at', 'DESC')->take($get_prev_results)->get();
		} else{
			$media_prev = array();
		}

		$data = array(
			'media' => $media,
			'media_next' => $media_next,
			'next' => $next,
			'previous' => $previous,
			'media_prev' => $media_prev
			);

		return View::make('media.show', $data);
	}

	/**
	 * Show the form for editing the specified resource.
	 *
	 * @param  int  $id
	 * @return Response
	 */
	public function edit($id)
	{
		$media = $this->media->find($id);

		if (is_null($media))
		{
			return Redirect::route('media.index');
		}

		return View::make('media.edit', compact('media'));
	}

	// Add Media Flag

	public function add_flag(){
		$id = Input::get('media_id');
		$media_flag = MediaFlag::where('user_id', '=', Auth::user()->id)->where('media_id', '=', $id)->first();

		if(isset($media_flag->id)){ 
			$media_flag->delete();
		} else {
			$flag = new MediaFlag;
			$flag->user_id = Auth::user()->id;
			$flag->media_id = $id;
			$flag->save();
			echo $flag;
		}
	}


	// Add Media Like

	public function add_like(){
		$id = Input::get('media_id');
		$media_like = MediaLike::where('user_id', '=', Auth::user()->id)->where('media_id', '=', $id)->first();

		if(isset($media_like->id)){ 
			$media_like->delete();
		} else {
			$like = new MediaLike;
			$like->user_id = Auth::user()->id;
			$like->media_id = $id;
			$like->save();
			echo $like;
		}
	}

	/**
	 * Update the specified resource in storage.
	 *
	 * @param  int  $id
	 * @return Response
	 */
	public function update($id)
	{
		$media = Media::find($id);
		if(Auth::user()->admin == 1 || Auth::user()->id == $media->user_id){
			
			try{
				$input = Input::all();
				$media->title = htmlspecialchars($input['title']);
				$media->category_id = $input['category'];
				$media->link_url = htmlspecialchars($input['source']);
				$media->tags = htmlspecialchars($input['tags']);
				$media->save();
				return Redirect::to($input['redirect'])->with(array('note' => 'Successfully Updated!', 'note_type' => 'success') );
			} catch(Exception $e){
				return Redirect::to('/media/' . $id)->with(array('note' => 'Sorry there were problems updating, please try again', 'note_type' => 'error') );
			}

		} else {
			return Redirect::to('/');
		}
	}

	/**
	 * Remove the specified resource from storage.
	 *
	 * @param  int  $id
	 * @return Response
	 */
	public function delete($id)
	{
		$media = Media::find($id);

		if($media->user_id == Auth::user()->id || Auth::user()->admin == 1){

			$media_flags = MediaFlag::where('media_id', '=', $id)->get();
			foreach($media_flags as $media_flag){
				$media_flag->delete();
			}

			$media_likes = MediaLike::where('media_id', '=', $id)->get();
			foreach($media_likes as $media_like){
				$media_like->delete();
			}

			$comments = Comment::where('media_id', '=', $id)->get();
			foreach($comments as $comment){
				$comment_votes = CommentVote::where('comment_id', '=', $comment->id)->get();
				foreach($comment_votes as $comment_vote){
					$comment_vote->delete();
				}

				$comment_flags = CommentFlag::where('comment_id', '=', $comment->id)->get();
				foreach($comment_flags as $comment_flag){
					$comment_flag->delete();
				}

				$comment->delete();
			}

			$media->delete();

		}

		return Redirect::to('admin/media');
	}


	// Upload Images from file upload

	private function uploadImage($img){
		
		$file = $img;
		$month_year = date('FY');
		$upload_folder = 'uploads/images/'.$month_year.'/';

		if (!file_exists($upload_folder)) {
			mkdir($upload_folder, 0777, true);
		}

		$filename =  $file->getClientOriginalName();

		if (file_exists($upload_folder.$filename)) {
			$filename =  uniqid() . '-' . $filename;
		}

		$uploadSuccess = $img->move($upload_folder, $filename);

		if(strpos($filename, '.gif') > 0){
			$new_filename = str_replace('.gif', '-animation.gif', $filename);
			copy($upload_folder . $filename, $upload_folder . $new_filename);
		}
	   
	
		Image::make($upload_folder . $filename)->resize(700, null, true)->save($upload_folder . $filename);

		return $month_year . '/' . $filename;

	}

	// Upload Images from file url

	private function uploadImageFromURL($file_url, $filename){
		
		
		$file = file_get_contents($file_url);

		$month_year = date('FY');
		$upload_folder = 'uploads/images/'.$month_year.'/';

		if (!file_exists($upload_folder)) {
			mkdir($upload_folder, 0777, true);
		}

		if(strpos($file_url, '.gif') > 0){
			$extension = '-animation.gif';
		} else {
			$extension = '.jpg';
		}

		$filename = $filename . $extension;

		if (file_exists($upload_folder.$filename)) {
			$filename =  uniqid() . '-' . $filename . $extension;
		}

	    if(strpos($file_url, '.gif') > 0){
			file_put_contents($upload_folder.$filename, $file);
			$filename = str_replace('-animation.gif', '.gif', $filename);
		}

	    //$extension = $file->getClientOriginalExtension(); //if you need extension of the file
	    file_put_contents($upload_folder.$filename, $file); //$img->move($upload_folder, $filename);


		Image::make($upload_folder . $filename)->resize(700, null, true)->save($upload_folder . $filename);

		return $month_year . '/' . $filename;

	}

	// Sanitize Image URL's

	private function sanitize($string, $force_lowercase = true, $anal = false) {
	    $strip = array("~", "`", "!", "@", "#", "$", "%", "^", "&", "*", "(", ")", "_", "=", "+", "[", "{", "]",
	                   "}", "\\", "|", ";", ":", "\"", "'", "&#8216;", "&#8217;", "&#8220;", "&#8221;", "&#8211;", "&#8212;",
	                   "â€”", "â€“", ",", "<", ".", ">", "/", "?");
	    $clean = trim(str_replace($strip, "", strip_tags($string)));
	    $clean = preg_replace('/\s+/', "-", $clean);
	    $clean = ($anal) ? preg_replace("/[^a-zA-Z0-9]/", "", $clean) : $clean ;
	    return ($force_lowercase) ?
	        (function_exists('mb_strtolower')) ?
	            mb_strtolower($clean, 'UTF-8') :
	            strtolower($clean) :
	        $clean;
	}

	// Slugify Media titles

	public function slugify($text) {
	    // replace non letter or digits by -
		$text = preg_replace('~[^\\pL\d]+~u', '-', $text);

		// trim
		$text = trim($text, '-');

		// transliterate
		$text = iconv('utf-8', 'us-ascii//TRANSLIT', $text);

		// lowercase
		$text = strtolower($text);

		// remove unwanted characters
		$text = preg_replace('~[^-\w]+~', '', $text);

		if (empty($text))
		{
		return 'n-a';
		}

	  	return $text;
	}

}
