<?php
/**
 * Translation functions
 *
 * @author Daniel Bakovic <contact@myarcadeplugin.com>
 * @copyright 2009-2015 Daniel Bakovic
 * @license http://myarcadeplugin.com
 */

/**
 * Copyright @ Daniel Bakovic - contact@myarcadeplugin.com
 * Do not modify! Do not sell! Do not distribute! -
 * Check our license Terms!
 */

// No direct access
if( !defined( 'ABSPATH' ) ) {
  die();
}

/**
 * Retrieve Microsoft Translator Authentification Token
 * http://docs.microsofttranslator.com/oauth-token.html
 *
 * @version 5.29.0
 * @access  public
 * @return  string|bool String on success or FALSE on error
 */
function myarcade_microsoft_get_token() {
  global $myarcade_feedback;

  /* The access token expires after 10 minutes.
   * It is always better to check elapsed time between time at which token issued and current time.
   * If elapsed time exceeds 10 minute time period renew access token by following obtaining access token procedure.
   */

  $general = get_option('myarcade_general');

  if ( empty( $general['azure_key'] ) ) {
    $myarcade_feedback->add_error( __("Microsoft Translator", 'myarcadeplugin') .": ". __("Azure Key missing!", 'myarcadeplugin') );
    return false;
  }

  // Check if we have a valid transient
  $token = get_transient('myarcade_microsoft_token');

  if ( $token ) {
    myarcade_log_translator( "Microsoft Translator - Cached Token Found: " . print_r($token, true) );
    return $token;
  }

  $response = wp_remote_post( 'https://api.cognitive.microsoft.com/sts/v1.0/issueToken' , array(
      'timeout' => 45,
      'sslverify' => false,
      'headers' => array ( 'Ocp-Apim-Subscription-Key' => $general['azure_key'] ),
    )
  );

  if ( is_wp_error( $response ) ) {
    myarcade_log_translator("Microsoft Translator - Request Token Error: ". $response->get_error_message() );
    $myarcade_feedback->add_error( __("Microsoft Translator", 'myarcadeplugin') .": ".$response->get_error_message() );
  }
  else {
    $code = wp_remote_retrieve_response_code( $response );

    if ( '200' == $code ) {
      $token = wp_remote_retrieve_body( $response );

      // Set new transient
      // Azure token is only valid for 10 minutes. Thereby we will set the transient to 9 minutes.
      $expiration = 60 * 8; // 8 Minutes
      set_transient('myarcade_microsoft_token', $token, $expiration);

      myarcade_log_translator("Microsoft Translator - New Token: ". print_r($token, true) );
    }
    else {
      $token = json_decode( wp_remote_retrieve_body( $response ) );

      if ( ! empty( $token->message ) ) {
        myarcade_log_translator("Microsoft Translator - New Token - ERROR Description: ". print_r($token->message, true) );
        $myarcade_feedback->add_error( __("Microsoft Translator", 'myarcadeplugin') .": ". $token->message );
      }
      else {
        myarcade_log_translator("Microsoft Translator - New Token - Unexpected Error Code: ". $code );
        $myarcade_feedback->add_error( __("Microsoft Translator - New Token - Unexpected Error Code: ", 'myarcadeplugin') . $token->message );
      }

      return false;
    }
  }

  return $token;
}

/**
 * Translate a given string with Microsoft Translator
 * http://docs.microsofttranslator.com/text-translate.html
 *
 * @version 5.29.0
 * @access  public
 * @param   string $text Text to translate
 * @return  string|bool Translated string or FALSE on error
 */
function myarcade_microsoft_translate($text = '') {
  global $myarcade_feedback;

  $translatedStr = '';

  if ( empty($text) ) {
    return false;
  }

  if ( !is_myarcade_feedback($myarcade_feedback) ) {
    $myarcade_feedback = new MyAracade_Feedback();
  }

  $token = myarcade_microsoft_get_token();

  if ( $token ) {
    $general = get_option('myarcade_general');

    $url = add_query_arg( array(
        'appid' => "Bearer {$token}",
        'text'  => urlencode( $text ),
        'to'    => $general['translate_to'],
        'contentType' => 'text/html',
      ),
      'https://api.microsofttranslator.com/V2/Http.svc/Translate'
    );

    $response = wp_remote_get( $url, array(
        'timeout' => 45,
        'sslverify' => false,
      )
    );

    if ( is_wp_error( $response ) ) {
      myarcade_log_translator("Microsoft Translator - Error: " .$response->get_error_message() );
      $myarcade_feedback->add_error( __("Microsoft Translator", 'myarcadeplugin') .": ".$response->get_error_message() );
    }
    else {
      // Interprets a string of XML into an object.
      $xmlObj = simplexml_load_string( wp_remote_retrieve_body( $response ) );

      myarcade_log_translator("Microsoft Translator - xmlObj Received: " .print_r($xmlObj, true) );

      // Check if we got an error
      if ( isset($xmlObj[0]->body->h1) ) {
        switch ($xmlObj[0]->body->h1) {
          case 'TranslateApiException': {
            myarcade_log_translator("Microsoft Translator - TranslateApiException: " . $xmlObj[0]->body->p[1] );
            $myarcade_feedback->add_error( __("Microsoft Translator", 'myarcadeplugin') .": ".$xmlObj[0]->body->p[1] );
            return false;
          } break;

          case 'Argument Exception': {
            myarcade_log_translator("Microsoft Translator - Argument Exception: " . $xmlObj[0]->body->p[2] );
            $myarcade_feedback->add_error( __("Microsoft Translator", 'myarcadeplugin') .": ".$xmlObj[0]->body->p[2]  );
            return false;
          } break;
        }
      }

      foreach( (array) $xmlObj[0] as $val ) {
        $translatedStr = $val;
      }

      myarcade_log_translator("Microsoft Translator - Translated Text: " . $translatedStr);
    }
  }
  else {
    myarcade_log_translator("Microsoft Translator - No Valid Access Token");
  }

  return !empty($translatedStr) ? $translatedStr : false;
}

/**
 * Translate a given string with Google Translator
 *
 * @version 5.13.0
 * @access  public
 * @param   string $text Text to translate
 * @return  string|bool Translated string or FALSE on error
 */
function myarcade_google_translate($text) {
  global $myarcade_feedback;

  $result = false;

  $general = get_option('myarcade_general');

  if ( !empty($general['google_id']) ) {

    myarcade_log_translator("Google Translator - Original Text: " . $text);

    // build the url for the google request
    // reference documentation: http://code.google.com/intl/de-DE/apis/ajaxlanguage/documentation/reference.html
    // example API v1 - 'http://ajax.googleapis.com/ajax/services/language/translate?v=1.0&q=hello%20world&langpair=en%7Cit'
    // example API v2 - [ GET https://www.googleapis.com/language/translate/v2?key=INSERT-YOUR-KEY&source=en&target=de&q=Hello%20world ]
    $search = array('\\\\\\\"', '\\\\\"','\\\\n', '\\\\r', '\\\\t', '\\\\$','\\0', "\\'", '\\\\');
    $replace = array('\"', '"', "\n", "\r", "\\t", "\\$", "\0", "'", "\\");
    $text = str_replace( $search, $replace, $text );
    add_filter('https_ssl_verify', '__return_false');

    $url = "https://www.googleapis.com/language/translate/v2?key=".$general['google_id']."&source=en&target=".$general['google_translate_to']."&q=".urlencode($text);

    // Translate given content
    $translation = myarcade_get_file($url);

    myarcade_log_translator("Google Translator - Response: " . print_r($translation, true) );

    if ( !isset($translation['error']) ) {
      $response = json_decode($translation['response']);

      if ( isset($response->error) ) {
        myarcade_log_translator("Google Translator - Response Error: " . $response->error->code . " - " . $response->error->message );
        $myarcade_feedback->add_error( __("Google Translation Error", 'myarcadeplugin') . ": " .$response->error->code . " - " . $response->error->message );
      }
      else {
        // Get translated content
        if ( isset($response->data)
          && isset($response->data->translations)
          && isset($response->data->translations[0]->translatedText) ) {
          // Get translation
          $result = $response->data->translations[0]->translatedText;
          myarcade_log_translator("Google Translator - Translated Text: " . $result);
        }
        else {
          $myarcade_feedback->add_error( __("Google Translation Error", 'myarcadeplugin') . ": " . __("Unknown Error", 'myarcadeplugin') );
        }
      }
    }
    else  {
      myarcade_log_translator("Google Translator - Response Error: " . $translation['error'] );
      $myarcade_feedback->add_error( __("Google Translation Error", 'myarcadeplugin') . ": " .$translation['error']);
    }
  }
  else {
    myarcade_log_translator("Google Translator - Google API Key not provided.");
  }

  return $result;
}

/**
 * Translate a given string with Yandex Translator
 *
 * https://api.yandex.com/translate/doc/dg/reference/translate.xml
 * http://code.google.com/p/translate-api/source/browse/trunk/
 *
 * @version 5.16.0
 * @access  public
 * @param   string $text Text to translate
 * @return  string|bool Translated string or FALSE on error
 */
function myarcade_yandex_translate( $text = '' ) {
  global $myarcade_feedback;

  $result = false;

  $general = get_option('myarcade_general');

  if ( !empty($general['yandex_key']) ) {

    myarcade_log_translator("Yandex Translator - Original Text: " . $text);

    // build the url for the request
    $search = array('\\\\\\\"', '\\\\\"','\\\\n', '\\\\r', '\\\\t', '\\\\$','\\0', "\\'", '\\\\');
    $replace = array('\"', '"', "\n", "\r", "\\t", "\\$", "\0", "'", "\\");
    $text = str_replace( $search, $replace, $text );
    add_filter('https_ssl_verify', '__return_false');

    $url = "https://translate.yandex.net/api/v1.5/tr.json/translate?key=".$general['yandex_key']."&lang=en-".$general['yandex_translate_to']."&format=html&text=".urlencode($text);

    // Translate given content
    $translation = myarcade_get_file($url);

    myarcade_log_translator("Yandex Translator - Response: " . print_r($translation, true) );

    if ( !isset($translation['error']) ) {
      $response = json_decode($translation['response']);

      if ( isset($response->code) && $response->code != 200 ) {
        $status_codes = array(
          200 => __( "Operation completed successfully", 'myarcadeplugin' ),
          401 => __( "Invalid API key", 'myarcadeplugin' ),
          402 => __( "This API key has been blocked", 'myarcadeplugin' ),
          403 => __( "You have reached the daily limit for requests", 'myarcadeplugin' ),
          404 => __( "You have reached the daily limit for the volume of translated text", 'myarcadeplugin' ),
          413 => __( "The text size exceeds the maximum", 'myarcadeplugin' ),
          422 => __( "The text could not be translated", 'myarcadeplugin' ),
          501 => __( "The specified translation direction is not supported", 'myarcadeplugin' ),
        );

        myarcade_log_translator("Yandex Translator - Response Error: " . $status_codes[ $response->code ] );
        $myarcade_feedback->add_error( __("Yandex Translation Error", 'myarcadeplugin') . ": " . $status_codes[ $response->code ] );
      }
      else {
        // Get translated content
        if ( isset($response->text)
          && isset($response->text[0]) ) {
          // Get translation
          $result = $response->text[0];
          myarcade_log_translator("Yandex Translator - Translated Text: " . $result);
        }
        else {
          $myarcade_feedback->add_error( __("Yandex Translation Error", 'myarcadeplugin') . ": " . __("Unknown Error", 'myarcadeplugin') );
        }
      }
    }
    else  {
      myarcade_log_translator("Yandex Translator - Response Error: " . $translation['error'] );
      $myarcade_feedback->add_error( __("Yandex Translation Error", 'myarcadeplugin') . ": " .$translation['error']);
    }
  }
  else {
    myarcade_log_translator("Yandex Translator - Yandex API Key not provided.");
  }

  return $result;
}

/**
 * Translate a given text using Mircosoft, Google Translator API, Yandex
 *
 * @version 5.16.0
 * @access  public
 * @param   string $content String to translate
 * @return  string|bool Translated string or FALSE on error
 */
function myarcade_translate($content) {
  global $myarcade_feedback;

  // Initialite the result
  $result = false;

  // Get general settings
  $general = get_option('myarcade_general');

  switch ( $general['translation'] ) {
    case 'google': {
      $result = myarcade_google_translate($content);
    } break;

    case 'microsoft': {
      $result = myarcade_microsoft_translate($content);
    } break;

    case 'yandex' : {
      $result = myarcade_yandex_translate($content);
    } break;

    default: {
      $myarcade_feedback->add_error( __("Translation service not selected!", 'myarcadeplugin') );
    }
  }

  if ( empty($result) ) {
    $result = false;
  }

  return $result;
}
?>