define(['jquery',
        'underscore',
        'backbone',
        'app/models/rentals',
        'app/models/books',
        'app/models/clients',
        'app/models/staff',
        'app/collections/rentals',
        'app/collections/books',
        'app/collections/clients',
        'app/collections/staff',
        'app/views/header',
        'app/views/books',
        'app/views/book',
        'app/views/clients',
        'app/views/client',
        'app/views/rentals',
        'app/views/rental',
        'app/views/settings',
        'app/views/staff',
        'i18n!nls/language'],
    function ($, _, Backbone, Rental, Book, Client, Staff, RentalsCollection, BooksCollection, ClientsCollection, UsersCollection, HeaderView, BooksListView, BookView, ClientsListView, ClientView, RentalsListView, RentalView, SettingsView, StaffView, lang) {

        'use strict';

        var appRouter = Backbone.Router.extend({

            routes: {
                "": "overview",
                "rentals/page/:page": "overview",
                "rental/add(/:book)": "newRental",
                "rental/:id": "rentalDetails",
                "books": "viewBooks",
                "books/page/:page": "viewBooks",
                "book/add": "bookDetails",
                "book/:id": "bookDetails",
                "history/page/:page": "updateHistory",
                "clients": "viewClients",
                "clients/page/:page": "viewClients",
                "client/add": "clientDetails",
                "client/:id": "clientDetails",
                "settings": "viewSettings",
                "settings/staff/add": "staffDetails",
                "settings/staff/:id": "staffDetails"
            },

            rentalsCollection: {},

            booksCollection: {},

            clientsCollection: {},

            staffCollection: {},

            currentView: null,

            initialized: false,

            eventsAggregator: _.extend({}, Backbone.Events),

            initialize: function () {
                if (!this.initialized) {
                    var self = this,
                        booksList = new BooksCollection(),
                        clientsList = new ClientsCollection(),
                        rentalsList = new RentalsCollection(),
                        users = new UsersCollection();
                    booksList.fetch({
                        success: function () {
                            self.booksCollection = booksList;
                            NProgress.inc();
                            clientsList.fetch({
                                success: function () {
                                    self.clientsCollection = clientsList;
                                    NProgress.inc();
                                    rentalsList.fetch({
                                        success: function () {
                                            self.rentalsCollection = rentalsList;
                                            NProgress.inc();
                                            users.fetch({
                                                success: function () {
                                                    self.staffCollection = users;
                                                    self.initInterface();
                                                    NProgress.inc();
                                                    self.initialized = true;
                                                    Backbone.history.loadUrl(Backbone.history.fragment);
                                                    NProgress.done();
                                                    $('.loader-container').fadeOut("slow");
                                                    console.log("App loaded !");
                                                }
                                            });
                                        }
                                    });
                                }
                            });
                        }
                    });
                    _.extend(Backbone.Validation.messages, {
                        required: lang.error_required,
                        acceptance: lang.error_acceptance,
                        min: lang.error_min,
                        max: lang.error_max,
                        range: lang.error_range,
                        length: lang.error_length,
                        minLength: lang.error_minLength,
                        maxLength: lang.error_maxLength,
                        rangeLength: lang.error_rangeLength,
                        oneOf: lang.error_oneOf,
                        equalTo: lang.error_equalTo,
                        digits: lang.error_digits,
                        number: lang.error_number,
                        email: lang.error_email,
                        url: lang.error_url,
                        inlinePattern: lang.error_inlinePattern
                    });
                }
            },

            execute: function (callback, args, name) {
                if (!this.initialized) {
                    return false;
                }
                if (callback) callback.apply(this, args);
            },

            initInterface: function () {
                $('header').html(new HeaderView({app: this}).render().el);
            },

            switchView: function (view) {
                if (this.currentView) {
                    this.currentView.remove();
                }
                $(view.render().el).appendTo('#content');
                this.currentView = view;
            },

            overview: function (page) {
                var cur_page = page ? parseInt(page, 10) : 1;
                var rentalsView = new RentalsListView({
                    model: this.rentalsCollection,
                    page: cur_page,
                    app: this
                });
                this.switchView(rentalsView);
            },

            newRental: function (book) {
                var book_model = book ? this.booksCollection.get(book) : null,
                    rental = book ? new Rental({
                        book_id: book_model.get('id'),
                        book: book_model.get('title')
                    }) : new Rental(),
                    rentalView = new RentalView({model: rental, collection: this.rentalsCollection});
                this.switchView(rentalView);
            },

            rentalDetails: function (id) {
                var rental = id ? this.rentalsCollection.get(id) : new Rental(),
                    collection = id ? null : this.rentalsCollection,
                    rentalView = new RentalView({model: rental, collection: collection});
                this.switchView(rentalView);
            },

            viewBooks: function (page) {
                var cur_page = page ? parseInt(page, 10) : 1;
                var booksView = new BooksListView({model: this.booksCollection, page: cur_page});
                this.switchView(booksView);
            },

            bookDetails: function (id) {
                var book = id ? this.booksCollection.get(id) : new Book(),
                    collection = id ? null : this.booksCollection,
                    bookView = new BookView({model: book, collection: collection, app: this});
                this.switchView(bookView);
            },

            viewClients: function (page) {
                var cur_page = page ? parseInt(page, 10) : 1,
                    clientsView = new ClientsListView({model: this.clientsCollection, page: cur_page});
                this.switchView(clientsView);
            },

            clientDetails: function (id) {
                var client = id ? this.clientsCollection.get(id) : new Client(),
                    collection = id ? null : this.clientsCollection,
                    clientView = new ClientView({model: client, collection: collection, app: this});
                this.switchView(clientView);
            },

            viewSettings: function () {
                var settingsView = new SettingsView({model: this.staffCollection});
                this.switchView(settingsView);
            },

            staffDetails: function (id) {
                var client = id ? this.staffCollection.get(id) : new Staff(),
                    collection = id ? null : this.staffCollection,
                    clientView = new StaffView({model: client, collection: collection});
                this.switchView(clientView);
            }

        });

        return appRouter;

    }
);
