<?php
use \Carbon\Carbon;

class WebSite extends Eloquent
{

    protected $table = 'website';


    public static function add($protocol, $site, $redirectName = '')
    {
        $datas = [
            'site' => $site
        ];

        $rules = [
            'site' => 'required|unique:website,name'
        ];

        $labels = [
            'site' => 'Site Name'
        ];

        $validator = Validator::make($datas, $rules, [], $labels);

        if ($validator->fails()) {
            return false;
        }

        $newSite = new WebSite();
        $newSite->hash = create_site_hash($site);
        $newSite->protocol = $protocol;
        $newSite->name = $site;
        if($redirectName != ''){
            $newSite->redirect_name = $redirectName;
        }
        //$newSite->charset = $charset;
        $newSite->status = 0;

        if (!$newSite->save()) {
            return false;
        }

        return $newSite->hash;
    }

    public static function deleteWebSite($url)
    {
        return self::where('name', $url)->update([
            'status'       => -1,
            'publish'      => 0
        ]);
    }

    public static function removeWebSite($url)
    {
        return self::where('name', $url)->where('status', 0)->delete();
    }

    public static function getAll($start = 0, $limit = 10)
    {
        return self::where('status', 1)->where('publish', 1)->take($limit)->skip($start)->get();
    }

    public static function getSiteById($id)
    {
        return self::where('id', $id)->first();
    }

    public static function getSingleSite($id)
    {
        return self::where('id', $id)->first()->toArray();
    }

    public static function getSiteByHash($hash)
    {
        return self::where('hash', $hash)->first();
    }

    public static function getSiteByName($name)
    {
        return self::where('name', $name)->where('status', 1)->first();
    }

    public static function siteExists($site)
    {
        return self::where('name', $site)->where('status', 1)->count();
    }

    public static function getRecentWebsites($limit = 10, $remember = 0)
    {
        $recentWebsites = self::where('publish', 1)->where('status', 1)->orderBy('id', 'desc')->take($limit);
        if($remember){
            $recentWebsites->remember($remember);
        }
        return $recentWebsites->get();
    }

    public static function getWebsitesByCountry($countryCode, $remember = 0)
    {
        $query = self::where('publish', 1)->where('status', 1)->where('country_code', $countryCode);

        if($remember){
            $query->remember($remember);
        }

        return $query;
    }

    public static function getAllWithoutCached()
    {
        return self::where('publish', 1)->where('status', 1)->where('es_cache', 0)->get();
    }

    public static function getSitesByIpAddress($ipAddress)
    {
        return self::where('ip_address', $ipAddress)->get(['id', 'name', 'country_name', 'country_code', 'ip_address', 'google_rank', 'alexa_rank'])->toArray();
    }

    public static function getSitesByAdsense($adsenseId)
    {
        return self::where('google_adsense_id', $adsenseId)->get(['id', 'name', 'country_name', 'country_code', 'ip_address', 'google_rank', 'alexa_rank'])->toArray();
    }

    public static function getSitesByCClassIpAddress($ipAddress)
    {
        $cClassIp = explode('.', $ipAddress);
        array_pop($cClassIp);
        $cClassIp = implode('.', $cClassIp);

        return self::where('ip_address', '!=', $ipAddress)->where('ip_address', 'LIKE', $cClassIp . '%')->get(['id', 'name', 'country_name', 'country_code', 'ip_address', 'google_rank', 'alexa_rank'])->toArray();
    }

    public static function getWebSiteCount($countryCode)
    {
        return self::where('status', 1)->where('publish', 1)->where('country_code', $countryCode)->count();
    }

    public static function getTotalCount()
    {
        return self::where('status', 1)->where('publish', 1)->count();
    }

    public static function getWebsiteCountToday()
    {
        return self::where('status', 1)->where('publish', 1)->where('created_at', '>=', Carbon::today()->startOfDay())->count();
    }




}
