<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Cviebrock\EloquentSluggable\Sluggable;
use Cache;

class Image extends Model
{
    use SoftDeletes;
    use Sluggable;

    protected $fillable = ['album_id', 'slug', 'title', 'description', 'hash', 'user_id', 'extension', 'order', 'privacy', 'deleted_by', 'deleted_reason'];

    protected $dates = [
        'created_at',
        'deleted_at',
        'updated_at',
    ];

    public function sluggable()
    {
        return [
            'slug' => [
                'source' => 'title'
            ]
        ];
    }

    public static function boot()
    {
        parent::boot();

        static::deleting(function ($model) {
            echo "deleting image: ".$model->hash;
            Cache::forget('image_'.$model->hash);
            Cache::forget('thumb_'.$model->hash);
            $filename =  $model->path .'/'. $model->hash .'.'. $model->extension;
            $thumbnail =  $model->path .'/'.'/thumb/'. $filename;
            @unlink(public_path($filename));
            @unlink(public_path($thumbnail));
        });
   }

    public function album()
    {
        return $this->belongsTo(Album::class, 'album_id');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function deletedBy()
    {
        return $this->belongsTo(User::class, 'deleted_by');
    }

    public function scopePublic($query)
    {
        return $query->where('privacy', '=', false);
    }

    public function createdAt()
    {
        $created_at = $this->created_at->diffForHumans();
        $created_at = str_replace([' seconds', ' second'], ' sec', $created_at);
        $created_at = str_replace([' minutes', ' minute'], ' min', $created_at);
        $created_at = str_replace([' hours', ' hour'], ' h', $created_at);
        $created_at = str_replace([' months', ' month'], ' m', $created_at);

        if(preg_match('(years|year)', $created_at)){
            $created_at = $this->created_at->toFormattedDateString();
        }

        return $created_at;
    }

}
