<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Cviebrock\EloquentSluggable\Sluggable;
use Cache;

class Album extends Model
{
    use Sluggable;
    use SoftDeletes;
    use Votable;
    use Reportable;

    protected $table = 'albums';

    protected $fillable = ['title', 'hash', 'slug', 'user_id', 'privacy', 'mature', 'published', 'published_at', 'unpublished_by', 'deleted_by', 'reason'];

    protected $dates = [
        'created_at',
        'deleted_at',
        'updated_at',
        'published_at',
        'unpublished_at',
    ];

    protected $casts = [
        'published' => 'boolean',
        'mature' => 'boolean',
    ];

    public function sluggable()
    {
        return [
            'slug' => [
                'source' => 'title'
            ]
        ];
    }

    public static function boot()
    {
        parent::boot();

        static::deleting(function ($model) {
            foreach ($model->images()->get() as $image) {
                Cache::forget('image_'.$image->$hash);
                Cache::forget('thumb_'.$image->$hash);
                $image->reason = $model->reason;
                $image->deleted_by = $model->deleted_by;
                $image->save();
                $filename =  $image->path .'/'. $image->hash .'.'. $image->extension;
                $thumbnail =  $image->path .'/'.'/thumb/'. $filename;
                @unlink(public_path($filename));
                @unlink(public_path($thumbnail));
            }
            $model->images()->delete();
            $model->comments()->delete();
        });

        // static::restoring(function ($model) {
        //     echo "restoring...";
        // });

        // static::restored(function ($model) {
        //     echo "restored...";
        // });
   }

    public function creator(){
        return $this->belongsTo(User::class, 'user_id');
    }

    public function deletedBy()
    {
        return $this->belongsTo(User::class, 'deleted_by');
    }

    public function unpublishedBy()
    {
        return $this->belongsTo(User::class, 'unpublished_by');
    }

    public function isFavoritedBy(User $user)
    {
        return $user->favorites()->where('album_id', $this->id)->exists();
    }

    public function tag()
    {
        return $this->belongsTo(Tag::class);
    }

    public function images()
    {
        return $this->hasMany(Image::class);
    }

    public function comments()
    {
        return $this->hasMany(Comment::class);
    }

    public function scopePublished($query)
    {
        return $query->where('published', '=', true);
    }

    public function scopePublic($query)
    {
        return $query->where('privacy', '=', false);
    }

    public function publishedAt()
    {
        $created_at = $this->published_at->diffForHumans();
        $created_at = str_replace([' seconds', ' second'], ' sec', $created_at);
        $created_at = str_replace([' minutes', ' minute'], ' min', $created_at);
        $created_at = str_replace([' hours', ' hour'], ' h', $created_at);
        $created_at = str_replace([' months', ' month'], ' m', $created_at);

        if(preg_match('(years|year)', $created_at)){
            $created_at = $this->created_at->toFormattedDateString();
        }

        return $created_at;
    }

    public function createdAt()
    {
        $created_at = $this->created_at->diffForHumans();
        $created_at = str_replace([' seconds', ' second'], ' sec', $created_at);
        $created_at = str_replace([' minutes', ' minute'], ' min', $created_at);
        $created_at = str_replace([' hours', ' hour'], ' h', $created_at);
        $created_at = str_replace([' months', ' month'], ' m', $created_at);

        if(preg_match('(years|year)', $created_at)){
            $created_at = $this->created_at->toFormattedDateString();
        }

        return $created_at;
    }
}
