<?php
class NewsController extends AppController 
{
    public $name = 'News';
    public $uses = array('News', 'NewsComment', 'User', 'Profile', 'LogUtil');
    public $components = array('Activity', 'Notification');
    
    public function beforeFilter() 
    {
        parent::beforeFilter();
    }
    
    public function isAuthorized($user) 
    {
        //If member hasn't filled in profile redirect to Edit Profile
        if (!$this->User->access($this->Auth->user('id'))) {
            $this->Session->setFlash(__('You must first complete your profile.'), 
                'default', array('class' => 'alert alert-danger'));
            return $this->redirect(array('controller' => 'users', 'action' => 'edit', $this->Auth->user('id')));
        }
        
        //All members have access to these actions
        if (in_array($this->action, array('index', 'add'))) {
            return true;
        }
        
        //If item is for administrators only, only administrators have access to these actions. Else all members have access to these actions
        if (in_array($this->action, array('view'))) {
            $itemId = (int) $this->request->params['pass'][0];
            if ($this->Auth->user('role') == 'user') {
                $conditions = array(
                    'News.id' => $itemId,
                    'News.admin' => '0',
                    'News.deleted' => 0
                ); 
            } else {
                $conditions = array(
                    'News.id' => $itemId,
                    'News.deleted' => 0
                ); 
            }
            $item = $this->News->find('first', array(
                'conditions' => $conditions
            ));
            if (!empty($item)) {
                return true;
            }
        }
        
        //All members have access to these actions if the item is their own
        if (in_array($this->action, array('edit', 'delete'))) {
            $itemId = (int) $this->request->params['pass'][0];
            if ($this->News->isOwnedBy($itemId, $user['id'])) {
                return true;
            }
        }
        
        if ($this->action === 'comment') {
            //All members have access to this action if the comment is their own
            if (!empty($this->request->data['NewsComment']['id'])) {
                // Update comment
                $itemId = (int) $this->request->data['NewsComment']['id'];
                if ($this->NewsComment->isOwnedBy($itemId, $user['id'])) {
                    return true;
                }
            } else {
                //All members have access to this action
                return true;
            }
        }
        
        //All members have access to these actions if the comment is their own
        if ($this->action === 'comment_delete') {
            $itemId = (int) $this->request->params['pass'][0];
            if ($this->NewsComment->isOwnedBy($itemId, $user['id'])) {
                return true;
            }
        }
    
        return parent::isAuthorized($user);
    }

    public function index() 
    {
        $this->News->Behaviors->load('Containable');
        
        //Get all news based on member role (administrator/user)
        if ($this->Auth->user('role') == 'user') {
            $conditions = array(
                'News.admin' => '0',
                'News.deleted' => 0
            ); 
        } else {
            $conditions = array(
                'News.deleted' => 0
            ); 
        }
        if (Configure::read('CFG.users_follow')) {
            $user = $this->User->find('first', array(
                    'conditions' => array('User.id' => $this->Auth->user('id'), 'User.deleted' => 0),
                )
            );            
            $user_id = $user['UserFollow'];
            $user_id[] = $this->Auth->user('id');
            
            $conditions['AND'] = array(
                'News.user_id' => $user_id,
            );
        }
        $items = $this->News->find('all', array(
            'conditions' => $conditions,
            'order' => array('News.created DESC'),
            'contain' => array(
                'User' => array(
                    'Profile',
                ),
                'NewsComment'
            ),
            'recursive' => 1
            )
        );
        
        App::uses('Pagination', 'Lib');
        $paginate = new Pagination($items, 12);
        if (isset($this->request->query['p']) && empty($paginate->data)) {
            return $this->redirect(array('action' => 'index'));
        }
        
        $this->set('items', $paginate);
        $this->set('title_for_layout', __('News'));
    }
    
    public function view($id = null) 
    {
        if (!$id) {
            return $this->redirect(array('action' => 'index'));
        }
        
        $this->News->Behaviors->load('Containable');
        $item = $this->News->find('first', array(
                'conditions' => array('News.id' => $id, 'News.deleted' => 0),
                'contain' => array(
                    'User' => array(
                        'Profile',
                    ),
                    'NewsComment' => array(
                        'User' => array(
                            'Profile',
                        ),
                    ),
                ),
            )
        );
        if (!$item) {
            return $this->redirect(array('action' => 'index'));
        }
        $this->set('item', $item);
        $this->set('title_for_layout', $item['News']['title']);
    }
    
    public function add() 
    {
        //If data submitted save data
        if ($this->request->is('post')) {
            $this->request->data['News']['user_id'] = $this->Auth->user('id');
            if ((isset($this->request->data['News']['admin'])) && ($this->Auth->user('role') == 'user')) {
                //security check
                $this->request->data['News']['admin'] = 0;
            }
            if ($this->News->save($this->request->data)) {
                $this->Activity->log($this->News->id, 'News', 'add');
                
                // Notification
                if (isset($this->request->data['News']['notify']) && $this->request->data['News']['notify']) {           
                    $view_vars = array(
                        'title' => $this->request->data['News']['title'],
                        'url' => Configure::read('CFG.site_url') . '/news/view/' . $this->News->id,
                    );
                    $this->Notification->push(
                        'item_add',
                        null,
                        $view_vars,
                        (isset($this->request->data['News']['admin']) ? true : false)
                    );
                }
            
                $this->Session->setFlash(__('News successfully saved'), 
                    'default', array('class' => 'alert alert-success'));
                return $this->redirect(array('action' => 'index'));
            }
            $this->Session->setFlash(__('An error has occurred'), 
                'default', array('class' => 'alert alert-danger'));
        }
        $this->set('title_for_layout', __('Add News'));
    }
    
    public function edit($id = null) 
    {
        if (!$id) {
            return $this->redirect(array('action' => 'index'));
        }
        
        //If data submitted save data
        if ($this->request->is('post')) {
            $this->request->data['News']['id'] = $id;
            if ((isset($this->request->data['News']['admin'])) && ($this->Auth->user('role') == 'user')) {
                //security check
                $this->request->data['News']['admin'] = 0;
            }
            if ($this->News->save($this->request->data)) {
                $this->Activity->log($this->News->id, 'News', 'edit');
                
                $this->Session->setFlash(__('News successfully saved'), 
                    'default', array('class' => 'alert alert-success'));
                return $this->redirect(array('action' => 'view', $id));
            }
            $this->Session->setFlash(__('An error has occurred'), 
                'default', array('class' => 'alert alert-danger'));
            return $this->redirect(array('action' => 'index'));
        }
        
        //Find news to edit
        $this->News->Behaviors->load('Containable');
        $item = $this->News->find('first', array(
                'conditions' => array('News.id' => $id, 'News.deleted' => 0),
                'contain' => array(
                    'User' => array(
                        'Profile',
                    ),
                    'NewsComment' => array(
                        'User' => array(
                            'Profile',
                        ),
                    ),
                ),
            )
        );
        if (!$item) {
            return $this->redirect(array('action' => 'index'));
        }
        
        $this->set('item', $item);
        $this->set('title_for_layout', __('Edit News'));
    }
    
    public function delete($id = null) 
    {
        $this->layout = false;
        $this->render = false;
        
        //Check if item exists; Else redirect to index
        if (!$id) {
            return $this->redirect(array('action' => 'index'));
        }
        $item = $this->News->find('first', array(
                'conditions' => array('News.id' => $id, 'News.deleted' => 0),
            )
        );
        if (!$item) {
            return $this->redirect(array('action' => 'index'));
        }
        
        //Set deleted field to 1 for news
        $this->request->data['News']['id'] = $id;
        $this->request->data['News']['deleted'] = 1;
        if ($this->News->save($this->request->data)) {
            $this->Activity->log($this->News->id, 'News', 'delete');
            $this->Session->setFlash(__('News successfully deleted'), 
                'default', array('class' => 'alert alert-success'));
            return $this->redirect(array('action' => 'index'));
        }
        $this->Session->setFlash(__('An error has occurred'), 
            'default', array('class' => 'alert alert-danger'));
        return $this->redirect(array('action' => 'index'));
    }
    
    public function comment($id = null) 
    {
        $this->layout = false;
        $this->render = false;
        
        if (empty($id)) {
            return $this->redirect(array('action' => 'index'));
        }
        
        //If data submitted save data
        if ($this->request->is('post')) {
            if (!empty($this->request->data['NewsComment']['id'])) {
                // Update comment
                $action = 'edit';
                $item_comment = $this->NewsComment->find('first', array(
                        'conditions' => array(
                            'NewsComment.id' => $this->request->data['NewsComment']['id'], 
                            'NewsComment.deleted' => 0
                        ),
                    )
                );
                if ((empty($item_comment)) || ($this->Auth->user('role') != 'admin')) {
                    // Comment is not created by logged in user and logged in user is not an admin => abort
                    return $this->redirect(array('action' => 'index'));
                }
            } else {
                // Add comment
                $action = 'add';
                $this->request->data['NewsComment']['news_id'] = $id;
                $this->request->data['NewsComment']['user_id'] = $this->Auth->user('id');
            }
            if ($this->NewsComment->save($this->request->data)) {
                $this->Activity->log($this->NewsComment->id, 'NewsComment', $action);
                
                $this->Session->setFlash(__('Comment successfully saved'), 
                    'default', array('class' => 'alert alert-success'));
                return $this->redirect(array('action' => 'view', $id));
            }
            
        }
        $this->Session->setFlash(__('An error has occurred'), 
            'default', array('class' => 'alert alert-danger'));
        return $this->redirect(array('action' => 'index'));
    }
    
    public function comment_delete($id = null) 
    {
        $this->layout = false;
        $this->render = false;
        
        //Check if comment exists; Else redirect to index
        if (!$id) {
            return $this->redirect(array('action' => 'index'));
        }
        $item_comment = $this->NewsComment->find('first', array(
                'conditions' => array('NewsComment.id' => $id)
            )
        );
        if (!$item_comment) {
            return $this->redirect(array('action' => 'index'));
        }
        
        //Set deleted field to 1 for comment
        $this->request->data['NewsComment']['id'] = $id;
        $this->request->data['NewsComment']['deleted'] = 1;
        if ($this->NewsComment->save($this->request->data)) {
            $this->Activity->log($this->NewsComment->id, 'NewsComment', 'delete');
            
            $this->Session->setFlash(__('Comment successfully deleted'), 
                'default', array('class' => 'alert alert-success'));
            return $this->redirect(array('action' => 'view', $item_comment['NewsComment']['news_id']));
        }
        $this->Session->setFlash(__('An error has occurred'), 
            'default', array('class' => 'alert alert-danger'));
        return $this->redirect(array('action' => 'index'));
    }
}
