<?php
class FilesController extends AppController 
{
    public $name = 'Files';
    public $uses = array('File', 'FileComment', 'User', 'Profile');
    public $components = array('Activity', 'Notification');

    public function beforeFilter() 
    {
        parent::beforeFilter();
    }
    
    public function isAuthorized($user) 
    {
        //If member hasn't filled in profile redirect to Edit Profile
        if (!$this->User->access($this->Auth->user('id'))) {
            $this->Session->setFlash(__('You must first complete your profile.'), 
                'default', array('class' => 'alert alert-danger'));
            return $this->redirect(array('controller' => 'users', 'action' => 'edit', $this->Auth->user('id')));
        }
        
        //All members have access to these actions
        if (in_array($this->action, array('index', 'add'))) {
            return true;
        }
        
        //If item is for administrators only, only administrators have access to these actions. Else all members have access to these actions
        if (in_array($this->action, array('view'))) {
            $itemId = (int) $this->request->params['pass'][0];
            if ($this->Auth->user('role') == 'user') {
                $conditions = array(
                    'File.id' => $itemId,
                    'File.admin' => '0',
                    'File.deleted' => 0
                ); 
            } else {
                $conditions = array(
                    'File.id' => $itemId,
                    'File.deleted' => 0
                ); 
            }
            $item = $this->File->find('first', array(
                'conditions' => $conditions
            ));
            if (!empty($item)) {
                return true;
            }
        }
    
        //All members have access to these actions if the item is their own
        if (in_array($this->action, array('edit', 'delete'))) {
            $itemId = (int) $this->request->params['pass'][0];
            if ($this->File->isOwnedBy($itemId, $user['id'])) {
                return true;
            }
        }
        
        if ($this->action === 'comment') {
            //All members have access to this action if the comment is their own
            if (!empty($this->request->data['FileComment']['id'])) {
                // Update comment
                $itemId = (int) $this->request->data['FileComment']['id'];
                if ($this->FileComment->isOwnedBy($itemId, $user['id'])) {
                    return true;
                }
            } else {
                //All members have access to this action
                return true;
            }
        }
        
        //All members have access to these actions if the comment is their own
        if ($this->action === 'comment_delete') {
            $itemId = (int) $this->request->params['pass'][0];
            if ($this->FileComment->isOwnedBy($itemId, $user['id'])) {
                return true;
            }
        }
    
        return parent::isAuthorized($user);
    }

    public function index() 
    {
        $this->File->Behaviors->load('Containable');
        
        //Get all files based on member role (administrator/user)
        if ($this->Auth->user('role') == 'user') {
            $conditions = array(
                'File.admin' => '0',
                'File.deleted' => 0
            ); 
        } else {
            $conditions = array(
                'File.deleted' => 0
            ); 
        }
        if (Configure::read('CFG.users_follow')) {
            $user = $this->User->find('first', array(
                    'conditions' => array('User.id' => $this->Auth->user('id'), 'User.deleted' => 0),
                )
            );            
            $user_id = $user['UserFollow'];
            $user_id[] = $this->Auth->user('id');
            
            $conditions['AND'] = array(
                'File.user_id' => $user_id,
            );
        }
        $items = $this->File->find('all', array(
            'conditions' => $conditions,
            'order' => array('File.created DESC'),
            'contain' => array(
                'User' => array(
                    'Profile',
                ),
                'FileAttachment',
                'FileComment'
            ),
            'recursive' => 1
            )
        );
        
        App::uses('Pagination', 'Lib');
        $paginate = new Pagination($items, 12);
        if (isset($this->request->query['p']) && empty($paginate->data)) {
            return $this->redirect(array('action' => 'index'));
        }
        
        $this->set('items', $paginate);
        $this->set('title_for_layout', __('Files'));
    }
    
    public function view($id = null) 
    {
        if (empty($id)) {
            return $this->redirect(array('action' => 'index'));
        }
        
        $this->File->Behaviors->load('Containable');
        $item = $this->File->find('first', array(
                'conditions' => array('File.id' => $id, 'File.deleted' => 0),
                'contain' => array(
                    'User' => array(
                        'Profile',
                    ),
                    'FileAttachment',
                    'FileComment' => array(
                        'User' => array(
                            'Profile',
                        ),
                    ),
                ),
            )
        );
        if (!$item) {
            return $this->redirect(array('action' => 'index'));
        }
        $this->set('item', $item);
        $this->set('title_for_layout', $item['File']['title']);
    }
    
    public function add() 
    {
        //If data submitted save data
        if ($this->request->is('post')) {
            $this->request->data['File']['user_id'] = $this->Auth->user('id');
            if ((isset($this->request->data['File']['admin'])) && ($this->Auth->user('role') == 'user')) {
                //security check
                $this->request->data['File']['admin'] = 0;
            }
            
            //in no file selected unset file attachment so no row gets inserted
            if (isset($this->request->data['FileAttachment'][0]) && $this->request->data['FileAttachment'][0]['file']['error'] != 0) {
                unset($this->request->data['FileAttachment'][0]);
            }
            
            if ($this->File->saveAll($this->request->data)) {
                $this->Activity->log($this->File->id, 'File', 'add');
                
                // Notification
                if (isset($this->request->data['File']['notify']) && $this->request->data['File']['notify']) {           
                    $view_vars = array(
                        'title' => $this->request->data['File']['title'],
                        'url' => Configure::read('CFG.site_url') . '/files/view/' . $this->File->id,
                    );
                    $this->Notification->push(
                        'item_add',
                        null,
                        $view_vars,
                        (isset($this->request->data['File']['admin']) ? true : false)
                    );
                }
            
                $this->Session->setFlash(__('File successfully saved'), 
                    'default', array('class' => 'alert alert-success'));
                return $this->redirect(array('action' => 'index'));
            }
            $this->Session->setFlash(__('An error has occurred'), 
                'default', array('class' => 'alert alert-danger'));
        }
        $this->set('title_for_layout', __('Add File'));
    }
    
    public function edit($id = null) 
    {
        if (empty($id)) {
            return $this->redirect(array('action' => 'index'));
        }
        
        //If data submitted save data
        if ($this->request->is('post')) {
            $this->request->data['File']['id'] = $id;
            if ((isset($this->request->data['File']['admin'])) && ($this->Auth->user('role') == 'user')) {
                //security check
                $this->request->data['File']['admin'] = 0;
            }
            
            //in no file selected unset file attachment so no row gets inserted
            if (isset($this->request->data['FileAttachment'][0]) && $this->request->data['FileAttachment'][0]['file']['error'] != 0) {
                unset($this->request->data['FileAttachment'][0]);
            }
            
            if ($this->File->saveAll($this->request->data)) {
                $this->Activity->log($this->File->id, 'File', 'edit');
                
                $this->Session->setFlash(__('File successfully saved'), 
                    'default', array('class' => 'alert alert-success'));
                return $this->redirect(array('action' => 'view', $id));
            }
            $this->Session->setFlash(__('An error has occurred'), 
                'default', array('class' => 'alert alert-danger'));
            return $this->redirect(array('action' => 'index'));
        }
        
        //Find file to edit
        $this->File->Behaviors->load('Containable');
        $item = $this->File->find('first', array(
                'conditions' => array('File.id' => $id, 'File.deleted' => 0),
                'contain' => array(
                    'User' => array(
                        'Profile',
                    ),
                    'FileAttachment',
                    'FileComment' => array(
                        'User' => array(
                            'Profile',
                        ),
                    ),
                ),
            )
        );
        $this->set('item', $item);
        $this->set('title_for_layout', __('Edit File'));
    }
    
    public function delete($id = null) 
    {
        $this->layout = false;
        $this->render = false;
        
        //Check if item exists; Else redirect to index
        if (!$id) {
            return $this->redirect(array('action' => 'index'));
        }
        $item = $this->File->find('first', array(
                'conditions' => array('File.id' => $id, 'File.deleted' => 0),
            )
        );
        if (!$item) {
            return $this->redirect(array('action' => 'index'));
        }
        
        //Set deleted field to 1 for file
        $this->request->data['File']['id'] = $id;
        $this->request->data['File']['deleted'] = 1;
        if ($this->File->save($this->request->data)) {
            $this->Activity->log($this->File->id, 'File', 'delete');
            
            $this->Session->setFlash(__('File successfully deleted'), 
                'default', array('class' => 'alert alert-success'));
            return $this->redirect(array('action' => 'index'));
        }
        $this->Session->setFlash(__('An error has occurred'), 
            'default', array('class' => 'alert alert-danger'));
        return $this->redirect(array('action' => 'index'));
    }
    
    public function comment($id = null) 
    {
        $this->layout = false;
        $this->render = false;
        
        if (empty($id)) {
            return $this->redirect(array('action' => 'index'));
        }
        
        //If data submitted save data
        if ($this->request->is('post')) {
            if (!empty($this->request->data['FileComment']['id'])) {
                // Update comment
                $action = 'edit';
                $item_comment = $this->FileComment->find('first', array(
                        'conditions' => array(
                            'FileComment.id' => $this->request->data['FileComment']['id'], 
                            'FileComment.deleted' => 0
                        ),
                    )
                );
                if ((empty($item_comment)) || ($this->Auth->user('role') != 'admin')) {
                    // Comment is not created by logged in user and logged in user is not an admin => abort
                    return $this->redirect(array('action' => 'index'));
                }
            } else {
                // Add comment
                $action = 'add';
                $this->request->data['FileComment']['file_id'] = $id;
                $this->request->data['FileComment']['user_id'] = $this->Auth->user('id');
            }
            if ($this->FileComment->save($this->request->data)) {
                $this->Activity->log($this->FileComment->id, 'FileComment', $action);
                
                $this->Session->setFlash(__('Comment successfully saved'), 
                    'default', array('class' => 'alert alert-success'));
                return $this->redirect(array('action' => 'view', $id));
            }
            
        }
        $this->Session->setFlash(__('An error has occurred'), 
            'default', array('class' => 'alert alert-danger'));
        return $this->redirect(array('action' => 'index'));
    }
    
    public function comment_delete($id = null) 
    {
        $this->layout = false;
        $this->render = false;
        
        //Check if comment exists; Else redirect to index
        if (!$id) {
            return $this->redirect(array('action' => 'index'));
        }
        $item_comment = $this->FileComment->find('first', array(
                'conditions' => array('FileComment.id' => $id)
            )
        );
        if (!$item_comment) {
            return $this->redirect(array('action' => 'index'));
        }
        
        //Set deleted field to 1 for comment
        $this->request->data['FileComment']['id'] = $id;
        $this->request->data['FileComment']['deleted'] = 1;
        if ($this->FileComment->save($this->request->data)) {
            $this->Activity->log($this->FileComment->id, 'FileComment', 'delete');
            
            $this->Session->setFlash(__('Comment successfully deleted'), 
                'default', array('class' => 'alert alert-success'));
            return $this->redirect(array('action' => 'view', $item_comment['FileComment']['file_id']));
        }
        $this->Session->setFlash(__('An error has occurred'), 
            'default', array('class' => 'alert alert-danger'));
        return $this->redirect(array('action' => 'index'));
    }
}
