<?php

if(!defined('ABSPATH')) die('-1');

class AwesomeLiveChat_Frontend extends AwesomeLiveChat
{
    /**
     * Boots object
     *
     * @since  1.0.0
     * @access private
     * @var object
     */
    private $Boots;

    /**
     * Settings/Options
     *
     * @since  1.0.0
     * @access private
     * @var array
     */
    private $Options;

    /**
     * Library
     *
     * @since  1.0.0
     * @access private
     * @var object
     */
    private $_;

    /**
     * JavaScript Variables
     *
     * @since  1.0.0
     * @static
     * @access private
     * @var array
     */
    static private $Vars = array();

    /**
     * place
     *
     * @since  1.0.0
     * @access private
     * @var bool
     */
    private $place;

    /**
     * Get the Boots framework object.
     * Get the plugin settings.
     *
     * @since  1.0.0
     * @uses   AwesomeLiveChat::boots()
     *         return boots object
     * @uses   AwesomeLiveChat::options()
     *         return options
     * @uses   AwesomeLiveChat_Library
     *         plugin library/helper
     * @access public
     * @param  void
     * @return void
     */
    public function __construct()
    {
        $this->Boots   = $this->boots();
        $this->Options = $this->options();
        $this->_       = new AwesomeLiveChat_Library();
        $this->place   = null;

        add_action('init', array($this, 'setVars'));

        if($this->_->settings('general', 'iframe') == 'no')
        {
            add_action('wp_enqueue_scripts', array($this, 'enqueue'));
            add_action('wp_footer', array($this, 'html'));
        }
        else
        {
            add_action('wp_enqueue_scripts', array($this, 'ienqueue'));
            add_action('wp_footer', array($this, 'iframe'));
        }
    }

    /**
     * Set javascript variables.
     *
     * @since  1.0.0
     * @uses   wp_create_nonce() security nonce.
     * @access public
     * @param  void
     * @return bool   Exit if already set.
     */
    public function setVars()
    {
        if(self::$Vars) return false;

        $Fonts = array();
        $font_header = $this->_->settings('skin', 'font_header');
        if($font_header) $Fonts[] = $font_header;
        $font_body = $this->_->settings('skin', 'font_body');
        if($font_body) $Fonts[] = $font_body;
        $Fonts = array_unique($Fonts);
        $timeout = ini_get('max_execution_time');
        $timeout = $timeout ? ($timeout * 1000) : 30000;

        $Socket = false;
        if($this->_->settings('socket', 'socket') == 'yes')
        {
            $Socket['url'] = $this->_->settings('socket', 'host');
            $Socket['port'] = $this->_->settings('socket', 'port', false);
        }

        $Optional = array();
        if($this->_->isOnline() && $this->_->settings('general', 'email_field') == 'optional')
            $Optional[] = 'email';

        self::$Vars = array(
            'fonts'   => $Fonts,
            'nonce'   => wp_create_nonce('awesome-live-chat'),
            'ajaxurl' => $this->Options['ALC_API_URL_VISITOR'],
            'timeout' => $timeout,
            'socket'  => $Socket,
            'responsive' => $this->_->settings('general', 'responsive'),
            'strings' => array(
                'status_online' => $this->_->settings('strings', 'status_online'),
                'status_offline' => $this->_->settings('strings', 'status_offline'),
                'status_wait' => $this->_->settings('strings', 'status_wait'),
                'status_start' => $this->_->settings('strings', 'status_start')
            ),
            'meta' => array(
                'operator' => $this->_->settings('strings', 'meta_operator'),
                'visitor' => $this->_->settings('strings', 'meta_visitor')
            ),
            'optional' => $Optional,
            'iframe' => $this->_->settings('general', 'iframe')
        );
    }

    /**
     * Get javascript variables.
     *
     * @since  1.0.0
     * @static
     * @access public
     * @param  void
     * @return array  JavaScript variables.
     */
    public static function getVars()
    {
        return self::$Vars;
    }

    private function show()
    {
        if($this->place !== null) return $this->place;
        if(
            !$this->_->isOnline() &&
            $this->_->settings('general', 'show_offline') == 'no'
        ) return false;

        if(is_multisite() && get_current_blog_id() != 1)
        {
            $License = $this->Boots->Database
                ->term('alc_license', array())
                ->get();
            if(!isset($License['verified']) || !$License['verified'])
                return false;
        }

        $Block = explode(',', $this->_->settings('block'));
        if(in_array($_SERVER['REMOTE_ADDR'], $Block))
            return $this->place;

        $Placement = $this->_->settings('general', 'placement');

        if(is_archive())
        {
            $this->place = in_array('archive', $Placement)
                ? true
                : false;
        }
        else if(is_front_page())
        {
            $this->place = in_array('home', $Placement)
                ? true
                : false;
        }
        else if(is_singular())
        {
            $Post = get_post();
            if(!$Post) return false;
            $id = $Post->ID;

            $show = $this->Boots->Database
            ->term('_alc_show')
            ->get($id);
            if(in_array($show, array('yes', 'no')))
            {
                $this->place = $show == 'yes'
                    ? true
                    : false;
            }
            else
            {
                $this->place = in_array($Post->post_type, $Placement)
                    ? true
                    : false;
            }
        }

        return $this->place;
    }

    /**
     * Enqueue frontend css.
     *
     * @since  1.0.0
     * @uses   Boots::Enqueue Enqueue css
     * @access private
     * @param  void
     * @return void
     */
    private function css()
    {
        $this->Boots->Enqueue
        ->style('font-awesome')
            ->source('css/font-awesome.min.css')
            ->version('4.1.0')
            ->done()
        ->style('awesome-live-chat')
            ->source('frontend/css/awesome-live-chat.min.css')
            ->requires('font-awesome')
            ->done()
        ->style('awesome-live-chat-custom')
            ->source('frontend/css/alc.css.php')
            ->requires('awesome-live-chat')
            ->done();
    }

    /**
     * Enqueue frontend javascript.
     *
     * @since  1.0.0
     * @uses   Boots::Enqueue Enqueue js
     * @access private
     * @param  void
     * @return void
     */
    private function js()
    {
        $Socket = false;
        if($this->_->settings('socket', 'socket') == 'yes')
        {
            $Socket = array();
            $Socket['url'] = $this->_->settings('socket', 'host');
            $Socket['port'] = $this->_->settings('socket', 'port', false);
        }
        $this->Boots->Enqueue
        ->script('backbone')->done(true)
        ->script('bootstrap')
            ->source('js/bootstrap.min.js')
            ->version('3.3.1')
            ->done(true);
        if($Socket)
        $this->Boots->Enqueue
        ->raw_script('socket.io')
            ->source(rtrim($Socket['url'], '/')
                    . (empty($Socket['port']) ? '' : (':' . $Socket['port']))
                    . '/socket.io/socket.io.js')
            ->version('1.2.1')
            ->done(true);
        $this->Boots->Enqueue
        ->script('Autolinker')
            ->source('js/Autolinker.min.js')
            ->version('0.15.2')
            ->done(true)
        ->script('awesome-live-chat')
            ->source('frontend/js/awesome-live-chat.min.js')
            ->requires('backbone')
            ->requires('jquery-effects-core')
            ->requires('jquery-effects-drop')
            ->requires('bootstrap')
            ->requires('Autolinker');
        if($Socket)
        $this->Boots->Enqueue
            ->requires('socket.io');
        $this->Boots->Enqueue
            ->vars($this->getVars())
            ->done(true)
        ->script('awesome-live-chat-custom')
            ->source('frontend/js/alc.js.php')
            ->requires('awesome-live-chat')
            ->done(true);
    }

    /**
     * Enqueue for frontend
     *
     * @since  1.0.0
     * @uses   AwesomeLiveChat_Frontend::css()
     *         Enqueue css
     * @uses   AwesomeLiveChat_Frontend::js()
     *         Enqueue js
     * @access public
     * @param  void
     * @return void
     */
    public function enqueue()
    {
        if(!$this->show()) return false;
        $this->css();
        $this->js();
    }

    public function html()
    {
        if(!$this->show()) return false;
        include $this->Options['APP_DIR'] . '/frontend/chat.php';
    }

    public function ienqueue()
    {
        if(!$this->show()) return false;
        $this->Boots->Enqueue
        ->script('iframeResize')
            ->source('js/iframeResizer.min.js')
            ->version('2.5.0')
            ->done(true)
        ->script('iFrameResize-alc')
            ->source('frontend/js/iframe.js')
            ->requires('iframeResize')
            ->done(true);
    }

    public function iframe()
    {
        if(!$this->show()) return false;
        $url = $this->Options['APP_URL'] . '/frontend/frame.php';
        $position = $this->_->settings('general', 'position');
        echo '<iframe
            id="awesome_live_chat_frame"
            src="' . $url . '"
            frameborder="0"
            scrolling="no"
            style="position: fixed;
            bottom: 0;
            ' . ($position == 'left' ? 'left: 0;' : 'right: 0;' ) . '
            margin: 0;
            z-index: 999999;
            width: auto;"></iframe>
        ';
    }
}

new AwesomeLiveChat_Frontend();
