<?php 

namespace app;

use Illuminate\Database\Eloquent\Model;

class Site extends Model
{

    /**
     * belongs to a user
     */
    public function user()
    {
        return $this->belongsTo('App\User', 'author_id');
    }

    /**
     * Gets the LIVE hits for this site
     *
     * @return  int Number of hits
     */
    public function getLiveVisitors()
    {
        //-- Time for a live user
        $time = $time = getConfig('live_test', 10);
        //-- Return a count
        return $this->getVisitorDateRange(date("Y-m-d H:i:s", strtotime("-" . $time . " mins")), date("Y-m-d H:i:s"));
    }

    /**
     * Generates statistics for live users for this site
     *
     * @return  array
     */
    public function getLiveStats()
    {
        //-- Time for a live user
        $time = getConfig('live_test', 10);

        //-- Timezones don't matter as we are after live data, which is the same
        //-- server time for everyone
        $start    = date("Y-m-d H:i:s", strtotime("-" . $time . " mins"));
        $end    = date("Y-m-d") . " 23:59:59";

        //-- Working Code
        $output = array();
        $working = array();
        $debug_start = microtime(true);

        //-- Create array of pageviews
        for ($i = $time; $i >= 1; $i--) {
            $output['pageviews']["-" . $i . " mins"] = 0;
        }

        //-- Lets grab all rows to work out actual pages and visitors
        $raw = \DB::table('raw')->where('site_id', $this->id)->where('created_at', '>=', $start)->where('created_at', '<=', $end)->get();
        foreach ($raw as $r) {
            //-- Unique Visitors
            $working['visitors'][$r->visitor_id] = 1;
            //-- Hits per minute
            $diff = ceil((time() - strtotime($r->created_at)) / 60);
            @$output['pageviews']["-" . $diff . " mins"]++;
            @$working['actions']++;
        }

        //-- Lets grab distinct metrics 
        $working['devices']        = $this->getMetric($start, $end, 5, 'device', true, false);

        //-- Parse info
        $output['pages']        = $this->getMetric($start, $end, 5, 'page_url', true, false);
        $output['browsers']        = $this->getMetric($start, $end, 5, 'browser_name', true, false);
        $output['locations']    = $this->getMetric($start, $end, 5, 'country', true, false);
        $output['visitors']    = @count($working['visitors']);

        //-- Devices need some twiddling
        foreach ($working['devices'] as $device => $count) {
            $output['device'][$device] = @number_format(($count / $working['actions']) * 100, 0);
        }

        //-- Create our json for graphing pageviews
        if (count($output['pageviews']) > 0) {
            $json = array();

            foreach ($output['pageviews'] as $min => $amount) {
                $json[] = array( 'y' => $min, 'a' => $amount );
            }
            $output['pageviews_json'] = $json;
        }

        //-- Bit of debug
        $output['parse'] = number_format(microtime(true) - $debug_start, 5);
        return $output;
    }

    /**
     * Gets the hits for a specific date-range from the RAW table
     *
     * @param  string  Start date
     * @param  string  End date
     * @return  int Number of hits
     */
    public function getHitsDateRange($start = false, $end = false)
    {
        //-- Check for defaults, and pre fill with today
        if (strtotime($start) == 0) {
            $start = date("Y-m-d") . " 00:00:00";
        }
        if (strtotime($end) == 0) {
            $end   = date("Y-m-d") . " 23:59:59";
        }
        //-- Grab our difference in TZ
        $diff = get_timezone_offset(\Config::get('app.timezone'), $this->timezone);
        $start = mathSecondsToDate($diff, $start);
        $end = mathSecondsToDate($diff, $end);
        //-- Generate simple SQL
        $raw = \DB::table('raw')->where('site_id', $this->id)->where('created_at', '>=', $start)->where('created_at', '<=', $end)->get();
        //-- Return a count
        return count($raw);
    }

    /**
     * Gets the visitors for a specific date-range from the RAW table,
     * which is like the hits, but we look for unique visitor_ids
     *
     * @param  string  Start date
     * @param  string  End date
     * @return  int Number of hits
     */
    public function getVisitorDateRange($start = false, $end = false)
    {
        $t = $start;
        //-- Check for defaults, and pre fill with today
        if (strtotime($start) == 0) {
            $start = date("Y-m-d") . " 00:00:00";
        }
        if (strtotime($end) == 0) {
            $end   = date("Y-m-d") . " 23:59:59";
        }
        //-- Grab our difference in TZ
        $diff = get_timezone_offset(\Config::get('app.timezone'), $this->timezone);
        $start = mathSecondsToDate($diff, $start);
        $end = mathSecondsToDate($diff, $end);
        //-- Generate simple SQL
        $raw = \DB::table('raw')->where('site_id', $this->id)->where('created_at', '>=', $start)->where('created_at', '<=', $end)->groupBy('visitor_id')->get();
        //-- Return a count
        return count($raw);
    }

    /**
     * Gets an array of data based off another method
     * and a date range
     *
     * @param  string start
     * @param  string end
     * @param  string interval
     * @param  string method
     * @return array
     */
    public function arrayDatesMethod($start, $end, $interval, $method)
    {
        $dates = array_dates($start, $end, $interval);
        $output = array();
        foreach ($dates as $date) {
            $output[$date] = $this->$method($date . " 00:00:00", $date . " 23:59:59");
        }
        return $output;
    }

    /**
     * Gets the hits per visit average for a specific date range
     *
     * @param  string  Start date
     * @param  string  End date
     * @return  int Number of hits
     */
    public function getHitsPerVisit($start = false, $end = false)
    {
        //-- Generate our first stats
        $hits    = $this->getHitsDateRange($start, $end);
        $visits = $this->getVisitorDateRange($start, $end);
        //-- Catch divide by 0
        if ($visits == 0) {
            return 0;
        }
        return @number_format(($hits / $visits), 2);
    }

    /**
     * Get the total time on the site for the date-range from
     * the raw table
     *
     * @param  string  Start date
     * @param  string  End date
     * @return  int number of seconds in total
     */
    public function getTimeOnSite($start = false, $end = false)
    {
        //-- Check for defaults, and pre fill with today
        if (strtotime($start) == 0) {
            $start = date("Y-m-d") . " 00:00:00";
        }
        if (strtotime($end) == 0) {
            $end   = date("Y-m-d") . " 23:59:59";
        }
        //-- Grab our difference in TZ
        $diff = get_timezone_offset(\Config::get('app.timezone'), $this->timezone);
        $start = mathSecondsToDate($diff, $start);
        $end = mathSecondsToDate($diff, $end);
        //-- SQL
        $raw = \DB::table('raw')->where('site_id', $this->id)->where('created_at', '>=', $start)->where('created_at', '<=', $end)->sum('time_since_last_visit');
        //-- Return seconds
        return $raw;
    }

    /**
     * Get the average time on the site for the date-range from
     * the raw table
     *
     * @param  string  Start date
     * @param  string  End date
     * @return  int number of seconds in total
     */
    public function getAvgTimeOnSite($start = false, $end = false)
    {
        //-- Generate Stats
        $visits = $this->getVisitorDateRange($start, $end);
        $time = $this->getTimeOnSite($start, $end);
        //-- Catch divide by 0
        if ($visits == 0) {
            return 0;
        }
        $secs = number_format(floatval($time / $visits), 0, ".", "");
        return $secs;
    }

    /**
     * Gets a basic statistic of total amount of hits per metric 
     * that is given. Does a search of RAW.
     *
     * @param  string  Start date
     * @param  string  End date
     * @param  int  return how many results (0 = all)
     * @param  string  metric
     * @param  boolean  format as array with just metric and total hits
     * @param  boolean  tz changing
     * @return  \DB::row
     */
    public function getMetric($start = false, $end = false, $limit = 0, $metric, $format = false, $tz = true)
    {
        //-- Check for defaults, and pre fill with today
        if (strtotime($start) == 0) {
            $start = date("Y-m-d") . " 00:00:00";
        }
        if (strtotime($end) == 0) {
            $end   = date("Y-m-d") . " 23:59:59";
        }
        //-- Grab our difference in TZ
        if ($tz) {
            $diff = get_timezone_offset(\Config::get('app.timezone'), $this->timezone);
            $start = mathSecondsToDate($diff, $start);
            $end = mathSecondsToDate($diff, $end);
        }
         //-- SQL
        if ($limit == 0) {
            $raw = \DB::table('raw')->select('*', \DB::raw('COUNT(id) as total_hits'))->where('site_id', $this->id)->where('created_at', '>=', $start)->where('created_at', '<=', $end)->groupBy($metric)->orderBy('total_hits', 'DESC')->get();
        } else {
            $raw = \DB::table('raw')->select('*', \DB::raw('COUNT(id) as total_hits'))->where('site_id', $this->id)->where('created_at', '>=', $start)->where('created_at', '<=', $end)->groupBy($metric)->orderBy('total_hits', 'DESC')->take($limit)->get();
        }

        if ($format) {
            $output = array();
            foreach ($raw as $r) {
                $output[$r->$metric] = $r->total_hits;
            }
            return $output;
        } else {
            return $raw;
        }
    }

    /**
     * Gets extra information for specific db query for the data-range
     * from the raw table
     *
     * @param  string  Start date
     * @param  string  End date
     * @param  string  page_url
     * @return  array 
     */
    public function getContentStats($start = false, $end = false, $col_name, $search)
    {
        //-- Check for defaults, and pre fill with today
        if (strtotime($start) == 0) {
            $start = date("Y-m-d") . " 00:00:00";
        }
        if (strtotime($end) == 0) {
            $end   = date("Y-m-d") . " 23:59:59";
        }
        //-- Grab our difference in TZ
        $diff = get_timezone_offset(\Config::get('app.timezone'), $this->timezone);
        $start = mathSecondsToDate($diff, $start);
        $end = mathSecondsToDate($diff, $end);
        //-- Get Visitors for this date period
        $visitors = \DB::table('raw')->select('visitor_id', \DB::raw('MAX(id) as b'))->where('site_id', $this->id)->where('created_at', '>=', $start)->where('created_at', '<=', $end)->where($col_name, $search)->groupBy('visitor_id')->get();
        $visitors = count($visitors);
        $time = \DB::table('raw')->where('site_id', $this->id)->where('created_at', '>=', $start)->where('created_at', '<=', $end)->where($col_name, $search)->sum('time_since_last_visit');
        $avg = @number_format(($time / $visitors), 0, ".", "");
        //-- return
        return array('visitors' => $visitors, 'total_time' => $time, 'avg' => $avg);
    }

    /**
     * Does a check to see if this site should
     * delete historical data based off the package
     * the site sits in. If it should do, then
     * we'll go ahead and delete the raw data.
     *
     * @return void
     */
    public function deleteHistorical()
    {
        //-- Get the current package for this site
        $package = $this->user->package;

        //-- Generate date
        if ($package->history == 0) {
            //-- This package lets full historical data stay!
            return true;
        }

        //-- This package doesn't - lets work out the date from when we delete
        $diff = get_timezone_offset(\Config::get('app.timezone'), $this->timezone);
        $end = mathSecondsToDate($diff, date("Y-m-d", strtotime("-" . $package->history . " days")));

        //-- Do the delete!
        \DB::table('raw')->where('site_id', $this->id)->where('created_at', '<=', $end)->delete();
        return true;
    }
}
