<?php


/**
 * Check Request URL
 *
 * Will check the request URL, and if it matches using the
 * laravel function will echo the display var
 * 
 * @param string $check
 * @param string $display
 * @return str
*/
if (!function_exists('crurl')) {
    function crurl($check, $display = "active")
    {
        if (Request::is($check)) {
            echo $display;
        }
    }
}

/**
 * Function that will check the DB, and try and create a unique UUID
 * for the requested field.
 *
 * @param {string} $table - the table name to check
 * @param {string} $field - the field to create the UUID for
 * @param {integer} $length - Length of the UUID
 * @return {string} - The unique UUID
*/
if (!function_exists('createUniqueStrings')) {
    function createUniqueStrings($table, $field, $length = 255)
    {
        $random = str_random($length);
        //-- Check for randomness
        $check = DB::table($table)->where($field, $random)->get();
        if (count($check) > 0) {
            $this->createUniqueStrings($field, $length);
        } else {
            return $random;
        }
    }
}

/**
 * Timezones list with GMT offset
 *
 * @return array
 * @link http://stackoverflow.com/a/9328760
 */
if (!function_exists('tz_list')) {
    function tz_list()
    {
        $zones_array = array();
        $timestamp = time();
        foreach (timezone_identifiers_list() as $key => $zone) {
            date_default_timezone_set($zone);
            $zones_array[$key]['zone'] = $zone;
            $zones_array[$key]['diff_from_GMT'] = 'UTC/GMT ' . date('P', $timestamp);
        }
        return $zones_array;
    }
}

/**
 * Picks random from array
 *
 * @param {array} $array
 * @return {string} Random array element
 */
if (!function_exists('random_array')) {
    function random_array($array)
    {
        return $array[array_rand($array)];
    }
}

/** 
 * Grabs data from the session and creates 
 * time based data on it. When creating new 
 * times for the system, need to add here as 
 * well
 *
 * @param  Request
 * @return array
 *
 */
if (!function_exists('date_control_function')) {
    function date_control_function($request)
    {

        //-- Check for sessions, and grab data
        $date_data = $request->session()->get('date-control', array('start' => date("Y-m-d", strtotime("-7 days")), 'end' => date("Y-m-d")));

        //-- We'll convert to a format we can use
        $date_data['start_format'] = date("Y-m-d", strtotime($date_data['start']));
        $date_data['end_format'] = date("Y-m-d", strtotime($date_data['end'])) . " 23:59:59";

        //-- return data
        return ($date_data);
    }
}

/** 
 * Get a list of dates in array format from between
 * start and end dates
 *
 * @param  string  start
 * @param  string  end
 * @param  string  interval
 * @param  string  format
 * @return array
 */
if (!function_exists('array_dates')) {
    function array_dates($start, $end, $interval, $output_format = "Y-m-d")
    {
        $dates = array();
        $current = strtotime($start);
        $end = strtotime($end);

        while ($current <= $end) {
            $dates[] = date($output_format, $current);
            $current = strtotime($interval, $current);
        }

        return $dates;
    }
}

/**
 * Converts seconds to a human readable time
 *
 * @param  int seconds
 * @return string 
 */
if (!function_exists('secs_to_human')) {
    function secs_to_human($seconds)
    {
        $dtF = new DateTime("@0");
        $dtT = new DateTime("@$seconds");

        if ($seconds > 3600) {
            return $dtF->diff($dtT)->format('%h h %i m %s s');
        } else {
            return $dtF->diff($dtT)->format('%i m %s s');
        }
    }
}

/**   
 * Returns the offset from the origin timezone to the remote timezone, in seconds.
 * Found on: http://php.net/manual/en/function.timezone-offset-get.php
 *
 * @param $remote_tz; 
 * @param $origin_tz; If null the servers current timezone is used as the origin.
 * @return int;
*/
function get_timezone_offset($remote_tz, $origin_tz = null)
{
    if ($origin_tz === null) {
        if (!is_string($origin_tz = date_default_timezone_get())) {
            return false; // A UTC timestamp was returned -- bail out!
        }
    }
    $origin_dtz = new DateTimeZone($origin_tz);
    $remote_dtz = new DateTimeZone($remote_tz);
    $origin_dt = new DateTime("now", $origin_dtz);
    $remote_dt = new DateTime("now", $remote_dtz);
    $offset = $origin_dtz->getOffset($origin_dt) - $remote_dtz->getOffset($remote_dt);
    return $offset;
}

/**
 * Takes in how many seconds to take off a date
 *
 * @param  int seconds
 * @param  string date
 * @return string date
 */
function mathSecondsToDate($seconds, $date)
{
    return date("Y-m-d H:i:s", strtotime($date) - $seconds);
}

/**
 * Get config/status of a specific config variable. We'll
 * take whatever is in the db first, then config then null
 * value last
 *
 * @param  string Variable to check
 * @param  null value
 * @param  string which config file
 * @return value
 */
function getConfig($var, $null = null, $config = "analytics")
{

    //-- First check is DB
    $check = \DB::table('settings')->where('name', $var)->first();
    if ($check != null) {
        return $check->value;
    }

    //-- Secondly, check config files, and on empty then return null
    return config($config.".".$var, $null);
}

/**
 * Get either a Gravatar URL or complete image tag for a specified email address.
 *
 * @param string $email The email address
 * @param string $s Size in pixels, defaults to 80px [ 1 - 2048 ]
 * @param string $d Default imageset to use [ 404 | mm | identicon | monsterid | wavatar ]
 * @param string $r Maximum rating (inclusive) [ g | pg | r | x ]
 * @param boole $img True to return a complete IMG tag False for just the URL
 * @param array $atts Optional, additional key/value attributes to include in the IMG tag
 * @return String containing either just a URL or a complete image tag
 * @source http://gravatar.com/site/implement/images/php/
 */
function get_gravatar($email, $s = 80, $d = 'mm', $r = 'g', $img = false, $atts = array())
{
    $url = 'http://www.gravatar.com/avatar/';
    $url .= md5(strtolower(trim($email)));
    $url .= "?s=$s&d=$d&r=$r";
    if ($img) {
        $url = '<img src="' . $url . '"';
        foreach ($atts as $key => $val) {
            $url .= ' ' . $key . '="' . $val . '"';
        }
        $url .= ' />';
    }
    return $url;
}
