<?php

namespace app\Http\Controllers\User;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class UserController extends Controller
{

    /**
     * Show the dashboard
     *
     * @return Response
    */
    public function showDashboard(Request $request)
    {
        return view('themes.' . getConfig('theme') .'.user.dashboard');
    }

    /**
     * Changes the users date control, which is used
     * on the site pages. We'll then store that data
     * in a session on a user.
     *
     * @param  string  Date Type
     * @return void
    */
    public function changeDateControl(Request $request)
    {
        //-- What is the date control type we have got?
        $data = array();
        $data['start'] = $request->input('start', strtotime("Y-m-d", strtotime("-10 days")));
        $data['end'] = $request->input('end', strtotime("Y-m-d"));

        //-- Lets work out what we should give users etc
        $request->session()->put('date-control', $data);

        //-- Return user back to previous
        return redirect()->back();
    }

    /**
     * Show the user upgrade page
     *
     * @return Response
    */
    public function showUserUpgrade(Request $request)
    {
        $packages = \App\Package::where('active', 1)->get();

        return view('themes.' . getConfig('theme') .'.user.upgrade.upgrade', ['packages' => $packages]);
    }

    /**
     * Show the user upgrade payment page, allowing the user to pay the system
     *
     * @return Response
    */
    public function showUserUpgradePayment(Request $request)
    {
        $package = \App\Package::where('name_slug', $request->slug)->first();
        if ($package == null) {
            return redirect()->to(action('User\UserController@showUserUpgrade'));
        }

        //-- Setup Braintree
        \Braintree_Configuration::environment(getConfig('enviroment'));
        \Braintree_Configuration::merchantId(getConfig('merchantId'));
        \Braintree_Configuration::publicKey(getConfig('public_key'));
        \Braintree_Configuration::privateKey(getConfig('private_key'));

        //-- Lets create a payment token
        $token = \Braintree_ClientToken::generate();

        //-- Lets check for a POST
        if ($request->isMethod('post')) {

            //-- If there is no cost, we'll proceed instantly
            if ($package->cost == "0") {
                
                //-- Check to make sure user isn't already part of a subscription. 
                if (\Auth::user()->current_package_subscription != '') {
                    //-- We are, lets sever this link
                    $cancel = \Braintree_Subscription::cancel(\Auth::user()->current_package_subscription);
                    \Auth::user()->current_package_subscription = '';
                    \DB::table('log')->insert(['type' => 'Cancel Subscription', 'log' => "Cancel Sub [User: " . \Auth::user()->name . "]", 'created_at' => date("Y-m-d H:i:s")]);
                }

                //-- Add to User 
                \Auth::user()->package_id = $package->id;
                \Auth::user()->save();

                //-- Add to system log
                \DB::table('log')->insert(['type' => 'New Transaction', 'log' => "User Package Upgrade [Package: " . $package->name . "]", 'created_at' => date("Y-m-d H:i:s")]);

                //-- Finally, we'll need a success!
                $request->session()->flash('success', '<p><strong>Great!</strong></p><p>Payment has been taken, and you are now on your new package - <strong>' . $package->name . '</strong></p>');
                return redirect()->to(action('User\UserController@showUserUpgrade'));
            }

            if ($package->time_period == "One-off Payment") {

                //-- One off payment doesn't use the subscription system at all!
                $result = \Braintree\Transaction::sale([
                    'amount' => $package->cost,
                    'paymentMethodNonce' => $request->input('payment_method_nonce', ''),
                    'options' => [
                        'submitForSettlement' => true
                    ]
                ]);
            } elseif ($package->time_period == "Monthly") {

                //-- Monthly payments subscription - first lets find the customer. They'll
                //-- all the have the same type of ID
                $collection = \Braintree_Customer::search([
                    \Braintree_CustomerSearch::id()->is('wranalytics_' . \Auth::user()->id)
                    ]);

                //-- We'll get an iterator object returned
                if (iterator_count($collection) == 0) {
                    //-- Lets create them
                    $newCustomer = \Braintree_Customer::create([
                        'id'        => 'wranalytics_' . \Auth::user()->id,
                        'firstName' => \Auth::user()->name,
                        'email'    => \Auth::user()->email,
                        'paymentMethodNonce' => $request->input('payment_method_nonce', '')
                        ]);
                    $customer = $newCustomer->customer;
                } else {
                    //-- Lets use the first it finds
                    foreach ($collection as $customerz) {
                        $customer = $customerz;
                    }
                }

                //-- Now, create the subscription
                $result = \Braintree_Subscription::create([
                    'paymentMethodToken' => $customer->paymentMethods[0]->token,
                    'planId' => $package->braintree_id
                    ]);
            }

            //-- Check to see what our result is
            if ($result->success) {

                //-- Awesome, we know that this transaction was good. Lets do some
                //-- things to set this user at this package.

                if ($package->time_period == "One-off Payment") {
                    $trans_id = $result->transaction->id;
                    //-- Add to payment log
                    $p = new \App\Payment();
                    $p->user_id = \Auth::user()->id;
                    $p->package_id = $package->id;
                    $p->payment_id = $trans_id;
                    $p->amount = number_format($package->cost, 2);
                    $p->save();
                }

                //-- Check to make sure user isn't already part of a subscription. 
                if (\Auth::user()->current_package_subscription != '') {
                    //-- We are, lets sever this link
                    $cancel = \Braintree_Subscription::cancel(\Auth::user()->current_package_subscription);
                    \Auth::user()->current_package_subscription = '';
                    \DB::table('log')->insert(['type' => 'Cancel Subscription', 'log' => "Cancel Sub [User: " . \Auth::user()->name . "]", 'created_at' => date("Y-m-d H:i:s")]);
                }

                //-- Add to User 
                \Auth::user()->package_id = $package->id;
                if ($package->time_period == "One-off Payment") {
                    //-- We don't need any more data here
                } else {
                    //-- We'll store our link
                    \Auth::user()->current_package_subscription = $result->subscription->id;
                }

                //-- Save User model
                \Auth::user()->save();

                //-- Add to system log
                \DB::table('log')->insert(['type' => 'New Transaction', 'log' => "User Package Upgrade [User: " . \Auth::user()->name . "] [Package: " . $package->name . "]", 'created_at' => date("Y-m-d H:i:s")]);

                //-- Finally, we'll need a success!
                $request->session()->flash('success', '<p><strong>Great!</strong></p><p>Payment has been taken, and you are now on your new package - <strong>' . $package->name . '</strong></p>');
                return redirect()->to(action('User\UserController@showUserUpgrade'));
            } elseif ($result->transaction) {
                $request->session()->flash('error', '<p><strong>There has been a problem.</strong></p><p>' . $result->message . '</p>');
                return redirect()->to(action('User\UserController@showUserUpgrade'));
            } else {
                $request->session()->flash('error', '<p><strong>There has been a problem.</strong></p><p>' . $result->errors->deepAll()[0]->message . '</p>');
                return redirect()->to(action('User\UserController@showUserUpgrade'));
            }
        }

        return view('themes.' . getConfig('theme') .'.user.upgrade.upgrade_payment', ['package' => $package, 'token' => $token]);
    }
}
