<?php

namespace app\Http\Controllers\User;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class SiteController extends Controller
{

    /**
     * Shows the add a site page, which allows the user
     * to create a site
     *
     * @return Response
    */
    public function createSite(Request $request)
    {

        //-- Can we add a new site?
        $allowed = \Auth::user()->package->sites;
        if ($allowed != 0) {
            //-- How many sites do we have
            $sites = \App\Site::where('author_id', \Auth::user()->id)->count();
            if ($sites >= $allowed) {
                //-- We already have maxed out how many sites we are allowed
                $request->session()->flash('warning', '<p><strong>There was a problem adding a new site.</strong></p><p>Your package only allows ' . $allowed . ' sites to be added. Please upgrade if you would like to add more sites.</p>');
                return redirect()->to(action('User\UserController@showDashboard'));
            }
        }

        //-- Check if we have POST data
        if ($request->isMethod('post')) {

            //-- Run Validations
            $validator = \Validator::make($request->all(), [
                'name'        => 'required|unique:sites,name,null,id,author_id,'.\Auth::user()->id.'|min:1|max:32',
                'url'        => 'required|url|unique:sites,url,null,id,author_id,'.\Auth::user()->id,
                'timezone'    => 'required'
                ]);

            if ($validator->fails()) {
                //-- GO back and show errors
                return redirect()
                ->back()
                ->withErrors($validator)
                ->withInput();
            }

            //-- Passed main validation

            //-- Create site
            $s = new \App\Site;
            $s->uuid = createUniqueStrings('sites', 'uuid', 10);
            $s->name = $request->input('name');
            $s->url = $request->input('url');
            $s->timezone = $request->input('timezone');
            $s->author_id = \Auth::user()->id;
            $s->save();

            //-- Forward to sites new settings page so the user can
            //-- grab there embed code

            return redirect()->to(action('User\SiteController@siteSettings', $s->uuid));
        }

        return view('themes.' . getConfig('theme') .'.user.site.create');
    }

    /**
     * Does a check based on the UUID of a site, and returns 
     * \App\Site or False if the auth'd user can view it.
     *
     * @param  String  UUID
     * @return boolean
     */
    private function userViewSite($uuid)
    {
        $site = \App\Site::where('uuid', $uuid)->where('uuid', '!=', '')->where('author_id', \Auth::user()->id)->first();
        if ($site == false) {
            //-- Force the system to redrect
            return redirect()->to(action('User\UserController@showDashboard'))->send();
        } else {
            return $site;
        }
    }

    /**
     * Shows the sites dashboard
     *
     * @param  String  UUID
     * @return Response
     */
    public function siteDashboard(Request $request)
    {
        //-- Get info
        $site = $this->userViewSite($request->uuid);
        $report = date_control_function($request);

        //-- Render
        return view('themes.' . getConfig('theme') .'.user.site.dashboard', ['site' => $site, 'report' => $report]);
    }

    /**
     * Shows site content data
     *
     * @param  string UUID
     * @return Response
     */
    public function siteContent(Request $request)
    {
        //-- Get info
        $site = $this->userViewSite($request->uuid);
        $report = date_control_function($request);
        //-- Render
        return view('themes.' . getConfig('theme') .'.user.site.content', ['site' => $site, 'report' => $report]);
    }

    /**
     * Shows site location data
     *
     * @param  string UUID
     * @return Response
     */
    public function siteLocation(Request $request)
    {
        //-- Get info
        $site = $this->userViewSite($request->uuid);
        $report = date_control_function($request);
        //-- Render
        return view('themes.' . getConfig('theme') .'.user.site.location', ['site' => $site, 'report' => $report]);
    }

    /**
     * Shows site technology data
     *
     * @param  string UUID
     * @return Response
     */
    public function siteTech(Request $request)
    {
        //-- Get info
        $site = $this->userViewSite($request->uuid);
        $report = date_control_function($request);
        //-- Render
        return view('themes.' . getConfig('theme') .'.user.site.tech', ['site' => $site, 'report' => $report]);
    }

    /**
     * Shows site refs data
     *
     * @param  string UUID
     * @return Response
     */
    public function siteRefs(Request $request)
    {
        //-- Get info
        $site = $this->userViewSite($request->uuid);
        $report = date_control_function($request);
        //-- Render
        return view('themes.' . getConfig('theme') .'.user.site.ref', ['site' => $site, 'report' => $report]);
    }

    /**
     * Shows site settings
     *
     * @param  string UUID
     * @return Response
     */
    public function siteSettings(Request $request)
    {
        //-- Get info
        $site = $this->userViewSite($request->uuid);
        $report = date_control_function($request);

        //-- Check if we have POST data
        if ($request->isMethod('post')) {

            //-- Run Validations
            $validator = \Validator::make($request->all(), [
                'name'        => 'required|unique:sites,name,'.$site->id.',id,author_id,'.\Auth::user()->id.'|min:1|max:32',
                'url'        => 'required|url|unique:sites,url,'.$site->id.',id,author_id,'.\Auth::user()->id,
                'timezone'    => 'required'
                ]);

            if ($validator->fails()) {
                //-- GO back and show errors
                return redirect()
                ->back()
                ->withErrors($validator)
                ->withInput();
            }

            //-- Lets save!
            $site->name = $request->input('name');
            $site->url = $request->input('url');
            $site->timezone = $request->input('timezone');
            $site->save();

            //-- Return a view with our success var set
            return view('themes.' . getConfig('theme') .'.user.site.settings', ['site' => $site, 'report' => $report, 'success' => '<p><strong>Awesome, site settings saved!</strong></p>']);
        }

        //-- Render
        return view('themes.' . getConfig('theme') .'.user.site.settings', ['site' => $site, 'report' => $report]);
    }

    /**
     * Deletes a site
     *
     * @param  string UUID
     * @return Response
     */
    public function siteDelete(Request $request)
    {
        //-- Get info
        $site = $this->userViewSite($request->uuid);

        //-- Deletes all RAW data for this site
        \DB::table('raw')->where('site_id', $site->id)->delete();
        \DB::table('log')->insert(['type' => 'Site Delete', 'log' => "Site removed by user [site: " . $site->name . "] [user: ". \Auth::user()->name ."]", 'created_at' => date("Y-m-d H:i:s")]);

        //-- Delete the site
        $site->delete();

        //-- Forward to users dashbaord
        return redirect()->to(action('User\UserController@showDashboard'));
    }

    /**
     * Shows site live view
     *
     * @param  string UUID
     * @return Response
     */
    public function siteLive(Request $request)
    {
        //-- Get info
        $site = $this->userViewSite($request->uuid);
        $report = date_control_function($request);
        //-- Render
        return view('themes.' . getConfig('theme') .'.user.site.live', ['site' => $site, 'report' => $report]);
    }

    /**
     * Shows site live view
     *
     * @param  string UUID
     * @return Response
     */
    public function getLiveStats(Request $request)
    {
        //-- Get info
        $site = $this->userViewSite($request->uuid);
        $report = date_control_function($request);
        //-- Get Stats
        return response()->json($site->getLiveStats());
    }

    /**
     * Shows test
     *
     * @param  string UUID
     * @return Response
     */
    public function test(Request $request)
    {
        //-- Get info
        $site = $this->userViewSite($request->uuid);
        $report = date_control_function($request);

        $info = $site->deleteHistorical();
        dd($info);
    }
}
