<?php

namespace app\Http\Controllers\Auth;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class UserAuthController extends Controller
{

    /**
     * Run the login system
     *
     * @param Request $request
     * @return Response
    */
    public function loginSystem(Request $request)
    {

        //-- If auth'd already, send to dashboard
        if (\Auth::check()) {
            return redirect()->to(action('User\UserController@showDashboard'));
        }

        //-- Check if we have POST data
        if ($request->isMethod('post')) {

            //-- Run Validations
            $validator = \Validator::make($request->all(), [
                'username' => 'required',
                'password' => 'required',
            ]);

            if ($validator->fails()) {
                //-- GO back and show errors
                return redirect()
                ->back()
                ->withErrors($validator)
                ->withInput();
            }

            //-- Passed validation, can we try and login?
            if (\Auth::attempt(['name' => $request->input('username'), 'password' => $request->input('password'), 'active' => 1])) {

                //-- Lets record a login for this user
                //-- Redirect user to dashboard
                return redirect()->to(action('User\UserController@showDashboard'));
            } else {
                //-- We have failed out login - lets show a generic message to the user
                //-- TODO: Find better way to add message to validation error bag
                $validator->after(function ($validator) {
                    $validator->errors()->add('login', 'Username and/or Password does not exist.');
                });
                if ($validator->fails()) {
                    return redirect()
                    ->back()
                    ->withErrors($validator)
                    ->withInput();
                }
            }
        }

        //-- No POST data, lets display page
        return view('themes.' . getConfig('theme') .'.auth.login');
    }

    /**
     * Log the user out
     *
     * @return User logout and redirect to index
    */
    public function logoutSystem()
    {
        \Auth::logout();
        return redirect('/');
    }

    /**
     * Registers a new user to the system
     *
     * @param Request $request
     * @return Response
    */
    public function registerSystem(Request $request)
    {

        //-- If auth'd already, send to dashboard
        if (\Auth::check()) {
            return redirect()->to(action('User\UserController@showDashboard'));
        }

        //-- Check if we have POST data
        if ($request->isMethod('post')) {

            //-- Run Validations
            $validator = \Validator::make($request->all(), [
                'username'    => 'required|unique:users,name|min:5|max:100',
                'email'    => 'required|email|unique:users,email',
                'password'    => 'required|min:5',
                'password2' => 'required|same:password',
            ]);

            if ($validator->fails()) {
                //-- GO back and show errors
                return redirect()
                ->back()
                ->withErrors($validator)
                ->withInput();
            }

            //-- Passed validation, lets setup the user on the system
            $u = new \App\User;
            $u->name        = $request->input('username');
            $u->email        = $request->input('email');
            $u->password    = bcrypt($request->input('password'));
            $u->package_id    = 1;
            $u->user_validation = md5(time() . $request->input('username') . rand(0, 10000));
            $u->save();

            //-- Lets now email this person there validation token. They will not be able 
            //-- to login without it.
            \Mail::send('emails.register', ['user' => $u], function ($m) use ($u) {
                $m->from('hello@app.com', 'Your Application Name');
                $m->to($u->email, $u->name)->subject('Thanks for registering!');
            });

            //-- Finally, lets put the user back on the reg page with a nice
            //-- success message!
            return view('themes.' . getConfig('theme') .'.auth.register', ['success' => '<p><strong>Thanks for Registering</strong></p><p>Please check your email to validate your account before logging in.</p>']);
        }

        //-- No POST data, lets display page
        return view('themes.' . getConfig('theme') .'.auth.register');
    }

    /**
     * Validates a users new registration
     *
     * @param  Request  $request
     * @param  string  $val
     * @return Response
    */
    public function validateUserRegistration(Request $request)
    {

        //-- If auth'd already, send to dashboard
        if (\Auth::check()) {
            return redirect()->to(action('User\UserController@showDashboard'));
        }

        //-- Check to see if this validation token exists
        $exists = \App\User::where('user_validation', $request->val)->first();
        if ($exists != false) {
            //-- Lets validate this user
            $exists->user_validation = '';
            $exists->active = 1;
            $exists->save();
            return view('themes.' . getConfig('theme') .'.auth.register', ['success' => '<p><strong>Thanks for Validating</strong></p><p>You can now login!</p>']);
        }

        return redirect()->to(action('Auth\UserAuthController@registerSystem'));
    }

    /**
     * If a user forgets there password, they'll
     * come here to reset it!
     *
     * @param  Request  $request
     * @return Response
    */
    public function forgotPasswordSystem(Request $request)
    {

        //-- If auth'd already, send to dashboard
        if (\Auth::check()) {
            return redirect()->to(action('User\UserController@showDashboard'));
        }

        //-- Check if we have POST data
        if ($request->isMethod('post')) {

            //-- Run Validations
            $validator = \Validator::make($request->all(), [
                'email'    => 'required|email',
            ]);

            if ($validator->fails()) {
                //-- GO back and show errors
                return redirect()
                ->back()
                ->withErrors($validator)
                ->withInput();
            }

            //-- Check if users email exists
            $u = \App\User::where('email', $request->input('email'))->first();
            if ($u != false) {
                //-- If so, create a link for them
                $u->password_reset = md5(time() . $request->input('email') . rand(0, 10000));
                $u->save();
                //-- Email this link out
                \Mail::send('emails.forgot', ['user' => $u], function ($m) use ($u) {
                    $m->from('hello@app.com', 'Your Application Name');
                    $m->to($u->email, $u->name)->subject('Reset your password');
                });
            }

            //-- We'll tell the end user the same whateverrr
            return view('themes.' . getConfig('theme') .'.auth.forgot', ['success' => '<p><strong>Thanks</strong></p><p>We have sent a link to that email address if it exists in our system.</p>']);
        }

        return view('themes.' . getConfig('theme') .'.auth.forgot');
    }

    /**
     * A user will hit this page if they are trying to
     * reset there password
     *
     * @param  Request  $request
     * @return Response
    */
    public function resetPasswordSystem(Request $request)
    {

        //-- If auth'd already, send to dashboard
        if (\Auth::check()) {
            return redirect()->to(action('User\UserController@showDashboard'));
        }
        
        //-- Check to see if this validation token exists
        $exists = \App\User::where('password_reset', $request->val)->where('password_reset', '!=', '')->first();
        if ($exists == false) {
            return redirect()->to(action('Auth\UserAuthController@loginSystem'));
        }

        //-- Lets check for POST data
        //-- Check if we have POST data
        if ($request->isMethod('post')) {

            //-- Run Validations
            $validator = \Validator::make($request->all(), [
                'password'    => 'required|min:5',
                'password2' => 'required|same:password',
            ]);

            if ($validator->fails()) {
                //-- GO back and show errors
                return redirect()
                ->back()
                ->withErrors($validator)
                ->withInput();
            }

            //-- Lets change password
            $exists->password = bcrypt($request->input('password'));
            $exists->password_reset = '';
            $exists->save();

            //-- Lets forward to login page with info
            return view('themes.' . getConfig('theme') .'.auth.login', ['success' => '<p><strong>Password successfully changed!</strong></p><p>Please login below.</p>']);
        }

        return view('themes.' . getConfig('theme') .'.auth.reset');
    }
}
