<?php
namespace app\Http\Controllers\Admin;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class AdminController extends Controller
{
    
    /**
     * Shows the admin dashboard
     *
     * @return Response
    */
    public function showDashboard(Request $request)
    {
        $live = $this->getAdminStats();

        return view('admin.dashboard', ['live' => $live]);
    }

    /**
     * Gets JSON of the live datd
     *
     * @return JSON
     */
    public function getLiveJSON()
    {
        $live = $this->getAdminStats();
        return response()->json($live);
    }

    /**
     * Shows all user accounts
     *
     * @return Response
     */
    public function showUserAccounts(Request $request)
    {
        $users = \App\User::where('active', 1)->get();

        return view('admin.users.list', ['users' => $users]);
    }

    /**
     * Add user account
     *
     * @return Response
     */
    public function showAddUserAccount(Request $request)
    {
        $packages = \App\Package::get();

        //-- Check if we have POST data
        if ($request->isMethod('post')) {

            //-- Run Validations
            $validator = \Validator::make($request->all(), [
                'username'    => 'required|unique:users,name|min:5|max:100',
                'email'    => 'required|email|unique:users,email',
                'password'    => 'required|min:5',
                'package'    => 'required'
               ]);

            if ($validator->fails()) {
                //-- GO back and show errors
                return redirect()
                ->back()
                ->withErrors($validator)
                ->withInput();
            }

            //-- Passed validation, lets setup the user on the system
            $u = new \App\User;
            $u->name        = $request->input('username');
            $u->email        = $request->input('email');
            $u->password    = bcrypt($request->input('password'));
            $u->active        = 1;
            $u->package_id    = $request->input('package');
            $u->save();

            //-- Lets forward to edit user page
            return redirect()->to(action('Admin\AdminController@showEditUserAccount', $u->id));
        }

        return view('admin.users.add', ['packages' => $packages]);
    }

    /**
     * dit user account
     *
     * @return Response
     */
    public function showEditUserAccount(Request $request)
    {
        $user = \App\User::where('id', $request->id)->first();
        if ($user == null) {
            return redirect()->to(action('Admin\AdminController@showDashboard'));
        }
        $packages = \App\Package::get();

        //-- Check if we have POST data
        if ($request->isMethod('post')) {

            //-- Run Validations
            $validator = \Validator::make($request->all(), [
                'username'    => 'required|unique:users,name,' . $user->id . '|min:5|max:100',
                'email'    => 'required|email|unique:users,email,' . $user->id . '',
                'password'    => 'min:5',
                'package'    => 'required'
               ]);

            if ($validator->fails()) {
                //-- GO back and show errors
                return redirect()
                ->back()
                ->withErrors($validator)
                ->withInput();
            }

            $user->name = $request->input('username');
            $user->email = $request->input('email');
            $user->package_id    = $request->input('package');
            $user->active = 1;
            if ($request->input('password', '') != '') {
                $user->password    = bcrypt($request->input('password'));
            }
            $user->save();

            return view('admin.users.edit', ['packages' => $packages, 'user' => $user, 'success' => '<p><strong>Your edits were successful!</strong></p>']);
        }
        
        return view('admin.users.edit', ['packages' => $packages, 'user' => $user]);
    }

    /**
     * Login as a user
     *
     * @return response
     */
    public function loginAsUser(Request $request)
    {
        $user = \App\User::where('id', $request->id)->first();
        if ($user == null) {
            return redirect()->to(action('Admin\AdminController@showDashboard'));
        }

        //-- Logout first
        \Auth::logout();
        //-- Re-login as $user model
        \Auth::loginUsingId($user->id);
        //-- Forward to there dashboard
        return redirect()->to(action('User\UserController@showDashboard'));
    }

    /**
     * Delete a user and all info for them
     *
     * @return redirect
     */
    public function deleteUser(Request $request)
    {
        $user = \App\User::where('id', $request->id)->first();
        if ($user == null) {
            return redirect()->to(action('Admin\AdminController@showDashboard'));
        }

        //-- Delete all RAW data
        foreach ($user->sites as $site) {
            \DB::table('raw')->where('site_id', $site->id)->delete();
            \DB::table('log')->insert(['type' => 'Site Delete', 'log' => "Site removed [" . $site->name . "]", 'created_at' => date("Y-m-d H:i:s")]);
        }

        //-- Delete all sites
        \App\Site::where('author_id', $user->id)->delete();

        //-- Delete user
        \DB::table('log')->insert(['type' => 'User Delete', 'log' => "User removed [" . $user->name . " / " . $user->email . "]", 'created_at' => date("Y-m-d H:i:s")]);
        $user->delete();

        //-- Redirect to dashboard
        return redirect()->to(action('Admin\AdminController@showDashboard'));
    }

    /**
     * Shows all packages
     *
     * @return Response
     */
    public function showPackages(Request $request)
    {
        $packages = \App\Package::get();

        return view('admin.packages.list', ['packages' => $packages]);
    }

    /**
     * Add package
     *
     * @return Response
     */
    public function showAddPackage(Request $request)
    {

        //-- Check if we have POST data
        if ($request->isMethod('post')) {

            //-- Run Validations
            $validator = \Validator::make($request->all(), [
                'name'    => 'required|unique:packages,name',
                'ads'    => '',
                'hits'  => 'required|numeric',
                'history'  => 'required|numeric',
                'sites'  => 'required|numeric',
                'cost'  => 'required|numeric',
                'time_period' => 'required',
                'braintree_id' => 'required_if:time_period,Monthly|required_if:time_period,Yearly'
               ]);

            if ($validator->fails()) {
                //-- GO back and show errors
                return redirect()
                ->back()
                ->withErrors($validator)
                ->withInput();
            }

            //-- Awesome, lets add it
            $p = new \App\Package;
            $p->name = $request->input('name');
            $p->name_slug = str_slug($request->input('name'));
            $p->adverts = $request->input('ads', 0);
            $p->daily_hits = $request->input('hits', 0);
            $p->history = $request->input('history', 0);
            $p->sites = $request->input('sites', 0);
            $p->cost = $request->input('cost', 0);
            $p->time_period = $request->input('time_period', 0);
            $p->braintree_id = $request->input('braintree_id', 0);
            $p->active = 0;
            $p->save();

            //-- Lets forward to the edit page
            return redirect()->to(action('Admin\AdminController@showEditPackage', $p->id));
        }

        return view('admin.packages.add', []);
    }

    /**
     * Edit package
     *
     * @return Response
     */
    public function showEditPackage(Request $request)
    {
        $package = \App\Package::find($request->id);
        if ($package == null) {
            return redirect()->to(action('Admin\AdminController@showDashboard'));
        }

        //-- Check if we have POST data
        if ($request->isMethod('post')) {

            //-- Run Validations
            $validator = \Validator::make($request->all(), [
                'name'    => 'required|unique:packages,name,' . $package->id,
                'ads'    => '',
                'hits'  => 'required|numeric',
                'history'  => 'required|numeric',
                'sites'  => 'required|numeric',
                'cost'  => 'required|numeric',
                'time_period' => 'required',
                'braintree_id' => 'required_if:time_period,Monthly|required_if:time_period,Yearly'
               ]);

            if ($validator->fails()) {
                //-- GO back and show errors
                return redirect()
                ->back()
                ->withErrors($validator)
                ->withInput();
            }

            //-- Save
            $package->name = $request->input('name');
            ;
            $package->name_slug = str_slug($request->input('name'));
            $package->adverts = $request->input('ads', 0);
            $package->daily_hits = $request->input('hits', 0);
            $package->history = $request->input('history', 0);
            $package->sites = $request->input('sites', 0);
            $package->active = $request->input('active', 0);
            $package->cost = $request->input('cost', 0);
            $package->time_period = $request->input('time_period', 0);
            $package->braintree_id = $request->input('braintree_id', 0);
            $package->save();

            return view('admin.packages.edit', ['package' => $package, 'success' => '<p><strong>Your edits were successful!</strong></p>']);
        }

        return view('admin.packages.edit', ['package' => $package]);
    }

    /**
     * Gets full admin statistics
     *
     * @param  string start
     * @param  string end
     * @return array
     */
    private function getAdminStats($start = false, $end = false)
    {
        //-- Vars to use
        if (strtotime($start) == 0) {
            $start = date("Y-m-d") . " 00:00:00";
        }
        if (strtotime($end) == 0) {
            $end   = date("Y-m-d") . " 23:59:59";
        }
        $time = getConfig('live_test', 10);
        $output = array();
        $working = array();

        //-- Firstly get Live Information. WILL NOT CHANGE WITH START, END
     $live_start = date("Y-m-d H:i:s", strtotime("-" . $time . " mins"));
        $live_end    = date("Y-m-d") . " 23:59:59";
        $devices = $this->getSystemMetric($live_start, $live_end, 0, 'device', true);

        //-- Lets grab all rows to work out actual pages and visitors
     $raw = \DB::table('raw')->where('created_at', '>=', $live_start)->where('created_at', '<=', $live_end)->get();
        foreach ($raw as $r) {
            //-- Unique Visitors and actions
      $working['visitors'][$r->visitor_id] = 1;
            @$working['actions']++;
        }

        //-- Create Output
  $page_render = \DB::table('raw')->where('created_at', '>=', $start)->where('created_at', '<=', $end)->avg('system_time');
        $output['visitors'] = @count($working['visitors']);
        foreach ($devices as $device => $count) {
            $output['device'][$device] = @number_format(($count / $working['actions']) * 100, 0);
        }
        $output['hits'] = $this->getHitsDateRange($start, $end);
        $output['visits'] = $this->getVisitorDateRange($start, $end);
        $output['active'] = count(\DB::table('raw')->where('created_at', '>=', $start)->where('created_at', '<=', $end)->groupBy('site_id')->get());
        $output['render'] = number_format($page_render, 3);

        //-- Last 10 days
  $dates = array_dates("-10 days", date("Y-m-d", strtotime("+1 day")), "+1 day");
        $json = array();
        foreach ($dates as $date) {
            $amount1 = $this->getHitsDateRange($date . " 00:00:00", $date . " 23:59:59");
            $amount2 = $this->getVisitorDateRange($date . " 00:00:00", $date . " 23:59:59");
            $output['graph'][$date]['hits'] = $amount1;
            $output['graph'][$date]['visits'] = $amount2;
            $json[] = array( 'y' => $date, 'a' => $amount1, 'b' => $amount2 );
        }

        $output['json'] = $json;

        return $output;
    }

    /**
     * Gets the hits for a specific date-range from the RAW table
     *
     * @param  string  Start date
     * @param  string  End date
     * @return  int Number of hits
     */
    private function getHitsDateRange($start = false, $end = false)
    {
        //-- Check for defaults, and pre fill with today
        if (strtotime($start) == 0) {
            $start = date("Y-m-d") . " 00:00:00";
        }
        if (strtotime($end) == 0) {
            $end   = date("Y-m-d") . " 23:59:59";
        }
        //-- Generate simple SQL
        $raw = \DB::table('raw')->where('created_at', '>=', $start)->where('created_at', '<=', $end)->get();
        //-- Return a count
        return count($raw);
    }

    /**
     * Gets the visitors for a specific date-range from the RAW table,
     * which is like the hits, but we look for unique visitor_ids
     *
     * @param  string  Start date
     * @param  string  End date
     * @return  int Number of hits
     */
    private function getVisitorDateRange($start = false, $end = false)
    {
        //-- Check for defaults, and pre fill with today
        if (strtotime($start) == 0) {
            $start = date("Y-m-d") . " 00:00:00";
        }
        if (strtotime($end) == 0) {
            $end   = date("Y-m-d") . " 23:59:59";
        }
        //-- Generate simple SQL
        $raw = \DB::table('raw')->where('created_at', '>=', $start)->where('created_at', '<=', $end)->groupBy('visitor_id')->get();
        //-- Return a count
        return count($raw);
    }

    /**
     * Gets a basic statistic of total amount of hits per metric 
     * that is given. Does a search of RAW.
     *
     * @param  string  Start date
     * @param  string  End date
     * @param  int  return how many results (0 = all)
     * @param  string  metric
     * @param  boolean  format as array with just metric and total hits
     * @return  \DB::row
     */
    private function getSystemMetric($start = false, $end = false, $limit = 0, $metric, $format = false)
    {
        //-- Check for defaults, and pre fill with today
        if (strtotime($start) == 0) {
            $start = date("Y-m-d") . " 00:00:00";
        }
        if (strtotime($end) == 0) {
            $end   = date("Y-m-d") . " 23:59:59";
        }

         //-- SQL
        if ($limit == 0) {
            $raw = \DB::table('raw')->select('*', \DB::raw('COUNT(id) as total_hits'))->where('created_at', '>=', $start)->where('created_at', '<=', $end)->groupBy($metric)->orderBy('total_hits', 'DESC')->get();
        } else {
            $raw = \DB::table('raw')->select('*', \DB::raw('COUNT(id) as total_hits'))->where('created_at', '>=', $start)->where('created_at', '<=', $end)->groupBy($metric)->orderBy('total_hits', 'DESC')->take($limit)->get();
        }

        if ($format) {
            $output = array();
            foreach ($raw as $r) {
                $output[$r->$metric] = $r->total_hits;
            }
            return $output;
        } else {
            return $raw;
        }
    }

    /**
     * Show settings page
     *
     * @return response
     */
    public function showSettings(Request $request)
    {

        //-- Check if we have POST data
      if ($request->isMethod('post')) {

            //-- Run Validations
       $validator = \Validator::make($request->all(), [
        'name'            => 'required',
        'theme'        => 'required',
        'live_users'    => 'required|numeric',
        ]);

          if ($validator->fails()) {
              //-- GO back and show errors
        return redirect()
        ->back()
        ->withErrors($validator)
        ->withInput();
          }

            //-- Save!
    \DB::table('settings')->where('name', 'name')->update(['value' => $request->input('name')]);
          \DB::table('settings')->where('name', 'theme')->update(['value' => $request->input('theme')]);
          \DB::table('settings')->where('name', 'live_test')->update(['value' => $request->input('live_users')]);
          \DB::table('settings')->where('name', 'upgrade')->update(['value' => $request->input('upgrade', 'false')]);
          \DB::table('settings')->where('name', 'registration')->update(['value' => $request->input('reg', 'false')]);

            //-- Success
    $request->session()->flash('success', '<p><strong>Settings has been saved</strong></p>');
          return redirect()->to(action('Admin\AdminController@showSettings'));
      }

        return view('admin.settings.main');
    }

    /**
     * Shows the system logs
     *
     * @return response
     */
    public function showSystemLogs(Request $request)
    {
        //-- Paginate our results
        $logs = \DB::table('log')->orderBy('id', 'desc')->Paginate(10);
        //-- Render
        return view('admin.settings.logs', ['logs' => $logs]);
    }

    /**
     * Show package payments
     *
     * @return response
     */
    public function showPackagePayments(Request $request)
    {
        //-- Grab all payments
        $payments    = \App\Payment::orderBy('id', 'desc')->get();
        $total        = \App\Payment::sum('amount');
        //-- Render
        return view('admin.packages.payments', ['payments' => $payments, 'total' => $total]);
    }

    /**
     * Show the stats
     *
     * @return response
     */
    public function showStats(Request $request)
    {
        //-- Get dates
        $start    = $request->input('start', date("Y-m-d", strtotime("-7 days")));
        $end    = $request->input('end', date("Y-m-d"));

        //-- Refine our data
        $start    = date("Y-m-d", strtotime($start));
        $end    = date("Y-m-d", strtotime($end));

        //-- Grab our stats
        $output = array();

        //-- Last X days
        $dates = array_dates($start, $end, "+1 day");
        $json = array();
        foreach ($dates as $date) {
            $amount1 = $this->getHitsDateRange($date . " 00:00:00", $date . " 23:59:59");
            $amount2 = $this->getVisitorDateRange($date . " 00:00:00", $date . " 23:59:59");
            $output['graph'][$date]['hits'] = $amount1;
            $output['graph'][$date]['visits'] = $amount2;
        }

        //-- Get other data
        $output['visitors'] = $this->getVisitorDateRange($start . " 00:00:00", $end . " 23:59:59");
        $output['hits'] = $this->getHitsDateRange($start . " 00:00:00", $end . " 23:59:59");
        $output['hitsper'] = @number_format($output['hits'] / $output['visitors'], 2);

        $output['browsers'] = $this->getSystemMetric($start, $end, 10, 'browser_name');
        $output['devices'] = $this->getSystemMetric($start, $end, 10, 'device');
        $output['locations'] = $this->getSystemMetric($start, $end, 10, 'country');

        //-- Top sites in that date range
      $output['sites'] = $this->getSystemMetric($start, $end, 10, 'site_id');

        //-- Render
      return view('admin.stats', ['start' => $start, 'end' => $end, 'stats' => $output]);
    }
}
