<?php

namespace app\Http\Controllers\API;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Jenssegers\Agent\Agent;
use GeoIp2\Database\Reader;

class APIController extends Controller
{

    /**
     * Returns our tracking JS file
     *
     * @param  Request
     * @return File
     */
    public function sendJS(Request $request)
    {
        //-- Send the file
        $tracking_file = "external/track.js";
        return response()->file($tracking_file);
    }

    /**
     * Generates a hit from our external beacon
     *
     * @param  Request
     * @return mime image/png
     */
    public function visitBeacon(Request $request)
    {
        /*
         * Debug allows us to see where the time is being taken inside
         * our request. Perfect for allowing us to see whats sinking
         * the time!
         *
         * @param boolean
         */
        $debug = false;
        $debug_start = microtime(true);

        //-- Checks to see if the key is an actual SITE_ID
        $check = \App\Site::where('uuid', $request->input('key', ''))->first();
        if ($check == false) {
            return response()->file('external/pixel.png');
        }

        //-- Check to make sure if there are any limits on this site
        $allowed = $check->user->package->daily_hits;
        if ($allowed != 0) {
            //-- We have a daily limit, lets check how many hits
            //-- we have had today
            $todayHits = $check->getHitsDateRange();
            if ($todayHits >= $allowed) {
                //-- We are not allowed any more hits
                return response()->file('external/pixel.png');
            }
        }

        if ($debug) {
            echo "Start<br>";
        }
        if ($debug) {
            echo "Site Check: " . (microtime(true) - $debug_start) . "<br>";
        }

        //-- Compile data
        $agent = new Agent();
        if ($debug) {
            echo "Agent Instantiated: " . (microtime(true) - $debug_start) . "<br>";
        }
        $location = \Location::get();
        if ($debug) {
            echo "Location Instantiated: " . (microtime(true) - $debug_start) . "<br>";
        }

        //-- Sort Data
        $data = array();
        $data['site_id']            = $check->id;
        $data['visitor_id']        = $request->input('uuid', str_random(50));
        $data['page_url']            = $request->input('uri', '');
        $data['referer_type']        = 3;
        $data['referer_url']        = $request->input('referrer', '');
        $data['os']                = $agent->platform();
        $data['browser_name']        = $agent->browser();
        $data['browser_version']    = $agent->version($data['browser_name']);
        $data['browser_lang']        = $agent->languages()[0];
        $data['device']                = $this->getDevice($agent);
        $data['resolution']            = $request->input('res', '');
        $data['ip']                    = $location->ip;
        $data['country']            = $location->countryName;
        $data['city']                = $location->cityName;
        $data['javascript']            = $request->input('js', '0');
        $data['pageload_time']        = $request->input('pageload', '');
        $data['created_at']            = date("Y-m-d H:i:s");

        if ($debug) {
            echo "Data Sorted: " . (microtime(true) - $debug_start) . "<br>";
        }

            //-- Check for previous visitor
        $data['time_since_last_visit'] = 0;
        $check = \DB::table('raw')->where('site_id', $check->id)->where('visitor_id', $data['visitor_id'])->orderBy('id', 'DESC')->first();
        if ($check != false) {
            //-- Check to see if we are within our session time of 10 mins
            $diff = abs(strtotime(date('Y-m-d H:i:s')) - strtotime($check->created_at));
            if ($diff < 600) {
                $data['time_since_last_visit'] = $diff;
            }
        }
        if ($debug) {
            echo "Previous Visitor: " . (microtime(true) - $debug_start) . "<br>";
        }

        //-- Plug in system time to data
        $data['system_time'] = number_format((microtime(true) - $debug_start), 6);

        //-- Lets save
        \DB::table('raw')->insert($data);

        //-- Serve PNG
        if ($debug) {
            echo "Data Saved: " . (microtime(true) - $debug_start) . "<br>";
        } else {
            return response()->file('external/pixel.png');
        }
    }

    /**
     * Checks for device
     *
     * @param  agent
     * @return string
     */
    private function getDevice($agent)
    {
        if ($agent->isDesktop()) {
            return "Desktop";
        } elseif ($agent->isMobile()) {
            return "Mobile";
        } elseif ($agent->isTablet()) {
            return "Tablet";
        } elseif ($agent->isRobot()) {
            return "Robot";
        } else {
            return "Other";
        }
    }

    /**
     * Calculates location
     *
     * @param  string IP
     * @return array
     */
    public function getLocation($ip)
    {
    }

    /**
     * Gets in a webhook from Braintree
     *
     * @param POST DATA
     * @return void
     */
    public function paymentBraintreeWebhook(Request $request)
    {
        //-- We need two payloads from the server
        $sig = $request->input('bt_signature', '');
        $payload = $request->input('bt_payload', '');

        $this->parseBraintreeWebhook($sig, $payload);
    }

    /**
     * Parses the webhook
     *
     * @param  string sig
     * @param  string payload
     * @return void
     */
    public function parseBraintreeWebhook($sig, $payload)
    {

        //-- Setup Braintree
        \Braintree_Configuration::environment(getConfig('enviroment'));
        \Braintree_Configuration::merchantId(getConfig('merchantId'));
        \Braintree_Configuration::publicKey(getConfig('public_key'));
        \Braintree_Configuration::privateKey(getConfig('private_key'));

        $webhookNotification = \Braintree_WebhookNotification::parse($sig, $payload);
        
        //-- Check for KIND
        switch ($webhookNotification->kind) {
            //-- Subscription has been canceled in the braintree admin panel
            case "subscription_canceled":
                //-- lets grab user
                $user = \App\User::where('current_package_subscription', $webhookNotification->subscription->id)->first();
                //-- Remove users package and set at default
                $user->package_id = 1;
                $user->current_package_subscription = '';
                $user->save();
                //-- Log
                \DB::table('log')->insert(['type' => 'Cancel Subscription', 'log' => "Cancel Sub [user: " . $user->name . "]", 'created_at' => date("Y-m-d H:i:s")]);
                break;

            //-- Subscription has been charged successfully
            case 'subscription_charged_successfully':
                //-- lets grab user
                $user = \App\User::where('current_package_subscription', $webhookNotification->subscription->id)->first();
                //-- Add to payment log
                $p = new \App\Payment();
                $p->user_id = $user->id;
                $p->package_id = $user->package->id;
                $p->payment_id = $webhookNotification->subscription->id;
                $p->amount = number_format($user->package->cost, 2);
                $p->save();
                //-- Log
                \DB::table('log')->insert(['type' => 'New Transaction', 'log' => "User Package Upgrade [Pay ID: " . $p->id . "] [Package: " . $user->package->name . "]", 'created_at' => date("Y-m-d H:i:s")]);
                break;

            //-- Subscription failed
            case 'subscription_charged_unsuccessfully':
                //-- lets grab user
                $user = \App\User::where('current_package_subscription', $webhookNotification->subscription->id)->first();
                //-- We'll take this user off the subscription
                $cancel = \Braintree_Subscription::cancel($user->current_package_subscription);
                $user->package_id = 1;
                $user->current_package_subscription = '';
                $user->save();
                //-- Log
                \DB::table('log')->insert(['type' => 'Cancel Subscription', 'log' => "Cancel Sub [User: " . $user->name . "]", 'created_at' => date("Y-m-d H:i:s")]);
                break;
        }
    }
}
