<?php

/**
 * main font class
 */
define("FONTS_BASE_STORAGE_FOLDER", DOC_ROOT . "/storage/");
define("FONTS_ARCHIVE_FOLDER", FONTS_BASE_STORAGE_FOLDER . "archives/");
define("FONTS_PREVIEW_CACHE_FOLDER", FONTS_BASE_STORAGE_FOLDER . "preview_cache/");
define("FONTS_UNPACKED_ARCHIVE", FONTS_BASE_STORAGE_FOLDER . "unpacked_archives/");
define("FONTS_CHARACTER_MAP_SYSTEM_FONT", FONTS_BASE_STORAGE_FOLDER . "systemFont.ttf");

/* zip library */
require_once(DOC_ROOT . '/includes/pclzip.lib.php');

class font
{

    public $fontId;
    public $fontData;
    public $previewOverride      = '';
    public $previewImageContents = null;
    public $characterMapContents = null;

    function __construct($fontId, $fontData = null)
    {
        if ($fontData == null)
        {
            $db       = Database::getDatabase();
            $fontData = $db->getRow("SELECT * FROM font WHERE id = " . (int) $fontId);
            if (!$fontData)
            {
                return false;
            }
        }
        $this->fontId = (int) $fontId;
        $this->fontData = $fontData;
    }

    function checkIpCanDownloadFont()
    {
        if ((int) SITE_CONFIG_MAX_DOWNLOADS_IP_PER_DAY > 0)
        {
            $db    = Database::getDatabase();
            $total = $db->getValue("SELECT COUNT(id) AS total FROM download WHERE ipAddress = " . $db->quote(getUsersIPAddress()) . " AND MID(dateCreated, 1, 10) = '" . date("Y-m-d") . "'");
            if ($total > (int) SITE_CONFIG_MAX_DOWNLOADS_IP_PER_DAY)
            {
                return false;
            }
        }
        return true;
    }

    function incrementFontDownloads()
    {
        $db = Database::getDatabase();
        return $db->query("UPDATE font SET totalDownloads = totalDownloads+1 WHERE id = " . (int) $this->fontId . " LIMIT 1");
    }

    function addFontDownloadReference()
    {
        // get session
        $Auth = Auth::getAuth();
        
        $dbInsert = new DBObject("download", array("fontId", "ipAddress", "dateCreated", "userId"));
        $dbInsert->fontId = (int) $this->fontId;
        $dbInsert->ipAddress = getUsersIPAddress();
        $dbInsert->dateCreated = sqlDateTime();
        $dbInsert->userId = $Auth->loggedIn()?$Auth->id:0;
        
        Stats::track($this->fontId);
        
        return $dbInsert->insert();
    }

    function getServerArchivePath()
    {
        return FONTS_ARCHIVE_FOLDER . (int) $this->fontId . ".zip";
    }
	
	function calculateTextBox($fontSize,$fontAngle,$fontFile,$text)
    { 
        /************ 
        simple function that calculates the *exact* bounding box (single pixel precision). 
        The function returns an associative array with these keys: 
        left, top:  coordinates you will pass to imagettftext 
        width, height: dimension of the image you have to create 
        *************/ 
        $rect = imagettfbbox($fontSize,$fontAngle,$fontFile,$text); 
        $minX = min(array($rect[0],$rect[2],$rect[4],$rect[6])); 
        $maxX = max(array($rect[0],$rect[2],$rect[4],$rect[6])); 
        $minY = min(array($rect[1],$rect[3],$rect[5],$rect[7])); 
        $maxY = max(array($rect[1],$rect[3],$rect[5],$rect[7])); 

        return array( 
         "left"   => abs($minX) - 1, 
         "top"    => abs($minY) - 1, 
         "width"  => $maxX - $minX, 
         "height" => $maxY - $minY, 
         "box"    => $rect 
        ); 
    }

    function createFontPreview($previewText = null, $previewWidth = 1500, $previewHeight = 90, $textColour = "000000")
    {
		$padding = 3;
        if ($previewText == null)
        {
            /* load in site default preview text from config */
            $previewText = $this->getDefaultPreviewText($this->fontData['fontName']);
        }

        /* try to pick some bits up from the session */
        if (strlen($_SESSION['customPreviewText']))
        {
            $previewText = $_SESSION['customPreviewText'];
        }
        if (strlen($_SESSION['customPreviewTextColour']))
        {
            $textColour = $_SESSION['customPreviewTextColour'];
        }

        // if we have a preview text override, use it
        if (strlen($this->fontData['previewOverride']) > 0)
        {
            $previewText = $this->fontData['previewOverride'];
        }

        $textColour = str_replace("#", "", $textColour);

        /* compile font preview md5 filename */
        $cachedPreviewFilenameMD5  = MD5($previewText . $previewWidth . $previewHeight . $textColour);
        $cachedPreviewFilenameFull = $cachedPreviewFilenameMD5 . ".png";
        if (SITE_CONFIG_CACHE_PREVIEW_IMAGES == "yes")
        {
            if (!is_dir(FONTS_PREVIEW_CACHE_FOLDER . $this->fontId))
                @mkdir(FONTS_PREVIEW_CACHE_FOLDER . $this->fontId, 0777);
        }
        $cachedPreviewFilenamePath = FONTS_PREVIEW_CACHE_FOLDER . $this->fontId . "/" . $cachedPreviewFilenameFull;

        /* check preview cache */
        if (!file_exists($cachedPreviewFilenamePath))
        {
            /* make sure the correct storage folders exist */
            $this->setupStorageDirectories();

            /* make sure we have the font for preview, extract it if it's no already there */
            if (!$this->fontFilePath = $this->getFontForPreview())
            {
                return false;
            }

            /* create the preview image */
			$previewHeight = 55;
            $fontsize  = $previewHeight;
            $im        = @imagecreatetruecolor($previewWidth, $previewHeight) or die("Cannot Initialize new GD image stream");
            imagesavealpha($im, true);
            $rgbColour = self::hexColourToRGBArray($textColour);
            $bbox      = $this->calculateTextBox($fontsize, 0, $this->fontFilePath, $previewText);
            if(($bbox['height'] < $previewHeight))
            {
                $fontsize = $fontsize + 25;
                $bbox      = $this->calculateTextBox($fontsize, 0, $this->fontFilePath, $previewText);
                if(($bbox['height'] < $previewHeight))
                {
                    $fontsize = $fontsize + 25;
                    $bbox      = $this->calculateTextBox($fontsize, 0, $this->fontFilePath, $previewText);
                }
            }
            while (($bbox['height']) > $previewHeight)
            {
                $fontsize = $fontsize - 2;
                $bbox     = $this->calculateTextBox($fontsize, 0, $this->fontFilePath, $previewText);
            }

            /* setup the transparency */
            $black        = imagecolorallocate($im, 0, 0, 0);
            imagefilledrectangle($im, 0, 0, $previewWidth, $previewHeight, $black);
            $trans_colour = imagecolorallocatealpha($im, 0, 0, 0, 127);
            imagefill($im, 0, 0, $trans_colour);

            /* add the text */
            $xpos   = 17;
            $ypos   = $bbox["top"] + (($bbox["height"]+$padding) / 2) - ($bbox["height"] / 2);
            $color  = imagecolorallocate($im, $rgbColour[0], $rgbColour[1], $rgbColour[2]);
            $result = imagettftext($im, $fontsize, 0, $xpos, $ypos, $color, $this->fontFilePath, $previewText);

            /* store the png contents within the object */
            if (SITE_CONFIG_CACHE_PREVIEW_IMAGES == "yes")
            {
                imagepng($im, $cachedPreviewFilenamePath);
                /* get image contents */
                $this->previewImageContents = file_get_contents($cachedPreviewFilenamePath);
            }
            else
            {
                ob_start();
                imagepng($im);
                $this->previewImageContents = ob_get_contents();
                ob_end_clean();
            }
            imagedestroy($im);
        }
        else
        {
            /* get image contents */
            $this->previewImageContents = file_get_contents($cachedPreviewFilenamePath);
        }
        return true;
    }

    function outputPreviewImage()
    {
        if ($this->previewImageContents == null)
        {
            $this->createFontPreview();
        }

        header('Content-type: image/png');
        header('Content-Length: ' . strlen($this->previewImageContents));
        echo $this->previewImageContents;
    }

    function getFontForPreview()
    {
        $archivePath   = FONTS_ARCHIVE_FOLDER . $this->fontId . ".zip";
        $extractedPath = FONTS_UNPACKED_ARCHIVE . $this->fontId . "/";

        /* return it if we have it already */
        if (strlen($this->fontData['fontForPreview']) != 0)
        {
            if (file_exists($extractedPath . $this->fontData['fontForPreview']))
            {
                return $extractedPath . $this->fontData['fontForPreview'];
            }
        }

        /* extract the files */
        if (!$this->extractFile($archivePath, $extractedPath))
        {
            return false;
        }

        /* get a ttf font for the preview */
        $archiveContents = self::readDirectoryContents($extractedPath);
        $ttfFile         = null;
        foreach ($archiveContents AS $archiveContent)
        {
            if ($ttfFile != null)
                continue;
            /* find ttf */
            if (self::getFileExtension($archiveContent) == "ttf")
            {
                $ttfFile = $archiveContent;
            }
        }

        /* try to find otf if we didn't get a ttf */
        if ($ttfFile == null)
        {
            foreach ($archiveContents AS $archiveContent)
            {
                if ($ttfFile != null)
                    continue;
                /* find otf */
                if (self::getFileExtension($archiveContent) == "otf")
                {
                    $ttfFile = $archiveContent;
                }
            }
        }

        if (!$ttfFile)
        {
            return false;
        }

        /* update db */
        $this->updateFontTTFNameForPreview($ttfFile);

        /* return font path */
        return $extractedPath . $this->fontData['fontForPreview'];
    }

    function extractFile($pathFromZip, $extractedPathDir)
    {
        /* create extract folder if it's not there */
        if (!is_dir($extractedPathDir))
            @mkdir($extractedPathDir, 0777);

        /* extract font */
        $archive = new PclZip($pathFromZip);
        if ($archive->extract(str_replace("\\", "/", $extractedPathDir)) <= 0)
        {
            return false;
        }
        return true;
    }

    function updateFontTTFNameForPreview($fonttTTFName)
    {
        $db = Database::getDatabase();
        $db->query('UPDATE font SET fontForPreview = :fontForPreview WHERE id = :id', array('fontForPreview' => $fonttTTFName, 'id'             => $this->fontId));
        $this->fontData['fontForPreview'] = $fonttTTFName;
        return $db->affectedRows();
    }

    function getPreviewUrl()
    {
        return WEB_ROOT . "/font_preview.php?f=" . $this->fontId . "&r=" . time();
    }

    function getCharacterMapUrl()
    {
        return WEB_ROOT . "/font_preview_character_map.php?f=" . $this->fontId . "&r=" . time();
    }

    function hasUserIPRated($userIP)
    {
        $db = Database::getDatabase();
        if ($db->getValue("SELECT COUNT(id) AS total FROM rating WHERE fontId = " . (int) $this->fontId . " AND ipAddress = " . $db->quote($userIP)))
        {
            return true;
        }
        return false;
    }

    function createCharacterMap()
    {
        /* compile font character map md5 filename */
        $previewWidth  = 550;
        $previewHeight = 954;
        $textColour    = "000000";
        if (strlen($_SESSION['customPreviewTextColour']))
        {
            $textColour                = $_SESSION['customPreviewTextColour'];
        }
        $boxColour                 = "cccccc";
        $previewText               = "character_map";
        $cachedPreviewFilenameMD5  = MD5($previewText . $previewWidth . $previewHeight . $textColour);
        $cachedPreviewFilenameFull = $cachedPreviewFilenameMD5 . ".png";
        if (SITE_CONFIG_CACHE_PREVIEW_IMAGES == "yes")
        {
            if (!is_dir(FONTS_PREVIEW_CACHE_FOLDER . $this->fontId))
                @mkdir(FONTS_PREVIEW_CACHE_FOLDER . $this->fontId, 0777);
        }
        $cachedPreviewFilenamePath = FONTS_PREVIEW_CACHE_FOLDER . $this->fontId . "/" . $cachedPreviewFilenameFull;

        /* check preview cache */
        if (!file_exists($cachedPreviewFilenamePath))
        {
            /* make sure the correct storage folders exist */
            $this->setupStorageDirectories();

            /* make sure we have the font for preview, extract it if it's no already there */
            if (!$this->fontFilePath = $this->getFontForPreview())
            {
                return false;
            }

            /* create the preview image */
            $im           = @imagecreatetruecolor($previewWidth, $previewHeight) or die("Cannot Initialize new GD image stream");
            imagesavealpha($im, true);
            $rgbColour    = self::hexColourToRGBArray($boxColour);
            $boxRGBColour = imagecolorallocate($im, $rgbColour[0], $rgbColour[1], $rgbColour[2]);

            /* setup the transparency */
            $black        = imagecolorallocate($im, 0, 0, 0);
            imagefilledrectangle($im, 0, 0, $previewWidth, $previewHeight, $black);
            $trans_colour = imagecolorallocatealpha($im, 0, 0, 0, 127);
            imagefill($im, 0, 0, $trans_colour);

            /* write the initial grid */
            $fontsize        = 30;
            $boxWidth        = 60;
            $boxHeight       = 80;
            $x               = 0;
            $y               = 0;
            $rgbColour       = self::hexColourToRGBArray($textColour);
            $textColor       = imagecolorallocate($im, $rgbColour[0], $rgbColour[1], $rgbColour[2]);
            $characterSetArr = array(range('A', 'Z'), range('a', 'z'), range('0', '9'), array(".", ",", ";", ":", "@", "#", "'", "!", "\"", "/", "?", "<", ">", "%", "&", "*", "(", ")", "�", "$"));
            foreach ($characterSetArr AS $characterSet)
            {
                foreach ($characterSet AS $character)
                {
                    /* write the box */
                    imagerectangle($im, $x, $y, $x + $boxWidth, $y + $boxHeight, $boxRGBColour);

                    /* write system character */
                    imagettftext($im, 12, 0, $x + 5, $y + 17, $boxRGBColour, FONTS_CHARACTER_MAP_SYSTEM_FONT, $character);

                    /* write font character */
                    $bbox      = imagettfbbox($fontsize, 0, $this->fontFilePath, $character);
                    $charWidth = $bbox[0] + $bbox[2];
                    $positionX = $boxWidth - $charWidth;
                    if ($positionX < 2)
                        $positionX = 2;
                    $positionX = ceil($positionX / 2) + $x;
                    imagettftext($im, $fontsize, 0, $positionX, $y + 57, $textColor, $this->fontFilePath, $character);

                    /* update positions */
                    $x = $x + $boxWidth;
                    if (($x + $boxWidth) >= $previewWidth)
                    {
                        $x = 0;
                        $y = $y + ($boxHeight);
                    }
                }

                /* set new section */
                $y = $y + ($boxHeight) + 24;
                $x = 0;
            }

            /* store the png contents within the object */
            if (SITE_CONFIG_CACHE_PREVIEW_IMAGES == "yes")
            {
                imagepng($im, $cachedPreviewFilenamePath);
                /* get image contents */
                $this->characterMapContents = file_get_contents($cachedPreviewFilenamePath);
            }
            else
            {
                ob_start();
                imagepng($im);
                $this->characterMapContents = ob_get_contents();
                ob_end_clean();
            }
            imagedestroy($im);
        }
        else
        {
            /* get image contents */
            $this->characterMapContents = file_get_contents($cachedPreviewFilenamePath);
        }
        return true;
    }

    function outputCharacterMapImage()
    {
        if ($this->characterMapContents == null)
        {
            $this->createCharacterMap();
        }

        header('Content-type: image/png');
        header('Content-Length: ' . strlen($this->characterMapContents));
        echo $this->characterMapContents;
    }

    static function hexColourToRGBArray($hexColour)
    {
        return array(
            base_convert(substr($hexColour, 0, 2), 16, 10),
            base_convert(substr($hexColour, 2, 2), 16, 10),
            base_convert(substr($hexColour, 4, 2), 16, 10),
        );
    }

    static function readDirectoryContents($directory)
    {
        $directoryListing = array();
        if ($handle = opendir($directory))
        {
            while (false !== ($file = readdir($handle)))
            {
                if (filetype($directory . $file) != 'dir')
                {
                    $directoryListing[] = $file;
                }
            }
            closedir($handle);
        }
        return $directoryListing;
    }

    static function getFileExtension($filename)
    {
        return end(explode(".", strtolower($filename)));
    }

    static function getDefaultPreviewText($fontName = null)
    {
        switch (SITE_CONFIG_DEFAULT_PREVIEW_TEXT_TYPE)
        {
            case "Site Name":
                return SITE_CONFIG_SITE_NAME;
                break;
            case "Font Name":
                return $fontName ? $fontName : SITE_CONFIG_SITE_NAME;
                break;
            default:
                return SITE_CONFIG_DEFAULT_CUSTOM_FONT_PREVIEW_TEXT ? SITE_CONFIG_DEFAULT_CUSTOM_FONT_PREVIEW_TEXT : SITE_CONFIG_SITE_NAME;
                break;
        }
    }

    static function setupStorageDirectories()
    {
        if (!is_dir(FONTS_BASE_STORAGE_FOLDER))
            @mkdir(FONTS_BASE_STORAGE_FOLDER, 0777);
        if (!is_dir(FONTS_ARCHIVE_FOLDER))
            @mkdir(FONTS_ARCHIVE_FOLDER, 0777);
        if (!is_dir(FONTS_PREVIEW_CACHE_FOLDER))
            @mkdir(FONTS_PREVIEW_CACHE_FOLDER, 0777);
        if (!is_dir(FONTS_UNPACKED_ARCHIVE))
            @mkdir(FONTS_UNPACKED_ARCHIVE, 0777);
    }

    static function createFontDetailsUrl($fontId, $fontName)
    {
        return WEB_ROOT . "/fonts/" . (int) $fontId . "/" . htmlHelpers::formatForUrl($fontName) . "." . SITE_CONFIG_PAGE_EXTENSION;
    }

    static function createFontDownloadUrl($fontId, $fontName)
    {
        return WEB_ROOT . "/d/" . (int) $fontId . "/" . htmlHelpers::formatForUrl($fontName) . ".zip";
    }

    static function getFontsPerPageSetting()
    {
        return SITE_CONFIG_FONT_RESULTS_PER_PAGE;
    }

    static function getFileTypeBasedOnFilename($extension)
    {
        $extension = strtolower(trim($extension));
        $extension = self::getFileExtension($extension);
        $fileType  = "unknown";
        switch ($extension)
        {
            case "ttf":
            case "otf":
                $fileType = "Font File";
                break;
            case "txt":
                $fileType = "Text File";
                break;
            case "lnk":
                $fileType = "Link File";
                break;
            case "jpg":
            case "jpeg":
            case "png":
            case "gif":
            case "bmp":
            case "ebmp":
                $fileType = "Image";
                break;
            case "zip":
            case "rar":
                $fileType = "Archive";
                break;
        }
        return $fileType;
    }

    function getFontCategoryNames()
    {
        $db = Database::getDatabase();
        $rs = $db->getRows("SELECT font_categories.categoryName, font_categories.id AS categoryId FROM font_categories LEFT JOIN font_categories_join ON font_categories.id = font_categories_join.categoryId WHERE font_categories_join.fontId = " . (int) $this->fontId);
        if ($rs)
        {
            return $rs;
        }
        return false;
    }

}
