<?php
namespace SabaiApps\Directories\Component\Entity\Helper;

use SabaiApps\Directories\Application;
use SabaiApps\Directories\Exception;
use SabaiApps\Directories\Component\Field\IField;

class FieldHelper
{
    private $_fields = [];

    /**
     * Returns a field object of an entity
     * @param Application $application
     * @param Entity\Type\IEntity|Entity\Model\Bundle|string $entityOrBundle
     * @param string $fieldName
     */
    public function help(Application $application, $entityOrBundle, $fieldName = null, $componentName = null, $group = '')
    {
        if ($entityOrBundle instanceof \SabaiApps\Directories\Component\Entity\Type\IEntity) {
            $bundle_name = $entityOrBundle->getBundleName();
        } elseif ($entityOrBundle instanceof \SabaiApps\Directories\Component\Entity\Model\Bundle) {
            $bundle_name = $entityOrBundle->name;
        } elseif (!is_string($entityOrBundle)) {
            throw new Exception\InvalidArgumentException();
        } else {
            $bundle_name = isset($componentName) ? $application->Entity_Bundle($entityOrBundle, $componentName, $group)->name : $entityOrBundle;
        }
        // Check if fields for the entity are already loaded
        if (!isset($this->_fields[$bundle_name])) {
            // Load fields
            $this->_fields[$bundle_name] = [];
            foreach ($application->Entity_Bundle($bundle_name)->with('Fields', 'FieldConfig')->Fields as $field) {
                $this->_fields[$bundle_name][$field->getFieldName()] = $field;
            }
        }
        if (isset($fieldName)) {
            return isset($this->_fields[$bundle_name][$fieldName]) ? $this->_fields[$bundle_name][$fieldName] : null;
        }

        return $this->_fields[$bundle_name];
    }

    protected function _getEntity(Application $application, $entity)
    {
        if (!$entity instanceof \SabaiApps\Directories\Component\Entity\Type\IEntity) {
            if (is_array($entity)) {
                $entity_id = $entity[0];
                $entity_type = $entity[1];
            } else {
                $entity_id = $entity;
                $entity_type = 'post';
            }
            if (!$entity = $application->Entity_Entity($entity_type, $entity_id)) {
                return;
            }
        }
        return $entity;
    }

    public function options(Application $application, $entityOrBundle, $interface = null, $emptyValue = null, $prefix = '', $interfaceExclude = null)
    {
        $fields = [];
        if (isset($interface)) $interface = '\SabaiApps\Directories\Component\\' . $interface;
        if (isset($interfaceExclude)) $interfaceExclude = '\SabaiApps\Directories\Component\\' . $interfaceExclude;
        foreach ($this->help($application, $entityOrBundle) as $field_name => $field) {
            if (!$field_type = $application->Field_Type($field->getFieldType(), true)) continue;

            if (isset($interface)
                && !$field_type instanceof $interface
            ) continue;

            if (isset($interfaceExclude)
                && $field_type instanceof $interfaceExclude
            ) continue;

            $fields[$field_name] = (string)$prefix . $field->getFieldLabel() . ' (' . $field_type->fieldTypeInfo('label') . ')';
        }
        if (!empty($fields)) {
            asort($fields);
            if (isset($emptyValue)) {
                $fields = array($emptyValue => __('— Select —', 'directories')) + $fields;
            }
        }

        return $fields;
    }

    public function render(Application $application, $entity, $fieldName, $rendererName, array $settings, array $values = null, $index = null)
    {
        return $this->renderBySettingsReference($application, $entity, $fieldName, $rendererName, $settings, $values, $index);
    }

    public function renderBySettingsReference(Application $application, $entity, $fieldName, $rendererName, array &$settings, array $values = null, $index = null)
    {
        if (!$entity = $this->_getEntity($application, $entity)) {
            $application->logError('Invalid entity ' . $entity);
            return '';
        }
        if (!$field = $this->help($application, $entity, $fieldName)) {
            $application->logError('Invalid field ' . $fieldName);
            return '';
        }
        if (!isset($values)
            && (!$values = $entity->getFieldValue($field->getFieldName()))
        ) {
            return '';
        }

        if (isset($index)) {
            if (!array_key_exists($index, $values)) return '';

            $values = [$values[$index]];
        }
        try {
            $renderer = $application->Field_Renderers_impl($rendererName);
            if ($default_settings = $renderer->fieldRendererInfo('default_settings')) {
                $settings += $default_settings;
            }
            $html = $renderer->fieldRendererRenderField($field, $settings, $entity, $values);
            if (!is_array($html) && !strlen($html)) return '';
        } catch (Exception\IException $e) {
            $application->logError($e);
            return '';
        }

        return $application->Filter('entity_field_render', $html, array($entity, $field, $rendererName, $settings));
    }
}
