<?php
namespace SabaiApps\Directories\Component\Display\Helper;

use SabaiApps\Directories\Application;
use SabaiApps\Directories\Assets;
use SabaiApps\Directories\Component\Entity;
use SabaiApps\Directories\Component\Display\Element\IElement;
use SabaiApps\Directories\Exception;

class DisplayHelper
{
    protected $_displays = []; // runtime cache
    
    public function help(Application $application, $entityOrBundleName, $displayName = 'detailed', $type = 'entity', $useCache = true, $create = false)
    {
        $bundle = $application->Entity_Bundle($entityOrBundleName);
        if ($displayName instanceof \SabaiApps\Directories\Component\Display\Model\Display) {
            $display = $displayName;
            $displayName = $displayName->name;
            $type = $displayName->type;
        } else {
            $display = null;
        }
        
        if (!$useCache
            || !isset($this->_displays[$bundle->name][$type][$displayName])
        ) {
            if ($display = $this->_getDisplay($application, $bundle, $displayName, $type, $useCache, $create, $display)) {
                $display = $application->Filter('display_display', $display, array($bundle, $type, $displayName));
            }
            $this->_displays[$bundle->name][$type][$displayName] = $display;
        }
        
        return $this->_displays[$bundle->name][$type][$displayName];
    }
    
    protected function _getDisplay(Application $application, Entity\Model\Bundle $bundle, $displayName, $type, $useCache, $create, $display = null)
    {        
        $cache_id = $this->_getCacheId($bundle->name, $type, $displayName);
        if (!$useCache ||
            (!$ret = $application->getPlatform()->getCache($cache_id))
        ) {
            if (!isset($display)) {
                if (!$display = $application->getModel('Display', 'Display')
                    ->name_is($displayName)
                    ->type_is($type)
                    ->bundleName_is($bundle->name)
                    ->fetchOne()
                ) {
                    if (!$create) return;
                        
                    $display = $application->getModel('Display', 'Display')->create()->markNew();
                    $display->name = $displayName;
                    $display->type = $type;
                    $display->bundle_name = $bundle->name;
                    $display->commit();
                }
            }
            
            $ret = array(
                'id' => $display->id,
                'name' => $display->name,
                'elements' => [],
                'type' => $display->type,
                'check_roles' => false,
                'pre_render' => false,
                'bundle_name' => $display->bundle_name,
                'css' => isset($display->data['css']) && strlen($display->data['css']) ? $display->data['css'] : null,
                'class' => $display->getCssClass(),
                'is_amp' => $display->isAmp(),
            );
            
            $elements = [];
            foreach ($display->Elements as $element) {
                if ((!$element_impl = $application->Display_Elements_impl($bundle, $element->name, true))
                    || !$element_impl->displayElementSupports($bundle, $display)
                    || (!$element_data = $this->_getElementData($application, $bundle, $element, $element_impl))
                ) continue;

                $element_data = $application->Filter('display_element_data', $element_data, array($bundle, $type, $displayName, $element->data));
                if (!$element_impl->displayElementIsEnabled($bundle, $element_data, $display)) continue;
                
                $elements[$element->parent_id][$element->id] = $element_data;
            }
            $this->_getElementTree($ret['elements'], $elements);
            
            // Cache which elements should be pre-rendered, and assets required for render
            $assets = new Assets();
            foreach (array_keys($ret['elements']) as $element_id) {
                $element =& $ret['elements'][$element_id];
                $element_impl = $application->Display_Elements_impl($bundle, $element['name']);
                if ($element_impl->displayElementIsPreRenderable($bundle, $element, $display->type)) {
                    $ret['pre_render'] = true; // pre render display
                    $element['pre_render'] = true; // pre render element
                }
                $element_impl->displayElementAssets($bundle, $element, $assets);
            }
            $ret['assets'] = $assets->getAssets();
            
            $ret = $application->Filter('display_cache_display', $ret, array($bundle, $type, $displayName));
            $application->getPlatform()->setCache($ret, $cache_id);
        }
        
        return $ret;
    }
    
    protected function _getElementTree(&$tree, array $elements, $parentId = 0)
    {
        if (empty($elements[$parentId])) return;
        
        uasort($elements[$parentId], function ($a, $b) { return $a['weight'] < $b['weight'] ? -1 : 1; });
        foreach ($elements[$parentId] as $element_id => $element) {
            $tree[$element_id] = $element;
            $this->_getElementTree($tree[$element_id]['children'], $elements, $element_id);
        }
    }
    
    protected function _getElementData(Application $application, Entity\Model\Bundle $bundle, $element, IElement $impl)
    {
        try {
            $info = $impl->displayElementInfo($bundle);
        } catch (Exception\RuntimeException $e) {
            $application->logError($e);
            return;
        }
        $data = $element->data;
        $data['settings'] += $info['default_settings'];
        $classes = [
            'drts-display-element',
            'drts-display-element-' . $element->id,
        ];
        if ($inlineable = $impl->displayElementIsInlineable($bundle, $data['settings'])) {
            $classes[] = 'drts-display-element-inlineable';
        }
        if (isset($info['class'])) {
            $classes[] = $info['class'];
        }

        return array(
            'id' => $element->id,
            'name' => $element->name,
            'label' => $info['label'],
            'settings' => $data['settings'],
            'title' => $impl->displayElementAdminTitle($bundle, $data),
            'admin_attr' => $impl->displayElementAdminAttr($bundle, $data['settings']),
            'dimmed' => $impl->displayElementIsDimmed($bundle, $data['settings'])
                || (!empty($data['visibility']['globalize']) && !empty($data['visibility']['globalize_remove'])),
            'type' => $info['type'],
            'class' => implode(' ', $classes),
            'containable' => !empty($info['containable']),
            'cloneable' => !empty($info['cloneable']),
            'weight' => $element->weight,
            'children' => [],
            'child_element_type' => isset($info['child_element_type']) ? $info['child_element_type'] : null,
            'child_element_name' => isset($info['child_element_name']) ? $info['child_element_name'] : null,
            'add_child_label' => isset($info['add_child_label']) ? $info['add_child_label'] : __('Add Element', 'directories'),
            'parent_element_name' => isset($info['parent_element_name']) ? $info['parent_element_name'] : null,
            'heading' => empty($data['heading']) ? [] : $data['heading'],
            'visibility' => empty($data['visibility']) ? [] : $data['visibility'],
            'advanced' => empty($data['advanced']) ? [] : $data['advanced'],
            'system' => $element->system ? true : false,
            'inlineable' => $inlineable,
            'icon' => isset($info['icon']) ? $info['icon'] : null,
            'readable_settings' => $impl->displayElementReadableSettings($bundle, $data['settings']),
        );
    }
    
    protected function _getStyle(Application $application, array $design)
    {
        $styles = [];
        if (!empty($design['add_border'])) {
            $styles['border'] = sprintf(
                '%01.1fpx %s %s',
                empty($design['border']['width']) ? 0 : $design['border']['width'],
                $design['border']['style'],
                $design['border']['color'][1]
            );        
            if (!empty($design['border']['radius'])) {
                $styles['border-radius'] = sprintf('%01.1fpx', $design['border']['radius']);
            }
        }
        
        foreach (array('margin', 'padding') as $key) {
            if (empty($design['add_' . $key])) continue;
            
            if (isset($design[$key]['top'])
                || isset($design[$key]['right'])
                || isset($design[$key]['bottom'])
                || isset($design[$key]['left'])
            ) {
                $styles[$key] = sprintf(
                    '%2$01.1f%1$s %3$01.1f%1$s %4$01.1f%1$s %5$01.1f%1$s',
                    isset($design[$key . '_unit']) && $design[$key . '_unit'] === 'em' ? 'em' : 'px',
                    $design[$key]['top'],
                    $design[$key]['right'],
                    $design[$key]['bottom'],
                    $design[$key]['left']
                );
            }
        }
        
        if (!empty($design['set_bg'])) {
            if (isset($design['bg_color']) && strlen($design['bg_color'])) {
                $styles['background-color'] = $design['bg_color'];
            }
        }
        
        if (!empty($design['align']['enable'])) {
            $align = false;
            if ($design['align']['value'] === 'center') {
                $align = 'center';
            } else {
                if ($application->getPlatform()->isRtl() ? $design['align']['value'] === 'left' : $design['align']['value'] === 'right') {
                    $align = $design['align']['value'];
                }
            }
            if ($align) {
                $styles['text-align'] = $align;
            }
        }
        
        if (!empty($design['position']['enable'])) {
            $styles['z-index'] = 1001;
            $styles['position'] = 'absolute';
            foreach (array('top', 'bottom', 'left', 'right') as $pos) {
                if (isset($design['position']['value'][$pos])
                    && strlen($design['position']['value'][$pos])
                ) {
                    $styles[$pos] = sprintf('%01.1fpx', $design['position']['value'][$pos]);
                }
            }
        }
        
        if (!empty($design['set_font'])) {
            if (!empty($design['font']['color'])) {
                $styles['color'] = $design['font']['color'];
            }
            if ($design['font']['size_unit'] === 'em') {
                if (!empty($design['font']['size_em'])) {
                    $styles['font-size'] = sprintf('%01.2fem', $design['font']['size_em']);
                }
            } else {
                if (!empty($design['font']['size_px'])) {
                    $styles['font-size'] = sprintf('%01.1fpx', $design['font']['size_px']);
                }
            }
            if (!empty($design['font']['weight'])) {
                $styles['font-weight'] = $design['font']['weight'];
            }
            if (!empty($design['font']['style'])) {
                $styles['font-style'] = $design['font']['style'];
            }
        }
        
        if (!empty($design['overflow'])) {
            $styles['overflow'] = $design['overflow'];
        }
        
        if (empty($styles)) return '';
        
        $ret = [];
        foreach ($styles as $key => $value) {
            $ret[] = $key . ':' . $application->H($value);
        }
        
        return implode(';', $ret);
    }
    
    protected function _getCacheId($bundleName, $type, $displayName)
    {
        return 'display_display_' . $bundleName . '_' . $type . '_' . $displayName;
    }
    
    public function clearCache(Application $application, $bundleName, $type = null, $displayName = null)
    {
        if ($bundleName instanceof \SabaiApps\Directories\Component\Display\Model\Display) {
            $type = $bundleName->type;
            $displayName = $bundleName->name;
            $bundleName = $bundleName->bundle_name;
        }
        $application->getPlatform()->deleteCache($this->_getCacheId($bundleName, $type, $displayName));
    }
}