<?php
namespace SabaiApps\Directories\Component\CSV\Controller\Admin;

use SabaiApps\Directories\Component\Form;
use SabaiApps\Directories\Component\Field;
use SabaiApps\Directories\Context;
use SabaiApps\Directories\Exception;

class Import extends Form\AbstractMultiStepController
{    
    protected function _getBundle(Context $context)
    {
        return $context->bundle;
    }
    
    protected function _doExecute(Context $context)
    {
        parent::_doExecute($context);
        $this->getPlatform()->addCssFile('csv-admin.min.css', 'drts-csv-admin', array('drts'), 'directories');
    }
    
    protected function _getSteps(Context $context, array &$formStorage)
    {
        return array('upload' => [], 'map_fields' => [], 'importer_settings' => [], 'import' => []);
    }
    
    public function _getFormForStepUpload(Context $context, array &$formStorage)
    {
        $form = array(
            '#validate' => array(array(array($this, 'validateUpload'), array($context))),
            'file' => array(
                '#type' => 'file',
                '#title' => __('CSV file(s)', 'directories'),
                '#description' => __('Select one or more CSV files to import.', 'directories'),
                '#upload_dir' => $this->getComponent('System')->getTmpDir(),
                '#allowed_extensions' => array('csv'),
                '#required' => true,
                // The finfo_file function used by the uploader to check mime types for CSV files is buggy. We can skip it safely here since this is for admins only.
                '#skip_mime_type_check' => true,
                '#multiple' => true,
            ),
            'delimiter' => array(
                '#type' => 'textfield',
                '#title' => __('CSV column delimiter', 'directories'),
                '#size' => 5,
                '#description' => __('Enter the character used as CSV column delimiters.', 'directories'),
                '#min_length' => 1,
                '#max_length' => 1,
                '#default_value' => ',',
                '#required' => true,
            ),
            'enclosure' => array(
                '#type' => 'textfield',
                '#title' => __('CSV column enclosure', 'directories'),
                '#size' => 5,
                '#description' => __('Enter the character used as CSV column enclosures.', 'directories'),
                '#min_length' => 1,
                '#max_length' => 1,
                '#default_value' => '"',
                '#required' => true,
            ),
        );
        
        foreach (array_keys($form) as $key) {
            if (strpos($key, '#') !== 0 ) {
                $form[$key]['#horizontal'] = true;
            }
        }
        
        return $form;
    }
    
    public function validateUpload($form, $context)
    {
        @setlocale(LC_ALL, $this->getPlatform()->getLocale());
        $delimiter = $form->values['delimiter'];
        $enclosure = $form->values['enclosure'];
        
        // Extract headers of each CSV file and a first row
        $csv_columns_sorted = null;
        foreach ($form->values['file'] as $i => $csv_file) {
            if (!isset($csv_columns_sorted)) {
                $rows = $this->_getCsvRows($csv_file, $delimiter, $enclosure, 2);
                $csv_columns_sorted = $rows[0];
                sort($csv_columns_sorted);
            } else {
                $rows = $this->_getCsvRows($csv_file, $delimiter, $enclosure, 1);
                sort($rows[0]);
                if ($rows[0] !== $csv_columns_sorted) {
                    $form->setError(sprintf(__('CSV file %s contains headers not found in other CSV file(s).', 'directories'), $csv_file['name']));
                }
            }
        }
    }
    
    protected function _getCsvRows($file, $delimiter, $enclosure, $limit = 1)
    {
        $ret = [];
        if (!ini_get('auto_detect_line_endings')) {
            ini_set('auto_detect_line_endings', true);
        }
        if (is_array($file)) $file = $file['saved_file_path'];
        if (false === $fp = fopen($file, 'r')) {
            throw new Exception\RuntimeException(sprintf('An error occurred while opening file %s.', $file));
        }
        while ($limit) {
            $row = fgetcsv($fp, 0, $delimiter, $enclosure);
            if (false === $row) {
                fclose($fp);
                throw new Exception\RuntimeException(sprintf('Failed reading row from CSV file %s.', is_array($file) ? $file['name'] : $file));
            }
            if (is_array($row)
                && array(null) !== $row
            ) {
                $ret[] = $row;
                --$limit;
            }
        }
        fclose($fp);
        return $ret;
    }
    
    public function _getFormForStepMapFields(Context $context, array &$formStorage)
    {
        @setlocale(LC_ALL, $this->getPlatform()->getLocale());
        $delimiter = $formStorage['values']['upload']['delimiter'];
        $enclosure = $formStorage['values']['upload']['enclosure'];
        $csv_files = $formStorage['values']['upload']['file'];
        
        // Extract header and a first row
        $csv_file = array_shift($csv_files);
        $rows = $this->_getCsvRows($csv_file, $delimiter, $enclosure, 2);
        $csv_columns = $rows[0];
        $csv_row1 = $rows[1];
        
        $options = [];
        $optgroups = $custom_field_options = [];
        $importers_by_field_type = $this->CSV_Importers(true);
        $bundle = $this->_getBundle($context);
        $fields = $this->Entity_Field($bundle->name);
        $properties = $this->Entity_Types_impl($bundle->entitytype_name)->entityTypeInfo('properties');
        $id_column_field_type = $properties['id']['type'];
        $id_column_key = null;
        foreach ($fields as $field_name => $field) {
            if ((!$importer_name = @$importers_by_field_type[$field->getFieldType()])
                || (!$importer = $this->CSV_Importers_impl($importer_name, true))
                || !$importer->csvImporterSupports($bundle, $field)
            ) continue;
                
            $columns = $importer->csvImporterInfo('columns');
            if (is_array($columns)) {
                $optgroups[$importer_name] = $this->_getFieldLabel($field);
                if ($field->isCustomField()) {
                    foreach ($columns as $column => $label) {
                        $custom_field_options[$this->_getFieldOptionValue($field_name, $column)] = array(
                            '#group' => $importer_name,
                            '#title' => $label,
                        );
                    }
                } else {
                    foreach ($columns as $column => $label) {
                        $options[$this->_getFieldOptionValue($field_name, $column)] = array(
                            '#group' => $importer_name,
                            '#title' => $label,
                        );
                    }
                }
            } else {
                $option = $this->_getFieldOptionValue($field_name, (string)$columns);
                if ($field->isCustomField()) {
                    $custom_field_options[$option] = $this->_getFieldLabel($field);
                } else {
                    $options[$option] = $this->_getFieldLabel($field);
                    if ($field->getFieldType() === $id_column_field_type) {
                        $id_column_key = $option;
                    }
                }
            }
        }
        asort($options);
        if (!empty($custom_field_options)) {
            asort($custom_field_options);
            $options += $custom_field_options;
        }
        $options = array('' => __('— Select —', 'directories')) + $options;
        
        $form = array(
            '#header' => array(
                '<div class="drts-bs-alert drts-bs-alert-info">' . $this->H(__('Set up the associations between the CSV file columns and content fields.', 'directories')) . '</div>',
            ),
            '#options' => $options,
            '#optgroups' => $optgroups,
            'header' => array(
                '#type' => 'markup',
                '#markup' => '<table class="drts-bs-table drts-csv-table"><thead><tr><th style="width:25%;">' . __('Column Header', 'directories') . '</th>'
                    . '<th style="width:35%;">' . __('Row 1', 'directories') . '</th>'
                    . '<th style="width:40%;">' . __('Select Field', 'directories') . '</th></tr></thead><tbody>',
            ),
            'fields' => array(
                '#tree' => true,
                '#element_validate' => array(array(array($this, 'validateMapFields'), array($context, $fields, $bundle))),
            ),
            'footer' => array(
                '#type' => 'markup',
                '#markup' => '</tbody></table>',
            ),
        );
        
        foreach ($csv_columns as $column_key => $column_name) {    
            if (isset($csv_row1[$column_key])) {
                $column_value = $csv_row1[$column_key];
                if (strlen($column_value) > 300) {
                    $column_value = $this->Summarize($column_value, 300);
                }
                if (strlen($column_value)) {
                    $column_value = '<code>' . $this->H($column_value) . '</code>';
                }
            } else {
                $column_value = '';
            }
            $form['fields'][$column_name] = array(
                '#prefix' => '<tr><td>' . $this->H($column_name) . '</td><td>' . $column_value . '</td><td>',
                '#suffix' => '</td></tr>',
                '#type' => 'select',
                '#options' => $options,
                '#default_value' => isset($options[$column_name]) && $column_name !== $id_column_key ? $column_name : null, // do not make id column selected by default
                '#optgroups' => $optgroups,
            );
        }
        
        return $form;
    }
    
    protected function _getFieldOptionValue($fieldName, $column = '')
    {
        return strlen($column) ? $fieldName . '__' . $column : $fieldName;
    }
    
    protected function _getFieldLabel(Field\IField $field)
    {
        $label = $field->getFieldLabel() . ' (' . $field->getFieldName() . ')';
        if ($field->isCustomField()) {
            $label = sprintf(__('Custom field - %s', 'directories'), $label);
        }
        return $label;
    }
    
    public function validateMapFields($form, &$value, $element, $context, $fields, $bundle)
    {
        $value = array_filter($value);
        
        $required_fields = [];
        
        if (!empty($bundle->info['parent'])) {
            $required_fields[] = $bundle->entitytype_name . '_parent';
        }

        // Make sure required fields are going to be imported
        foreach ($required_fields as $field_name) {
            if (isset($fields[$field_name]) && !in_array($this->_getFieldOptionValue($field_name), $value)) {
                $form->setError(sprintf(
                    __('The following field needs to be selected: %s.', 'directories'),
                    $this->_getFieldLabel($fields[$field_name])
                ));
            }
        }
        
        $count = array_count_values($value);
        foreach ($count as $option => $_count) {
            if ($option === '' || $_count <= 1) continue;
            
            $_option = explode('__', $option);
            $field_name = $_option[0];
            $form->setError(sprintf(
                __('You may not associate multiple columns with the field: %s', 'directories'),
                $this->_getFieldLabel($fields[$field_name])
            ));
        }
    }
    
    public function _getFormForStepImporterSettings(Context $context, array &$formStorage)
    {     
        $form = array('settings' => []);
        
        $enclosure = $formStorage['values']['upload']['enclosure'];
        $mapped_fields = $formStorage['values']['map_fields']['fields'];
        $fields = $this->Entity_Field($this->_getBundle($context)->name);
        $importers_by_field_type = $this->CSV_Importers(true);
        foreach ($mapped_fields as $column_name => $mapped_field) {
            if (!$_mapped_field = explode('__', $mapped_field)) continue;

            $field_name = $_mapped_field[0];
            $column = (string)@$_mapped_field[1];      
            
            if (!$field = @$fields[$field_name]) continue;
                    
            $importer_name = $importers_by_field_type[$field->getFieldType()];
            if (!$importer = $this->CSV_Importers_impl($importer_name, true)) {
                continue;
            }
            $info = $importer->csvImporterInfo();
            $parents = array('settings', $field_name);
            if (strlen($column)) {
                $parents[] = $column;
            }
            if ($column_settings_form = $importer->csvImporterSettingsForm($field, (array)@$info['default_settings'], $column, $enclosure, $parents)) {
                foreach (array_keys($column_settings_form) as $key) {
                    if (strpos($key, '#') !== 0 ) {
                        $column_settings_form[$key]['#horizontal'] = true;
                    }
                }
                if (strlen($column)) {
                    $form['settings'][$field_name][$column] = $column_settings_form;
                    $form['settings'][$field_name][$column]['#title'] = $info['columns'][$column];
                    $form['settings'][$field_name][$column]['#collapsible'] = false;
                } else {
                    $form['settings'][$field_name] = $column_settings_form;
                }
                $form['settings'][$field_name]['#collapsible'] = true;
                if (!isset($form['settings'][$field_name]['#title'])) {
                    $form['settings'][$field_name]['#title'] = $this->_getFieldLabel($field);
                }
            }
        }
        if (empty($form['settings'])) {
            return $this->_skipStepAndGetForm($context, $formStorage);
        }
        
        $form['settings']['#tree'] = true;
        $form['#header'][] = '<div class="drts-bs-alert drts-bs-alert-info">' . __('Please configure additional options for each field.', 'directories') . '</div>';
        
        return $form;
    }
    
    public function _getFormForStepImport(Context $context, array &$formStorage)
    {
        $context->addTemplate('system_progress');
        $this->_ajaxSubmit = true;
        $this->_ajaxOnSubmit = $this->System_Progress_formSubmitJs('csv_import');
        $this->_ajaxOnSuccessRedirect = $this->_ajaxOnErrorRedirect = false;
        $this->_submitButtons[] = array('#btn_label' => __('Import Now', 'directories'), '#btn_color' => 'primary', '#btn_size' => 'lg');
        
        return array(
            'test' => array(
                '#type' => 'checkbox',
                '#title' => __('Test import', 'directories'),
                '#description' => __('Check this option to test import only and not actually saving data to the database.', 'directories'),
                '#horizontal' => true,
            ),
            'test_num' => array(
                '#type' => 'number',
                '#title' => __('Number of rows to test import', 'directories'),
                '#description' => __('Enter the max number of rows to perform import test, 0 for all rows.', 'directories'),
                '#default_value' => 0,
                '#horizontal' => true,
                '#states' => array(
                    'visible' => array(
                        'input[name="test"]' => array('type' => 'checked', 'value' => true),
                    )
                ),
            ),
        );
    }
    
    public function _submitFormForStepImport(Context $context, Form\Form $form)
    {
        define('DRTS_CSV_IMPORTING', true);
        @set_time_limit(0);        
        $csv_files = $form->storage['values']['upload']['file'];
        $delimiter = $form->storage['values']['upload']['delimiter'];
        $enclosure = $form->storage['values']['upload']['enclosure'];
        $csv_columns = $this->_getCsvRows($csv_files[0], $delimiter, $enclosure);
        $csv_columns = $csv_columns[0];
        $mapped_fields = $form->storage['values']['map_fields']['fields'];
        $importer_settings = (array)@$form->storage['values']['importer_settings']['settings'];
        if ($test = !empty($form->values['test'])) {
            $test_num = $form->values['test_num'];
        }

        $_mapped_fields = [];
        $bundle = $this->_getBundle($context);
        $fields = $this->Entity_Field($bundle->name);
        $importers_by_field_type = $this->CSV_Importers(true);
        $total = $rows_imported = $rows_updated = $num_total = 0;
        $rows_failed = [];
        $previous_file = null;
        while ($csv_file = array_shift($csv_files)) {
            $row_number = 0;
            $file = new \SplFileObject($csv_file['saved_file_path']);
            $file->setFlags(\SplFileObject::READ_CSV | \SplFileObject::READ_AHEAD | \SplFileObject::SKIP_EMPTY);
            $file->setCsvControl($delimiter, $enclosure);
            $file->seek(PHP_INT_MAX);
            $num_total = $file->key() - 1; 
            $progress = $this->System_Progress('csv_import')->start($num_total, $this->H($csv_file['name']) . ': ' . __('Importing (%1$d/%2$d) ... %3$s', 'directories'));
            foreach ($file as $csv_row) {
                if (!is_array($csv_row)
                    || array(null) === $csv_row // skip invalid/empty rows
                ) continue;
                
                ++$row_number;
                
                if ($row_number === 1) continue; // skip header row
                
                ++$total;
                
                if ($test && $test_num && $total > $test_num) break 2;
                                
                $values = [];
                foreach ($csv_columns as $column_index => $column_name) {
                    if (!isset($csv_row[$column_index])
                        || !strlen($csv_row[$column_index])
                    ) continue; // no valid value for this row column

                    if (!isset($_mapped_fields[$column_name])) {
                        if (!isset($mapped_fields[$column_name])
                            || (!$_mapped_fields[$column_name] = explode('__', $mapped_fields[$column_name]))
                        ) {
                            // Unset column since mapped field is invalid, to stop further processing the column
                            unset($csv_columns[$column_index], $_mapped_fields[$column_name]);
                            continue;
                        }
                    }
            
                    // Check importer and field
                    $field_name = $_mapped_fields[$column_name][0];    
                    if ((!$field = @$fields[$field_name])
                        || (!$importer_name = @$importers_by_field_type[$field->getFieldType()])
                        || (!$importer = $this->CSV_Importers_impl($importer_name, true))
                    ) {
                        // Unset column since mapped field is invalid, to stop further processing the column
                        unset($csv_columns[$column_index], $_mapped_fields[$column_name]);
                        continue;
                    }
                
                    $column = (string)@$_mapped_fields[$column_name][1];
                    
                    // Init importer settings
                    if (strlen($column)) {
                        $settings = isset($importer_settings[$field_name][$column]) ? $importer_settings[$field_name][$column] : [];
                    } else {
                        $settings = isset($importer_settings[$field_name]) ? $importer_settings[$field_name] : [];
                    }
                    
                    // Import
                    try {                
                        $field_value = $importer->csvImporterDoImport($field, $settings, $column, $csv_row[$column_index]);
                    } catch (\Exception $e) {
                        $rows_failed[$row_number] = $e->getMessage();
                        continue 2; // abort importing the current row
                    }
                    
                    // Skip if no value to import
                    if (null === $field_value || false === $field_value) {
                        continue;
                    }
                
                    if (is_array($field_value) && isset($values[$field_name])) {
                        foreach ($field_value as $field_index => $_field_value) {
                            if (!isset($values[$field_name][$field_index])) {
                                $values[$field_name][$field_index] = $_field_value;
                            } else {
                                $values[$field_name][$field_index] += $_field_value;
                            }
                        }
                    } else {
                        $values[$field_name] = $field_value;
                    }
                }

                try {
                    if (!empty($values[$bundle->entitytype_name . '_id'])
                        && ($entity = $this->Entity_Entity($bundle->entitytype_name, $values[$bundle->entitytype_name . '_id'], false))
                        && $entity->getBundleName() === $bundle->name
                    ) {
                        if (!$test) {
                            $entity = $this->Entity_Save($entity, $values);
                        }
                        ++$rows_updated;
                        $ret_title = $this->Entity_Title($entity);
                    } else {
                        if (!$test) {
                            $entity = $this->Entity_Save($bundle, $values);
                            $ret_title = $this->Entity_Title($entity);
                        } else {
                            $ret_title = $values[$bundle->entitytype_name . '_title'];
                        }
                        ++$rows_imported;
                    }
                    
                    // Notify
                    $this->Action('csv_import_entity', array($bundle, $entity, $values, $importer_settings));
                    
                    $progress->set($ret_title, true);
                } catch (\Exception $e) {
                    $rows_failed[$row_number] = $ret_title = $e->getMessage();
                    
                    $progress->set($ret_title, false);
                }
            }
            
            $progress->done($csv_file['name'] . ': %s', $more = count($csv_files));
            
            $previous_file = $csv_file['saved_file_path'];
            $file = null;
        }
        
        $form->storage['rows_imported'] = $rows_imported;
        $form->storage['rows_updated'] = $rows_updated;
        $form->storage['rows_failed'] = $rows_failed;
        
        // Cleanup
        foreach (array_keys($_mapped_fields) as $column_name) {            
            $field_name = $_mapped_fields[$column_name][0];
            $column = (string)@$_mapped_fields[$column_name][1];      
            if ((!$field = @$fields[$field_name])
                || (!$importer_name = @$importers_by_field_type[$field->getFieldType()])
                || (!$importer = $this->CSV_Importers_impl($importer_name, true))
            ) {
                continue;
            }
            $settings = isset($importer_settings[$field_name][$column_name]) ? $importer_settings[$field_name][$column_name] : [];
            $importer->csvImporterClean($field, $settings, $column);
        }
    }

    protected function _complete(Context $context, array $formStorage)
    {
        $error = $success = [];
        if (!empty($formStorage['rows_failed'])) {
            foreach ($formStorage['rows_failed'] as $row_num => $error_message) {
                $error[] = $this->H(sprintf(__('CSV data on row number %d could not be imported: %s', 'directories'), $row_num, $error_message));
            }
        }
        if (!empty($formStorage['rows_imported'])) {
            $success[] = $this->H(sprintf(
                __('%d item(s) created successfullly.', 'directories'),
                $formStorage['rows_imported']
            ));
        }
        if (!empty($formStorage['rows_updated'])) {
            $success[] = $this->H(sprintf(
                __('%d item(s) updated successfullly.', 'directories'),
                $formStorage['rows_updated']
            ));
        }
        $context->setSuccess(null, array(
            'success' => $success,
            'error' => $error,
        ));
        @unlink($formStorage['values']['upload']['file']['saved_file_path']);
    }
}