<?php
namespace SabaiApps\Directories\Component\CSV\Controller\Admin;

use SabaiApps\Directories\Component\Form;
use SabaiApps\Directories\Component\Field;
use SabaiApps\Directories\Component\Entity;
use SabaiApps\Directories\Context;
use SabaiApps\Directories\Exception;

class Export extends Form\AbstractMultiStepController
{    
    protected function _getBundle(Context $context)
    {
        return $context->bundle;
    }
        
    protected function _doExecute(Context $context)
    {
        parent::_doExecute($context);
        $this->getPlatform()->addCssFile('csv-admin.min.css', 'drts-csv-admin', array('drts'), 'directories');
    }
    
    protected function _getSteps(Context $context, array &$formStorage)
    {
        return array('select_fields' => [], 'exporter_settings' => [], 'export' => []);
    }
    
    public function _getFormForStepSelectFields(Context $context, array &$formStorage)
    {
        $exporters_by_field_type = $this->CSV_Exporters(true);
        $bundle = $this->_getBundle($context);
        $fields = $this->Entity_Field($bundle->name);
        $options = $custom_field_options = [];
        foreach ($fields as $field_name => $field) {
            if ((!$exporter_name = @$exporters_by_field_type[$field->getFieldType()])
                || (!$exporter = $this->CSV_Exporters_impl($exporter_name, true))
                || !$exporter->csvExporterSupports($bundle, $field)
            ) continue;
            
            $columns = $exporter->csvExporterInfo('columns');
            if (is_array($columns)) {
                if ($field->isCustomField()) {
                    foreach ($columns as $column => $label) {
                        $custom_field_options[$this->_getFieldOptionValue($field_name, $column)] = array(
                            'field' => $this->_getFieldLabel($field) . ' - ' . $label,
                            'column_header' => $field_name . '_' . $column,
                        );
                    }
                } else {
                    foreach ($columns as $column => $column_label) {
                        $options[$this->_getFieldOptionValue($field_name, $column)] = array(
                            'field' => $this->_getFieldLabel($field, $column_label),
                            'column_header' => $field_name . '_' . $column,
                        );
                    }
                }
            } else {
                $option = $this->_getFieldOptionValue($field_name, (string)$columns);
                if ($field->isCustomField()) {
                    $custom_field_options[$option] = array(
                        'field' => $this->_getFieldLabel($field),
                        'column_header' => $field_name,
                    );
                } else {
                    $options[$option] = array(
                        'field' => $this->_getFieldLabel($field),
                        'column_header' => $field_name,
                    );
                }
            }
        }
        uasort($options, function($a, $b) { return strnatcmp($a['field'], $b['field']); });
        if (!empty($custom_field_options)) {
            uasort($custom_field_options, function($a, $b) { return strnatcmp($a['field'], $b['field']); });
            $options += $custom_field_options;
        }
        
        // Disable required fields
        $options_disabled = [];
        foreach ($this->_getAllRequiredFields($context, $bundle) as $field_name) {
            $options_disabled[] = $this->_getFieldOptionValue($field_name);
        }
        
        $form = array(
            '#header' => array(
                '<div class="drts-bs-alert drts-bs-alert-info">' . __('Select the fields to export and configure CSV column headers.', 'directories') . '</div>'
            ),
            'fields' => array(
                '#type' => 'tableselect',
                '#header' => array(
                    'field' => __('Field name', 'directories'),
                    'column_header' => __('Column header', 'directories'),
                ),
                '#multiple' => true,
                '#js_select' => true,
                '#options' => $options,
                '#options_disabled' => $options_disabled,
                '#default_value' => array_keys($options),
                '#element_validate' => array(array(array($this, 'validateSelectFields'), array($context, $fields, $bundle))),
            ),
        );
        
        return $form;
    }
    
    protected function _getFieldOptionValue($fieldName, $column = '')
    {
        return $fieldName . '__' . $column;
    }
    
    protected function _getFieldLabel(Field\IField $field, $columnLabel = '')
    {
        $label = $this->H($field->getFieldLabel()) . ' (' . $field->getFieldName() . ')';
        if ($field->isCustomField()) {
            $label = '<span class="drts-bs-badge drts-bs-badge-secondary">' . $this->H(__('Custom field', 'directories')) . '</span> ' . $label;
        }
        if (strlen($columnLabel)) {
            $label .=  ' - ' . $this->H($columnLabel);
        }
        return $label;
    }
    
    protected function _getAllRequiredFields(Context $context, $bundle)
    {
        $required_fields = [];
        
        if (!empty($bundle->info['parent'])) {
            $required_fields[] = $bundle->entitytype_name . '_parent';
        }
        
        return $required_fields;
    }
    
    public function validateSelectFields($form, &$value, $element, $context, $fields, $bundle)
    {
        $value = array_filter($value);        

        // Make sure required fields are going to be imported
        foreach ($this->_getAllRequiredFields($context, $bundle) as $field_name) {
            if (isset($fields[$field_name])
                && !in_array($option_value = $this->_getFieldOptionValue($field_name), $value)
            ) {
                $value[] = $option_value;
            }
        }
    }
    
    public function _getFormForStepExporterSettings(Context $context, array &$formStorage)
    {
        $form = array('#header' => [], 'settings' => []);

        $selected_fields = $formStorage['values']['select_fields']['fields'];
        $exporters_by_field_type = $this->CSV_Exporters(true);
        $bundle = $this->_getBundle($context);
        $fields = $this->Entity_Field($bundle->name);
        foreach ($selected_fields as $selected_field) {
            if (!$_selected_field = explode('__', $selected_field)) continue;

            $field_name = $_selected_field[0];
            $column = $_selected_field[1];      
            
            if (!$field = @$fields[$field_name]) continue;
                 
            $exporter_name = $exporters_by_field_type[$field->getFieldType()];
            if (!$exporter = $this->CSV_Exporters_impl($exporter_name, true)) {
                continue;
            }
            $info = $exporter->csvExporterInfo();
            $parents = array('settings', $field_name);
            if (strlen($column)) {
                $parents[] = $column;
            }
            if ($column_settings_form = $exporter->csvExporterSettingsForm($field, (array)@$info['default_settings'], $column, '"', $parents)) {
                foreach (array_keys($column_settings_form) as $key) {
                    if (strpos($key, '#') !== 0 ) {
                        $column_settings_form[$key]['#horizontal'] = true;
                    }
                }
                if (strlen($column)) {
                    $form['settings'][$field_name][$column] = $column_settings_form;
                    $form['settings'][$field_name][$column]['#title'] = $info['columns'][$column];
                    $form['settings'][$field_name][$column]['#collapsible'] = false;
                } else {
                    $form['settings'][$field_name] = $column_settings_form;
                }
                $form['settings'][$field_name]['#collapsible'] = true;
                $form['settings'][$field_name]['#title_no_escape'] = true;
                if (!isset($form['settings'][$field_name]['#title'])) {
                    $form['settings'][$field_name]['#title'] = $this->_getFieldLabel($field);
                }
            }
        }
        if (empty($form['settings'])) {
            return $this->_skipStepAndGetForm($context, $formStorage);
        }
        
        $form['settings']['#tree'] = true;
        $form['#header'][] = '<div class="drts-bs-alert drts-bs-alert-info">' . $this->H(__('Please configure additional options for each field.', 'directories')) . '</div>';
        
        return $form;
    }
    
    public function _getFormForStepExport(Context $context, array &$formStorage)
    {
        $bundle = $this->_getBundle($context);
        $context->addTemplate('system_progress');
        $context->download = true;
        $this->_ajaxSubmit = true;
        $this->_ajaxOnSubmit = $this->System_Progress_formSubmitJs('csv_export');
        $this->_ajaxOnSuccess = $this->System_Progress_formSuccessDownloadJs();
        $this->_ajaxOnSuccessRedirect = $this->_ajaxOnErrorRedirect = false;
        $this->_submitButtons[] = array('#btn_label' => __('Export Now', 'directories'), '#btn_color' => 'primary', '#btn_size' => 'lg');
        
        return array(
            'filename' => array(
                '#title' => __('File name', 'directories'),
                '#type' => 'textfield',
                '#field_suffix' => '.csv',
                '#default_value' => $bundle->name . '-' . date('Ymd', time()),
                '#regex' => '/^[a-zA-Z0-9-_]+$/',
                '#required' => true,
                '#horizontal' => true,
            ),
            'limit' => array(
                '#type' => 'number',
                '#title' => __('Limit to X records (0 for all records)', 'directories'),
                '#default_value' => 0,
                '#min_value' => 0,
                '#integer' => true,
                '#horizontal' => true,
            ),
            'offset' => array(
                '#type' => 'number',
                '#title' => __('Start from Xth record', 'directories'),
                '#default_value' => 1,
                '#min_value' => 1,
                '#integer' => true,
                '#horizontal' => true,
            ),
        );
    }
    
    public function _submitFormForStepExport(Context $context, Form\Form $form)
    {
        try {
            $this->ValidateDirectory($this->getComponent('System')->getTmpDir(), true);
        } catch (\Exception $e) {
            throw new Exception\RuntimeException($e->getMessage());
        }
        
        @set_time_limit(0);
        $db = $this->getDB();
        if (false !== strpos(strtolower($db->getConnection()->getScheme()), 'mysql')) {
            try {
                $db->exec('SET SESSION wait_timeout = 600');
            } catch (Exception\IException $e) {
                $this->logError($e);
            }
        }

        $file = rtrim($this->getComponent('System')->getTmpDir(), '/') . '/' . $form->values['filename'] . '.csv';        
        if (!$file
            || false === ($fp = fopen($file, 'w+'))
        ) {
            throw new Exception\RuntimeException(sprintf('Failed opening file %s with write permission', $file));
        }

        $selected_fields = $form->storage['values']['select_fields']['fields'];
        $exporter_settings = (array)@$form->storage['values']['exporter_settings']['settings'];
        $exporters_by_field_type = $this->CSV_Exporters(true);
        $bundle = $this->_getBundle($context);
        $fields = $this->Entity_Field($bundle->name);
        $export_fields = $columns = [];
        foreach ($selected_fields as $selected_field) {
            if (!$_selected_field = explode('__', $selected_field)) continue;

            $field_name = $_selected_field[0];
            $column = $_selected_field[1];      
            
            if (!$field = @$fields[$field_name]) continue;
                    
            $exporter_name = $exporters_by_field_type[$field->getFieldType()];
            if (!$exporter = $this->CSV_Exporters_impl($exporter_name, true)) {
                continue;
            }
            
            if (strlen($column)) {
                $columns[$field_name][$column] = $column;
            }
            $export_fields[$field_name] = $exporter_name;
        }
        unset($selected_fields);
        
        $headers = [];
        foreach (array_keys($export_fields) as $field_name) {
            if (isset($columns[$field_name])) {
                foreach ($columns[$field_name] as $column) {
                    $headers[] = $field_name . '__' . $column;
                }
            } else {
                $headers[] = $field_name;
            }
        }
        if (false === fputcsv($fp, $headers)) {
            throw new Exception\RuntimeException(sprintf('Failed writing CSV headers into file %s', $file));
        }
        
        $offset = (int)$form->values['offset'];
        $limit = (int)$form->values['limit'];
        $fetch_limit = $limit && $limit < 100 ? $limit : 100;
        $rows_exported = $rows_failed = $last_id = 0;
        $bundle = $this->_getBundle($context);
        $count = $num_total = $this->_getQuery($context, $bundle)->count();
        $files = array($file);
        $progress = $this->System_Progress('csv_export')->start($num_total, __('Exporting (%1$d/%2$d) ... %3$s', 'directories'));
        do {
            $i = 0;
            if ($offset > 1) {
                $entities = $this->_getQuery($context, $bundle)
                    ->sortById()
                    ->fetch($fetch_limit, $offset - 1, ['force' => true, 'cache' => false]);
                $offset = 0;
            } else {
                $entities = $this->_getQuery($context, $bundle)
                    ->fieldIsGreaterThan('id', $last_id)
                    ->sortById()
                    ->fetch($fetch_limit, 0, ['force' => true, 'cache' => false]);
            }
            
            // Notify
            $this->Action('csv_export_entities', array($bundle, $entities, $export_fields, $exporter_settings));
            
            foreach ($entities as $entity) {
                --$count;
                ++$i;
                
                $last_id = $entity->getId();
                $row = [];
                $field_values = $entity->getFieldValues(true);
                foreach ($export_fields as $field_name => $exporter_name) {
                    if (!isset($field_values[$field_name])) {
                        // No field value, so populate columns with empty values
                        if (isset($columns[$field_name])) {
                            foreach ($columns[$field_name] as $column) {
                                $row[] = '';
                            }
                        } else {
                            $row[] = '';
                        }
                        continue;
                    }
                
                    $settings = isset($exporter_settings[$field_name]) ? $exporter_settings[$field_name] : [];
                    $settings += array(
                        '_file' => $form->values['filename'],
                    );
                    $exported = $this->CSV_Exporters_impl($exporter_name)->csvExporterDoExport(
                        $fields[$field_name],
                        $settings,
                        $field_values[$field_name],
                        isset($columns[$field_name]) ? $columns[$field_name] : [],
                        $files
                    );
                    if (isset($columns[$field_name])) {
                        foreach ($columns[$field_name] as $column) {
                            $row[] = isset($exported[$column]) ? $exported[$column] : '';
                        }
                    } else {
                        $row[] = $exported;
                    }
                }
                if (false === fputcsv($fp, $row)) {
                    ++$rows_failed;
                    $progress->set($this->Entity_Title($entity), false);
                } else {
                    ++$rows_exported;
                    $progress->set($this->Entity_Title($entity), $rows_exported);
                }
                
                if ($limit && $rows_exported + $rows_failed >= $limit) break 2;
            }            
        } while ($count > 0
            && $i > 0 // prevent loop
        );
        
        $progress->done();
        
        $form->storage['rows_exported'] = $rows_exported;
        $form->storage['rows_failed'] = $rows_failed;
        $form->storage['files'] = $files;
        
        fclose($fp);
    }

    protected function _complete(Context $context, array $formStorage)
    {
        $success = $error = [];
        if (!empty($formStorage['rows_failed'])) {
            $error[] = $this->H(sprintf(__('Faield exporting %d item(s).', 'directories'), $formStorage['rows_failed']));
        }
        if ($formStorage['rows_exported'] > 0) {
            $success[] = $this->H(sprintf(
                __('%d item(s) exported successfullly.', 'directories'),
                $formStorage['rows_exported']
            ));
            $download_file = basename($formStorage['files'][0]);
            if (count($formStorage['files']) > 1
                && class_exists('\ZipArchive', false)
            ) {
                $zip = new \ZipArchive();
                $zip_file = basename($formStorage['files'][0], '.csv') . '.zip';
                if (true !== $result = $zip->open(rtrim(dirname($formStorage['files'][0]), '/') . '/' . $zip_file, \ZipArchive::CREATE)) {
                    $error[] = 'Failed creating zip archive. Error: ' . $this->H($result);
                } else {
                    foreach ($formStorage['files'] as $file) {
                        $zip->addFile($file, basename($file));
                    }
                    $zip->close();
                    $download_file = $zip_file; // let user download zip file
                }
            }
        }
        
        $context->setSuccess(null, array(
            'download_file' => isset($download_file) ? $download_file : null,
            'success' => $success,
            'error' => $error,
        ));
    }
    
    protected function _getQuery(Context $context, Entity\Model\Bundle $bundle)
    {
        return $this->Filter(
            'csv_export_query',
            $this->Entity_Query($bundle->entitytype_name)->fieldIs('bundle_name', $bundle->name),
            array($bundle)
        );
    }
}