<?php
namespace SabaiApps\Directories\Component\Location\FieldRenderer;

use SabaiApps\Directories\Component\Field\IField;
use SabaiApps\Directories\Component\Entity\Type\IEntity;
use SabaiApps\Directories\Component\Field\Renderer\AbstractRenderer;

class AddressFieldRenderer extends AbstractRenderer
{    
    protected function _fieldRendererInfo()
    {
        return array(
            'field_types' => array($this->_name),
            'default_settings' => array(
                'custom_format' => false,
                'format' => '{street}, {city}, {province} {zip}, {country}',
                'link' => false,
                '_separator' => '<br />',
            ),
            'inlineable' => true,
        );
    }
    
    protected function _fieldRendererSettingsForm(IField $field, array $settings, array $parents = [])
    {
        $tags = array('{country}', '{province}', '{city}', '{street}', '{street2}', '{zip}', '{full_address}');
        if (($location_bundle = $this->_getLocationBundle($field))
            && ($location_hierarchy = $this->_application->Location_Hierarchy($location_bundle))
        ) {
            $ret = array(
                'custom_format' => array(
                    '#type' => 'hidden',
                    '#value' => true,
                    '#render_hidden_inline' => true,
                ),
            );
            foreach (array_keys($location_hierarchy) as $key) {
                $tags[] = '{' . $key . '}';
            }
        } else {
            $ret = array(
                'custom_format' => array(
                    '#type' => 'checkbox',
                    '#title' => __('Customize address format', 'directories-pro'),
                    '#default_value' => !empty($settings['custom_format']),
                ),
            );
        }
        return $ret += array(
            'format' => array(
                '#type' => 'textfield',
                '#title' => __('Address format', 'directories-pro'),
                '#description' => sprintf(__('Available tags: %s', 'directories-pro'), implode(' ', array_unique($tags))),
                '#default_value' => $settings['format'],
                '#states' => array(
                    'visible' => array(
                        sprintf('input[name="%s[custom_format]"]', $this->_application->Form_FieldName($parents)) => [
                            'type' => $ret['custom_format']['#type'] === 'checkbox' ? 'checked' : 'value',
                            'value' => true
                        ],
                    ),
                ),
                '#required' => function($form) use ($parents) { $val = $form->getValue(array_merge($parents, array('custom_format'))); return !empty($val); },
            ),
            'link' => array(
                '#type' => 'checkbox',
                '#title' => __('Link to Google Maps', 'directories-pro'),
                '#default_value' => !empty($settings['link']),
            ),
        );
    }

    protected function _fieldRendererRenderField(IField $field, array &$settings, IEntity $entity, array $values, $more = 0)
    {
        $is_mile = $this->_application->getComponent('Map')->getConfig('map', 'distance_unit') === 'mi';
        if ($location_bundle = $this->_getLocationBundle($field)) {
            $location_hierarchy = $this->_application->Location_Hierarchy($location_bundle);
        }
        $ret = [];
        foreach ($values as $key => $value) {
            if ($settings['custom_format']) {
                $replace = array(
                    '{street}' => $this->_application->H($value['street']),
                    '{street2}' => $this->_application->H($value['street2']),
                    '{city}' => $this->_application->H($value['city']),
                    '{province}' => $this->_application->H($value['province']),
                    '{zip}' => $this->_application->H($value['zip']),
                    '{country}' => $this->_application->H($value['country']),
                    '{full_address}' => $this->_application->H($value['address']),
                );
                if (isset($location_hierarchy)) {
                    if (!empty($value['term_id'])
                        && ($term = $entity->getSingleFieldValue($location_bundle->type))
                    ) {
                        $location_titles = (array)$term->getCustomProperty('parent_titles');
                        $location_titles[$term->getId()] = $term->getTitle();
                        foreach (array_keys($location_hierarchy) as $key) {
                            $replace['{' . $key . '}'] = (string)array_shift($location_titles);
                        }
                    }
                }
                // Replace tags
                $formatted = strtr($settings['format'], $replace);
                // Replace multiple columns with single column
                $formatted = preg_replace('/,+/', ',', $formatted);
                // Replace columns with spaces in between
                $formatted = preg_replace('/,\s*,/', ',', $formatted);
                // Replace multiple spacess with single space
                $formatted = preg_replace('/\s+/', ' ', $formatted);
                // Remove starting/trailising spaces/commas
                $formatted = trim($formatted, ' ,');
            } else {
                $formatted = $this->_application->H($value['address']);
            }
            $formatted = $this->_application->Filter('location_address_formatted', $formatted, array($entity, $settings));
            if ($settings['link']) {
                $formatted = sprintf('<a href="http://maps.apple.com/?q=%s,%s">%s</a>', $value['lat'], $value['lng'], $formatted);
            }
            $ret[] = sprintf(
                '<span class="drts-location-address drts-location-address-%d">%s%s</span>',
                $key,
                $formatted,
                isset($value['distance'])
                    ? ' <span class="' . DRTS_BS_PREFIX . 'badge ' . DRTS_BS_PREFIX . 'badge-dark ' . DRTS_BS_PREFIX . 'mx-1 drts-location-distance">' . sprintf($is_mile ? __('%s mi', 'directories-pro') : __('%s km', 'directories-pro'), round($value['distance'], 2)) . '</span>'
                    : ''
            );
        }
        return implode($settings['_separator'], $ret);
    }
    
    protected function _getLocationBundle(IField $field)
    {
        return $this->_application->Entity_Bundle('location_location', $field->Bundle->component, $field->Bundle->group);
    }
    
    protected function _fieldRendererReadableSettings(IField $field, array $settings)
    {
        $ret = [];
        if (!empty($settings['custom_format'])
            && strlen($settings['format'])
        ) {
            $ret['format'] = [
                'label' => __('Address format', 'directories-pro'),
                'value' => $settings['format'],
            ];
        }
        $ret['link'] = [
            'label' => __('Link to Google Maps', 'directories-pro'),
            'value' => !empty($settings['link']),
            'is_bool' => true,
        ];
        return $ret;
    }
}